function DetermineCellMask(obj)

import siman.*;
source = obj.DataSource;

try
    switch obj.Orientation
        case 'linescan'
            DetermineLinescanCell(obj, source);
        case '2-d'
            Determine2DCell(obj, source);
        otherwise
            obj.SetErrorState('Invalid orientation value');
    end
catch ex
    message = 'Error during cell creation';
    siman.Debug.ReportError(ex, message);
    obj.SetErrorState(message);
end
end

function DetermineLinescanCell(obj, source)
dim = source.Size;
switch obj.Type
    case 'manual'
        if isempty(obj.ManualLowerEdge) || isempty(obj.ManualUpperEdge)
            obj.SetErrorState('Invalid cell edge values');
            return;
        end
        if ~isnumeric(obj.ManualLowerEdge) || ~isnumeric(obj.ManualUpperEdge)
            obj.SetErrorState('Invalid cell edge values');
            return;
        end
        if isnan(obj.ManualLowerEdge) || isnan(obj.ManualUpperEdge)
            obj.SetErrorState('Invalid cell edge values');
            return;
        end
        if numel(obj.ManualLowerEdge) ~= 1 || numel(obj.ManualUpperEdge) ~= 1
            obj.SetErrorState('Invalid cell edge values');
            return;
        end
        if obj.ManualLowerEdge < 1 || (obj.ManualUpperEdge > dim(1) && ~isinf(obj.ManualUpperEdge))
            obj.SetErrorState('Cell edges out of range');
            return;
        end
        if obj.ManualLowerEdge > obj.ManualUpperEdge
            obj.SetErrorState('Lower cell edge cannot be greater than upper cell edge');
            return;
        end
        obj.LowerEdge = obj.ManualLowerEdge;
        obj.UpperEdge = min(dim(1), obj.ManualUpperEdge);
        
    case 'threshold'
        data = source.GetData();
        crossSection = mean(data);
        switch obj.ThresholdType
            case 'image units'
                threshold = obj.Threshold_DataUnits;
            case 'percentile'
                percentile = obj.Threshold_Percentile;
                if percentile < 1
                    percentile = percentile * 100;
                end
                threshold = prctile(crossSection, percentile);
        end
        obj.Threshold_Actual = threshold;
        cellPixels = crossSection > threshold;
        
        filterWidth = obj.Threshold_MedianFilter;
        if (filterWidth > 1)
            cellPixels = medfilt1(double(cellPixels), filterWidth);
        end
        
        cellIndexes = find(cellPixels);
        if isempty(cellIndexes)
            startEdge = [];
            stopEdge = [];
        else
            startEdge = min(cellIndexes);
            stopEdge = max(cellIndexes);
        end
        obj.AutomaticLowerEdge = startEdge;
        obj.AutomaticUpperEdge = stopEdge;
        obj.LowerEdge = obj.AutomaticLowerEdge;
        obj.UpperEdge = obj.AutomaticUpperEdge;
end
end

function Determine2DCell(obj, source)
    data = source.GetData();
    switch obj.Type
        case 'manual'
            dim = size(data);
            cellMask = ones(dim([1 2]));
        case 'threshold'
            crossSection = mean(data, 3);
            switch obj.ThresholdType
                case 'image units'
                    threshold = obj.Threshold_DataUnits;
                case 'percentile'
                    percentile = obj.Threshold_Percentile;
                    if percentile < 1
                        percentile = percentile * 100;
                    end
                    threshold = prctile(crossSection(:), percentile);
            end
            obj.Threshold_Actual = threshold;
            cellMask = crossSection > threshold;
    end
    
    cellMask = imclose(cellMask, strel('disk', 4));
    cellMask = imfill(cellMask, 'holes');
    
    obj.Outlines = {};

    connectivityType = str2double(obj.ConnectivityType);
    result = bwconncomp(cellMask, connectivityType);

    rawCount = result.NumObjects;
    if result.NumObjects == 0
        return
    end

    % Filter out compartments that are too big or too small
    rawIndexList = result.PixelIdxList;
    validCellCompartments = cell(rawCount, 2);
    minSize = obj.MinSize;
    maxSize = obj.MaxSize;
    validCount = 0;
    for i = 1:rawCount
        len = numel(rawIndexList{i});
        if len > minSize && len < maxSize
            validCount = validCount + 1;
            validCellCompartments{validCount} = rawIndexList{i};
        else
            cellMask(rawIndexList{i}) = 0;
        end
    end
    
    obj.CellMask = cellMask;

    if validCount == 0
        return
    end

    % Create cell compartments
    dim = size(cellMask);
    for i = 1:validCount
        compartmentIndexes = validCellCompartments{i};
        [~, highestIndex] = min(compartmentIndexes);
        [x, y] = xyindexes(compartmentIndexes, dim);
        topCol = x(highestIndex);
        topRow = y(highestIndex);
        contour = bwtraceboundary(cellMask, [topRow, topCol], 'E');
        obj.Outlines{i} = contour;
    end
end







