classdef (ConstructOnLoad) FeatureDetectionMethod < siman.SimanObject
    % SparksMethod.
    properties
        FeatureType
        DetectionDefName
        UnitsDefName
        
        DetectionMethodOptions = {'detection image'};
    end
    properties
        DetectionDefinition
        UnitsDefinition
        
        DetectionDate
    end
    properties (Dependent)
        HasDetectionDefinition
        HasUnitsDefinition
    end
    properties (Transient)
        DetectionDefListener
        UnitsDefListener
    end
    properties (GetObservable, SetObservable) % Registered properties
        UnitConversion_Space = 1;
        UnitConversion_Time = 1;
        DetectionMethod = 'detection image';
        UnitsDefinitionName = 'sparklet units';
    end
    
    events
        MethodChange
        MethodLayoutChange
        MethodRan
    end
    
    methods % Properties
        function set.DetectionDefinition(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.DetectionDefinition = [];
                obj.DetectionDefListener = [];
                return
            end
            if ~isempty(obj.DetectionDefinition) && isvalid(obj.DetectionDefinition)
                delete(obj.DetectionDefinition);
                if ~isempty(obj.DetectionDefListener)
                    delete(obj.DetectionDefListener)
                end
            end
            obj.DetectionDefinition = value;
            if ~isempty(value)
                obj.DetectionDefListener = addlistener(obj.DetectionDefinition, 'DefinitionChange', @obj.OnDefinitionChanged);
            end
        end
        function set.UnitsDefinition(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.UnitsDefinition = [];
                obj.UnitsDefListener = [];
                return
            end
            if ~isempty(obj.UnitsDefinition) && isvalid(obj.UnitsDefinition)
                delete(obj.UnitsDefinition);
                if ~isempty(obj.UnitsDefListener)
                    delete(obj.UnitsDefListener)
                end
            end
            obj.UnitsDefinition = value;
            if ~isempty(value)
                obj.UnitsDefListener = addlistener(obj.UnitsDefinition, 'DefinitionChange', @obj.OnDefinitionChanged);
            end
        end
        function tf = get.HasDetectionDefinition(obj)
            tf = ~isempty(obj.DetectionDefName);
        end
        function tf = get.HasUnitsDefinition(obj)
            tf = ~isempty(obj.UnitsDefName);
        end
    end
    
    methods
        % Constructor
        function obj = FeatureDetectionMethod()
            obj@siman.SimanObject();
            
            obj.InitMethod();
            if isempty(obj.DetectionMethod) && ~isempty(obj.DetectionMethodOptions)
                obj.DetectionMethod = obj.DetectionMethodOptions{1};
            end
            if isempty(obj.UnitsDefinitionName)
                obj.UnitsDefinitionName = obj.UnitsDefName;
            end
            
            obj.InitListeners();
        end
        
        function InitListeners(obj)
            obj.SelfListener = addlistener(obj, 'PropertyChanged', @obj.OnPropertyChanged);
        end
        
        function OnPropertyChanged(obj, src, evnt)
            switch evnt.PropertyName
                otherwise
                    notify(obj, 'MethodChange');
            end
            % Indicate need for layout change
            switch evnt.PropertyName
                case 'DetectionMethod'
                    notify(obj, 'MethodLayoutChange');
            end
        end
        
        function OnDefinitionChanged(obj, src, evnt)
            notify(obj, 'MethodChange');
        end
        
       % Handle enabling of controls tied to a local property
        function enabling = GetEnabling(obj, id, prop, context)
            enabling = true;
        end
        
        function def = GetDefaultDetectionDefinition(obj)
            if isempty(obj.DetectionDefName)
                def = [];
                return;
            end
            def = siman.ProcessingDefinition;
            def.Name = obj.DetectionDefName;
            def.IsSystemDefinition = true;
        end
              
        function def = GetDefaultUnitsDefinition(obj)
            if isempty(obj.UnitsDefName)
                def = [];
                return;
            end
            def = siman.ProcessingDefinition;
            def.Name = obj.UnitsDefName;
            def.IsSystemDefinition = true;
        end
        
        function def = GetUnitsDefinition(obj)
            def = [];
            if isempty(obj.UnitsDefinitionName)
                return;
            end
            mgr = siman.ImageManager.GetManager();
            try
                def = mgr.FindDefinition(obj.UnitsDefinitionName);
            catch
            end
        end
        
        function FireMethodRan(obj)
            notify(obj, 'MethodRan');
        end
        
        function ApplyChanges(obj)
            if ~isempty(obj.DetectionDefinition)
                obj.DetectionDefinition.ApplyChanges();
            end
            if ~isempty(obj.UnitsDefinition)
                obj.UnitsDefinition.ApplyChanges();
            end
        end
        
        function tf = HasPendingChanges(obj)
            tf = false;
            if ~isempty(obj.DetectionDefinition)
                tf = obj.DetectionDefinition.HasPendingChanges();
                if (tf)
                    return;
                end
            end
            if ~isempty(obj.UnitsDefinition)
                tf = obj.UnitsDefinition.HasPendingChanges();
                if (tf)
                    return;
                end
            end
        end
        
        function delete(obj)
            if ~isempty(obj.DetectionDefinition)
                delete(obj.DetectionDefinition);
                delete(obj.DetectionDefListener);
                obj.DetectionDefinition = [];
            end
            if ~isempty(obj.UnitsDefinition)
                delete(obj.UnitsDefinition);
                delete(obj.UnitsDefListener);
                obj.UnitsDefinition = [];
            end
        end
        
    end
    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@siman.SimanObject(obj);
            cpObj.InitListeners();
            % Make a deep copy of the definitions
            if ~isempty(obj.DetectionDefinition)
                cpObj.DetectionDefinition = [];
                cpObj.DetectionDefinition = copy(obj.DetectionDefinition);
            end
            if ~isempty(obj.UnitsDefinition)
                cpObj.UnitsDefinition = [];
                cpObj.UnitsDefinition = copy(obj.UnitsDefinition);
            end
        end
    end
end





