classdef FeatureListPanel < siman.ToolPanel
    % FeatureListPanel
    
    properties
        FeatureType;
    end
    properties (Transient)
        FeatureSourceListener
        CurrentFeatureSource
    end
    properties (Dependent)
        TargetDataSources  % Detect command interface
        FeatureList
        SelectedFeatureList
    end
    
    methods % Properties
        
        function list = get.TargetDataSources(obj)
            list = siman.List;
            if ~isempty(obj.CurrentFeatureSource)
                list.Add(obj.CurrentFeatureSource)
            end
        end
        
        function list = get.FeatureList(obj)
            list = obj.GetFeatureList();
        end
        
        function list = get.SelectedFeatureList(obj)
            list = obj.GetSelectedFeatureList();
        end
    end
    
    methods
        % Constructor
        function obj = FeatureListPanel(parent, type)
            obj@siman.ToolPanel(parent);
            obj.FeatureType = type;
        end
        
        function mgr = GetManager(obj)
            siman.Debug.ReportProblem(['GetManager was not overloaded in base class FeatureListPanel: ' obj.Name]);
            mgr = [];
        end
        
        function InitDefaults(obj)
            if isempty(obj.Name)
                obj.Name = 'Unknown feature list panel';
            end
            if isempty(obj.Size)
                obj.Size = [260 400];
            end
            if isempty(obj.DockedLocation)
                obj.DockedLocation = 'east';
            end
            if isempty(obj.Stretchable)
                obj.Stretchable = true;
            end
        end
        
        function InitPanel(obj)
            InitPanel@siman.ToolPanel(obj);
        end
        
        function OnAttachedPlotChanged(obj, type)
            if ~strcmp(type, 'definition')
                obj.ResetCurrentFeatureSource();
            end
        end
        
        function ResetCurrentFeatureSource(obj)
            if isempty(obj.CurrentAttachedPlot)
                obj.SetFeatureSource([]);
                return;
            end
            source = obj.CurrentAttachedPlot.RawDataSource;
            obj.SetFeatureSource(source);
        end
        
        function SetFeatureSource(obj, source)
            if isempty(source) && isempty(obj.CurrentFeatureSource)
                return;
            end
            if length(source) ~= length(obj.CurrentFeatureSource) || source ~= obj.CurrentFeatureSource
                if isempty(source) || isempty(obj.CurrentFeatureSource)
                    obj.QueueLayout();
                end
                if ~isempty(obj.FeatureSourceListener)
                    delete(obj.FeatureSourceListener);
                    obj.FeatureSourceListener = [];
                end
                obj.CurrentFeatureSource = source;
                if ~isempty(source)
                    obj.FeatureSourceListener = addlistener(source, 'FeaturesChange', @obj.OnFeatureSourceChanged);
                    obj.FeatureSourceListener(2) = addlistener(source, 'FeatureChange', @obj.OnFeatureChanged);
                end
                obj.SetPanelDataSource(obj.CurrentFeatureSource);
            end
        end
        
        function OnFeatureSourceChanged(obj, src, evt)
            obj.QueueRedraw();
        end
        
        function OnFeatureChanged(obj, src, evt)
            obj.QueueRedraw();
        end
        
        function list = GetFeatureList(obj)
            source = obj.CurrentFeatureSource;
            if isempty(source)
                list = siman.List;
            else
                list = source.GetFeatureType(obj.FeatureType);
            end            
        end
        function list = GetSelectedFeatureList(obj)
            source = obj.CurrentFeatureSource;
            if isempty(source)
                list = siman.List;
            else
                list = source.GetSelectedFeatureType(obj.FeatureType);
            end            
        end
        
        function Redraw(obj)
            obj.RefreshControls();
        end
        
        function ResetLayout(obj)
            obj.MakeControls();
        end
        
        function UpdateControl(obj, id) % Handle manual refreshing
            tag = get(id, 'Tag');
            switch tag
                case 'FeatureListLabel'
                    enabling = ~isempty(obj.CurrentFeatureSource);
                case 'FeatureList'
                    enabling = ~isempty(obj.CurrentFeatureSource);
                    obj.RefreshFeatureList(id);
                otherwise
                    enabling = 'off';
                    siman.Debug.ReportProblem(['Unknown control to manual UpdateControl: ' tag]);
            end
            if ~ischar(enabling)
                if enabling
                    enabling = 'on';
                else
                    enabling = 'off';
                end
            end
            set(id, 'Enable', enabling);
        end
        
        function RefreshFeatureList(obj, id)
            list = obj.GetFeatureList();
            count = list.Count;
            if count == 0
                strings = ' ';
                selected = [];
                max = 2;
            else
                strings = cell(count, 1);
                array = list.ToArray();
                selected = [array.IsSelected];
                for i = 1:count
                    feature = array(i);
                    strings{i} = feature.GetListString();
                end
                selected = find(selected);
                max = 2;
            end

            set(id, 'String', strings, 'Value', selected, 'Max', max);
        end
        
        function ControlChanged(obj, id, data_obj, prop)
            switch prop
                case {'FeatureList'}
                    obj.ResetSelectedFeatures(id);
                otherwise
                    ControlChanged@siman.FeatureListPanel(obj, id, data_obj, prop);
            end
        end
        
        function ResetSelectedFeatures(obj, id)
            list = obj.GetFeatureList();
            count = list.Count;
            if count == 0
                return;
            end
            selected = get(id, 'Value');
            source = obj.CurrentFeatureSource;
            if numel(selected) == 0
                source.SelectFeatures([], 'select');
                return;
            elseif numel(selected) == 1
                feature = list.ElementAt(selected);
                source.SelectFeatures(feature, 'select');
                obj.CurrentAttachedPlot.ShowFeature(feature);
                return;
            end
            selectedList = siman.List;
            for i = 1:numel(selected)
                feature = list.ElementAt(selected(i));
                selectedList.Add(feature);
            end
            source.SelectFeatures(selectedList, 'select');
        end
        
        function delete(obj)
            delete(obj.FeatureSourceListener);
            obj.FeatureSourceListener = [];
        end
    end
end


