classdef (ConstructOnLoad) FilterKernel < siman.SimanObject
    % FilterKernel - an object that specifies the shape of a filter's kernel
    properties (SetObservable, GetObservable)
        Shape = 'circle';
        Radius = 1;
        Width = 3;
        Height = 3;
    end

    properties (Dependent)
        StructuringElement
        KernelElementCount
    end
    
    properties (Constant)
        ShapeOptions = {'square', 'circle'};
    end
    
    events
        LayoutChanged
    end
    
    methods  % Properties
        function value = get.StructuringElement(obj)
            switch obj.Shape
                case 'circle'
                    r = obj.Radius;
                    if (r > 1.4 && r < 2)
                        value = strel('square', 3);
                        return
                    end
                    if r == round(r)
                        value = strel('disk', r, 4);
                    else
                        value = strel('arbitrary', getnhood(r)); % use manual function for non-integer radii
                    end
                case 'square'
                    value = strel('square', obj.Width);
                %case 'rectangle'
                %    value = strel('square', obj.Width);
                otherwise
                    siman.Debug.ReportProblem('Unknown filter kernel shape');
            end
        end

        function value = get.KernelElementCount(obj)
            value = numel(find(obj.StructuringElement.getnhood));
        end
    end
    
    methods
        function obj = FilterKernel()
            obj.SelfListener = addlistener(obj, 'PropertyChanged', @obj.OnPropertyChanged);
        end
        
        function OnPropertyChanged(obj, src, evnt)
            switch evnt.PropertyName
                case 'Shape'
                    notify(obj, 'LayoutChanged');
                otherwise
            end
        end
        
        function str = GetDescription(obj)
            switch obj.Shape
                case 'circle'
                    str = [obj.Shape ' r=' num2str(obj.Radius)];
                case 'square'
                    str = [obj.Shape ' width=' num2str(obj.Width)];
                otherwise
                    siman.Debug.ReportProblem('Unknown filter kernel shape');
                    str = 'unknown shape';
            end
        end
        
        function [xoffsets, yoffsets] = GetKernelOffsets(obj)
            nhood = obj.StructuringElement.getnhood;
            dim = size(nhood);
            [x, y] = xyindexes(find(nhood), dim);
            xoffsets = x - round(dim(2)/2);
            yoffsets = y - round(dim(2)/2);     
        end
        
        function [valid, message] = Validate(obj)
            valid = ismember(obj.Shape, obj.ShapeOptions);
            message = '';
            if ~valid
                message = ['Invalid kernel shape: ' obj.Shape];
                return;
            end
        end
        
        function location = MakeControls(obj, location, controlParams)
            controlParams.DataObject = obj;            

            controlParams.Width = 90;
            location = siman.UIHelper.MakeLabeledPopup('Shape', 'Kernel shape:', ...
                location, controlParams);
        
            sideLocation = location;
            switch obj.Shape
                case {'circle'}
                    controlParams.Width = 30;
                    location = siman.UIHelper.MakeLabeledTextbox('Radius', '   Shape radius:', ...
                        location, controlParams);
                case {'square'}
                    controlParams.Width = 30;
                    location = siman.UIHelper.MakeLabeledTextbox('Width', '   Shape width:', ...
                        location, controlParams);
            end
            
            controlParams.Width = 20;
            sideLocation(1) = location(1) + 130;
            siman.UIHelper.MakeLabeledIndicator('KernelElementCount', '   Kernel area:', ...
                sideLocation, controlParams);
        
        end
        
        function enabling = GetEnabling(id, prop, context)
            enabling = true;
        end
        
        function ControlChanged(obj, id)
            siman.UIHelper.ProcessGenericControl(id, obj);
        end
        
        function delete(obj)
        end
    end
end





