classdef ImageManager < siman.SimanObject & siman.CommandManager
    % ImageManager.
    properties (Constant)
        Instance = siman.ImageManager; % Singleton object
    end
    properties (Constant)
        StoredDefinitionsFilename = 'UserDefinitions.mat';
    end
    properties (Access = private)
        FileInventory
        OpenDisplays
        ProcessingDefs
        ProcessingDefNames
    end
    properties (Dependent)
        InventoryCount
        CurrentPlot
    end
    properties (GetObservable, SetObservable) % Registered properties
        LastImagePath
        LastImageDir
    end
    properties (Transient)
        IsInitialized = false;
        DefinitionsNeedSaving = false;
        DefinitionListeners = [];
        NamesNeedResetting = true;
        
        CurrentDisplay
        SparkManager
        SparkletManager
        SparkletSiteManager
        TransientManager
        ContractionManager
        ROIManager
    end
    
    events
        InventoryChanged
        DefinitionsChanged
    end
    
    methods
        % Properties
        function count = get.InventoryCount(obj)
            count = obj.FileInventory.Count;
        end
        function plot = get.CurrentPlot(obj)
            plot = [];
            if ~isempty(obj.CurrentDisplay)
                plot = obj.CurrentDisplay.CurrentPlot;
            end
        end
    end
    
    methods
        % Singleton Constructor
        function obj = ImageManager()
            obj@siman.SimanObject();
            obj@siman.CommandManager();
            
            obj.FileInventory = siman.List;
            obj.OpenDisplays = siman.List;
            obj.ProcessingDefs = siman.List;
            obj.InitProcessingDefs();
            obj.InitCommands();
        end
    end
    
    methods
        function Init(obj)
            if obj.IsInitialized
                return
            end
            obj.IsInitialized = true;
            
            % Init managers
            obj.ContractionManager = siman.ContractionManager.GetManager();
            obj.ContractionManager.InitDefinitions();
            obj.TransientManager = siman.TransientManager.GetManager();
            obj.TransientManager.InitDefinitions();
            obj.SparkManager = siman.SparkManager.GetManager();
            obj.SparkManager.InitDefinitions();
            obj.SparkletManager = siman.SparkletManager.GetManager();
            obj.SparkletManager.InitDefinitions();
            obj.SparkletSiteManager = siman.SparkletSiteManager.GetManager();
            obj.SparkletSiteManager.InitDefinitions();
            obj.ROIManager = siman.ROIManager();
        end
        
        function OnInventoryChanged(obj)
            notify(obj, 'InventoryChanged');
        end
        
        function RegisterDefinition(obj, def)
            oldDef = obj.FindDefinition(def.Name);
            if ~isempty(oldDef)
                siman.Debug.ReportProblem(['Tried to register an existing definition: ' def.Name]);
                return;
            end
            obj.ProcessingDefs.Add(def);
            notify(obj, 'DefinitionsChanged');
        
            obj.DefinitionsNeedSaving = true;
            if isempty(obj.DefinitionListeners)
                obj.DefinitionListeners = addlistener(def, 'DefinitionChange', @obj.OnDefinitionChanged);
            else
                obj.DefinitionListeners(end + 1) = addlistener(def, 'DefinitionChange', @obj.OnDefinitionChanged);
            end
            obj.NamesNeedResetting = true;
        end
        
        function OnDefinitionChanged(obj, ~, ~)
            obj.DefinitionsNeedSaving = true;
        end
        
        function names = GetDefinitionNameOptions(obj)
            if obj.NamesNeedResetting
                set = obj.ProcessingDefs;
                if isempty(set)
                    names = {};
                    return;
                end
                names = cell(1, set.Count);
                for i = 1:set.Count
                    def = set.ElementAt(i);
                    names{i} = def.Name;
                end
                obj.ProcessingDefNames = names;
                obj.NamesNeedResetting = false;
                return;
            end
            names = obj.ProcessingDefNames;
        end
        function def = FindDefinition(obj, name)
            def = [];
            set = obj.ProcessingDefs;
            for i = 1:set.Count           
                if strcmp(set.ElementAt(i).Name, name)
                    def = set.ElementAt(i);
                    return;
                end
            end
        end
        
        function ShowNoticeMessage(obj, str)
            if ~isempty(obj.CurrentDisplay)
                obj.CurrentDisplay.ShowNoticeMessage(str);
            end
        end
        
        function delete(obj)
            delete(obj.OpenDisplays);
            obj.OpenDisplays = [];
            delete(obj.ProcessingDefs);
            obj.ProcessingDefs = [];
            delete(obj.OpenFiles);
            obj.OpenFiles = [];
            delete(obj.ROIManager);
        end
        
        function ApplyChanges(obj)
            count = 0;
            while(true)
                list = obj.FileInventory;
                for i = 1:list.Count
                    file = list.ElementAt(i);
                    file.ApplyChanges();
                end
                list = obj.ProcessingDefs;
                for i = 1:list.Count
                    def = list.ElementAt(i);
                    def.ApplyChanges();
                end
                list = obj.OpenDisplays;
                for i = 1:list.Count
                    display = list.ElementAt(i);
                    if isvalid(display)
                        display.ApplyChanges();
                    end
                end
                obj.ContractionManager.ApplyChanges();
                obj.TransientManager.ApplyChanges();
                obj.SparkManager.ApplyChanges();
                obj.SparkletManager.ApplyChanges();
                obj.SparkletSiteManager.ApplyChanges();
                obj.ROIManager.ApplyChanges();
                count = count + 1;
                if count > 10
                    siman.Debug.ReportProblem('Event loop encountered too many repeats');
                    return;
                end
                tf = obj.HasPendingChanges();
                if ~tf
                    break;
                end
            end
            
            % Registry events should only happen at the very end (should never cause other objects to change)
            siman.DefaultsRegistry.ApplyChanges();
            if obj.DefinitionsNeedSaving
                obj.StoreProcessingDefs();
                obj.DefinitionsNeedSaving = false;
            end  
        end
        
        function tf = HasPendingChanges(obj)
            tf = true;
            list = obj.FileInventory;
            for i = 1:list.Count
                file = list.ElementAt(i);
                if file.HasPendingChanges();
                    return;
                end
            end
            list = obj.ProcessingDefs;
            for i = 1:list.Count
                def = list.ElementAt(i);
                def.ApplyChanges();
                if def.HasPendingChanges();
                    return;
                end
            end
            list = obj.OpenDisplays;
            for i = 1:list.Count
                display = list.ElementAt(i);
                if isvalid(display)
                    display.ApplyChanges();
                    if display.HasPendingChanges();
                        return;
                    end
                end
            end
            if obj.ContractionManager.HasPendingChanges();
                return;
            end
            if obj.TransientManager.HasPendingChanges();
                return;
            end
            if obj.SparkManager.HasPendingChanges();
                return;
            end
            if obj.SparkletManager.HasPendingChanges();
                return;
            end
            if obj.SparkletSiteManager.HasPendingChanges();
                return;
            end
            if obj.ROIManager.HasPendingChanges();
                return;
            end
            tf = false;
            return;
        end
        
    end
    
    methods (Static = true)
        function mgr = GetManager()
            mgr = siman.ImageManager.Instance;
        end
        
        function ApplyAllChanges()
            mgr = siman.ImageManager.GetManager();
            mgr.ApplyAllChanges();
        end
        
        function sources = CreateImageFileObjs(filenames)
            import siman.ImageManager;
            mgr = ImageManager.Instance;
            sources = siman.ImageFile(length(filenames)); % Allocate array
            for i = 1:length(filenames)
                sources(i) = siman.ImageFile(filenames{i});      
            end
            mgr.AddToInventory(sources);
        end
        function AddToInventory(files)
            import siman.ImageManager;
            mgr = ImageManager.Instance;
            for i = 1:length(files)
                mgr.FileInventory.Add(files(i));
            end
            mgr.OnInventoryChanged();
        end
        function ClearImageFiles()
            import siman.ImageManager;
            mgr = ImageManager.Instance;
            mgr.FileInventory.Clear();
            mgr.OnInventoryChanged();
        end
        function list = GetInventoryList()
            import siman.ImageManager;
            mgr = ImageManager.Instance;
            list = mgr.FileInventory;
        end
        
        function RegisterDisplay(display)
            siman.ImageManager.Instance.OpenDisplays.Add(display);
            siman.Debug.WriteLine('--------------------------------------', true);
            siman.Debug.WriteLine('Opened display', true);
        end
        function UnregisterDisplay(display)
            mgr = siman.ImageManager.Instance;
            mgr.OpenDisplays.Remove(display);
            mgr.ClearCommands(display);            
            mgr.CurrentDisplay = [];
            siman.Debug.WriteLine('Closed display', true);
            siman.Debug.WriteLine('', true);
        end
        
        function enabling = ResetCommandEnabling(cmd, context)
            enabling = false;
            if ~isempty(strfind(cmd.Name, 'ProcessingDef.'))                
                if (~isprop(context, 'ProcessingDef'))
                    siman.Debug.ReportProblem(['Image manager command refresh lacks ProcessingDef in context: ' cmd.Name]);
                    return;
                end
                if (~isprop(context, 'CurrentOperation'))
                    siman.Debug.ReportProblem(['Image manager command refresh lacks CurrentOperation in context: ' cmd.Name]);
                    return;
                end
            elseif ~isempty(strfind(cmd.Name, 'File.'))
                if (~ismethod(context, 'OnManagerSourcesOpened'))
                    siman.Debug.ReportProblem(['Image manager command refresh lacks OnManagerSourcesOpened method in context: ' cmd.Name]);
                    return;
                end
            end
            switch cmd.Name
                case {'File.OpenImageFiles', 'File.OpenImageDirectory'}
                    enabling = true;
                case {'ProcessingDef.InsertOperation'}
                    def = context.ProcessingDef;
                    enabling = ~isempty(def) && def.IsEditable;
                case {'ProcessingDef.MoveUpOperation', 'ProcessingDef.MoveDownOperation', ...
                        'ProcessingDef.DeleteOperation', 'ProcessingDef.ChangeOperationType', ...
                        'ProcessingDef.SkipOperation', 'ProcessingDef.StopOperation', }
                    op = context.CurrentOperation;
                    enabling = ~isempty(op);
                otherwise
                    siman.Debug.ReportProblem(['Image manager unknown command to enable: ' cmd.Name]);
            end
        end
        
        function OnCommand(cmd, id, context)
            if ~isempty(strfind(cmd.Name, 'ProcessingDef.'))                
                def = context.ProcessingDef;
                op = context.CurrentOperation;
            elseif ~isempty(strfind(cmd.Name, 'File.'))
                listener = context;
            end
            
            switch cmd.Name
                case 'File.OpenImageFiles'
                    sources = siman.ImageManager.Instance.FileIO('open image files');
                    if ~isempty(sources)
                        listener.OnManagerSourcesOpened(sources);
                    end
                    
                case 'File.OpenImageDirectory'
                    sources = siman.ImageManager.Instance.FileIO('open image directory');
                    if ~isempty(sources)
                        listener.OnManagerSourcesOpened(sources);
                    end
                    
                case 'ProcessingDef.MoveUpOperation'
                    def.MoveOperationUp(op);
                case 'ProcessingDef.MoveDownOperation'
                    def.MoveOperationDown(op);
                case 'ProcessingDef.InsertOperation'
                    def.RequestNewOperation(op);
                case 'ProcessingDef.DeleteOperation'     
                    def.RemoveOperation(op);
                case 'ProcessingDef.ChangeOperationType'
                    def.ChangeOperationType(op);
                case 'ProcessingDef.SkipOperation'
                    op.Active = ~get(id, 'Value');
                case 'ProcessingDef.StopOperation'
                    op.Stop = get(id, 'Value');
                otherwise
                    siman.Debug.ReportProblem(['Unknown command to Imagemanager: ' cmd.name]);
            end
        end
        
        function InsertNewOperation(def, op)
            options = siman.ProcessingOperation.Types;
            currentType = 1;
            [selection,ok] = listdlg('ListString', options, 'SelectionMode', 'single', ...
                'InitialValue', currentType);
            if ok
                newOperation = def.InsertOperation(op);
                newOperation.Type = options{selection};
            end
        end
        
        function RemoveOperation(def, op)
            response = questdlg('Are you sure you want to delete the operation?', 'Verify Delete', 'Cancel');
            if strcmp(response, 'Yes')
                def.RemoveOperation(op);
            end
        end
        
        function ChangeOperationType(op)
            options = siman.ProcessingOperation.Types;
            currentType = find(strcmp(op.Type, options));
            [selection,ok] = listdlg('ListString', options, 'SelectionMode', 'single', ...
                'InitialValue', currentType);
            if ok && selection ~= currentType
                op.Type = options{selection};
            end
        end
        
    end
end





