classdef ImagePlot < siman.SimanObject & siman.ROIContainer
    % Must be a subclass of handle
    properties (SetObservable, GetObservable)
        LastProcessingDefName = 'raw';
        LastSavedImagePath;
        LastSavedMoviePath;
        
        Zoom = 100;
        Direction = 'top-bottom';
        ViewWindowOffset = [0 0];
        
        Colormap = 'gray';
        ColorLimMode = 'auto';
        ColorLim_Lower = 0;
        ColorLim_Upper = 100;
        ColorPercentLim_Lower = 10;
        ColorPercentLim_Upper = 90;
        ColorPercentileLim_Lower = 10;
        ColorPercentileLim_Upper = 90;
        ColorLimActual_Lower = 0;
        ColorLimActual_Upper = 100;
        
        CellVisible = true;
        CellIndicatorType = 'mask';
        CellColor = [0 1 0];
        CellAlpha = .3;
        
        ShowColorbar = false;
    end
    properties (Constant)
        DirectionOptions = {'top-bottom', 'bottom-top', 'left-right'};
        ColorLimModeOptions = {'auto', 'manual', 'manual percent', 'manual percentile'};
        ColormapOptions = {'gray', 'hsv', 'hot', 'jet', 'bone', 'copper', 'lines', 'colorcube'};
        
        CellIndicatorTypeOptions = {'border', 'mask'};
    end
    properties (Transient = true)
        AxesID
        ImageID
        MessageAxesID
        HorizontalScrollID
        VerticalScrollID
        ColorbarID
        
        Parent
        ErrorMessage
        
        RawDataSource
        CurrentProcessingDef
        SourceListeners
        DefinitionListener
        ProcessedData
        FullImage
        
        ViewImageXRange
        ViewImageYRange
        FrameCount
        FrameIndex = 1;
        FullDataSize
        ImageSize
        ImageDataLimits
        
        CellGraphics
        FeatureGraphics
        
        InteractionState = '';
        ButtonDownPoint = [];
        SelectionRectID = [];
        ButtonDownTime = [];
        
        IsLocked = false;
        NeedsRecreate = true;
        NeedsRedraw = true;
        NeedsCellRedraw = true;
        NeedsFeaturesRedraw = true;
        NeedsColorReset = true;
        IsDataOutOfDate = true;
        
        MouseListener
        
        NoticeMessage
    end
    properties (Dependent)
        CurrentProcessingDefName
        IsRotated
        IsFlipped
        HasError
        HasImageSeries
    end
    properties (Constant)
        MESSAGE_ERROR = 1;
        MESSAGE_INFO = 2;
    end
    
    events
        DataChanged
        DefinitionSwitched
        SourceSwitched
        CellMask
        PropertyLayoutChanged
    end
    
    methods % properties
        function value = get.IsRotated(obj)
            value = ismember(obj.Direction, {'left-right', 'right-left'});
        end
        function value = get.IsFlipped(obj)
            value = ismember(obj.Direction, {'top-bottom', 'right-left'});
        end
        function value = get.HasError(obj)
            value = ~isempty(obj.ErrorMessage);
        end
        function set.CurrentProcessingDef(obj, value)
            obj.CurrentProcessingDef = value;
            if ~isempty(obj.DefinitionListener)
                delete(obj.DefinitionListener);
                obj.DefinitionListener = [];
            end
            if ~isempty(obj.CurrentProcessingDef)
                obj.DefinitionListener = addlistener(value, 'DefinitionChange', @obj.OnDefinitionChanged);
            end
            obj.OnDataOutOfDate();
            notify(obj, 'DefinitionSwitched');
        end
        function set.RawDataSource(obj, value)
            obj.RawDataSource = value;
            obj.QueueRecreate();
            notify(obj, 'SourceSwitched');
        end
        function set.FrameIndex(obj, value)
            obj.FrameIndex = value;
            obj.ResetFrameIndicator();
            obj.QueueRedraw();
            obj.QueueRedrawFeatures();
        end
        function answer = get.HasImageSeries(obj)
            answer = ~isempty(obj.RawDataSource) && ~isempty(obj.CurrentProcessingDef);
            if (answer)
                dim = obj.CurrentProcessingDef.GetOutputSize(obj.RawDataSource);
                if length(dim) < 3 || dim(3) < 2
                    answer = false;
                end
            end
        end
        function name = get.CurrentProcessingDefName(obj)
            name = '';
            if (~isempty(obj.CurrentProcessingDef))
                name = obj.CurrentProcessingDef.Name;
            end
        end
    end
    
    methods
        function obj = ImagePlot(arg)
            obj@siman.SimanObject();
            
            if nargin > 0
                if isnumeric(arg)
                    obj(1:arg) = siman.ImagePlot;
                    return;
                else
                    obj.Parent = arg;
                    obj.InitPlot();
                    obj.MouseListener = addlistener(arg, 'MouseMotion', @obj.OnCursorMotion);
                    obj.SelfListener = addlistener(obj, 'PropertyChanged', @obj.OnPropertyChanged);
                end
            end
        end
        
        function FireDataChanged(obj)
            notify(obj, 'DataChanged');
        end
        
        function QueueRedraw(obj)
            obj.NeedsRedraw = true;
        end
        
        function QueueRecreate(obj)
            obj.NeedsRecreate = true;
        end
        
        function QueueRedrawCell(obj)
            obj.NeedsCellRedraw = true;
        end
        
        function QueueRedrawFeatures(obj)
            obj.NeedsFeaturesRedraw = true;
        end
        
        function QueueResetColor(obj)
            obj.NeedsColorReset = true;
        end
        
        function ApplyChanges(obj)
            if ~isempty(obj.CurrentProcessingDef)
                obj.CurrentProcessingDef.ApplyChanges();
            end
            try
                if obj.NeedsRecreate && ~obj.IsLocked
                    obj.ErrorMessage = '';
                    obj.RecreateImage();
                    obj.IsDataOutOfDate = ~isempty(obj.ErrorMessage);
                    obj.QueueRedraw();
                    obj.NeedsRecreate = false; %~isempty(obj.ErrorMessage); would create infinite apply loop
                    obj.FireDataChanged();
                end
                if obj.NeedsRedraw
                    obj.Redraw();
                    obj.QueueResetColor();
                    obj.NeedsRedraw = false;
                end
                if obj.NeedsCellRedraw
                    obj.DrawCell();
                    obj.NeedsCellRedraw = false;
                end
                if obj.NeedsFeaturesRedraw
                    obj.DrawFeatures();
                    obj.NeedsFeaturesRedraw = false;
                end
                if obj.NeedsColorReset
                    obj.ResetColor();
                    obj.NeedsColorReset = false;
                end
                obj.SetNoticeMessage('');
            catch ex
                if obj.NeedsRecreate
                    msg = 'Error processing image data';
                elseif obj.NeedsRedraw
                    msg = 'Error drawing image';
                elseif obj.NeedsCellRedraw
                    msg = 'Error drawing cell';
                elseif obj.NeedsFeaturesRedraw
                    msg = 'Error drawing features';
                elseif obj.NeedsColorReset
                    msg = 'Error setting colors';
                end
                obj.NeedsRedraw = false; % These all need to be turned off to prevent infinite loop
                obj.NeedsRecreate = false;
                obj.NeedsCellRedraw = false;
                obj.NeedsFeaturesRedraw = false;
                obj.NeedsColorReset = false;
                obj.IsDataOutOfDate = true;
                obj.SetNoticeMessage(msg, siman.UIHelper.MESSAGE_ERROR);
                siman.Debug.ReportError(ex, msg);
            end
        end
        
        function tf = HasPendingChanges(obj)
            tf = obj.NeedsRecreate || obj.NeedsRedraw || obj.NeedsColorReset || ...
                obj.NeedsCellRedraw || obj.NeedsFeaturesRedraw;
        end
        
        function Redraw(obj)
            if ~isempty(obj.ProcessedData)
                obj.ErrorMessage = '';
                % Indication of error in processing, so presumably error already displayed
            end
            obj.ResetAxes();
            obj.ResetSliderLayout();
            obj.ResetFrameIndicator();
            obj.DrawImage();
            if obj.HasError
                return
            end
        end
        
        function InitPlot(obj)
            parentID = obj.Parent.PlotPanelID;
            obj.AxesID = axes('Parent', parentID, 'Visible', 'off', 'Color', 'none', ...
                'Tag', 'Plot Axes', 'Units', 'pixels');
            obj.ResetDirection();
            obj.ResetColorbar();
            obj.MessageAxesID = axes('Parent', parentID, 'Visible', 'off', 'Color', 'none', ...
                'XLim', [0 1], 'YLim', [0 1], 'Tag', 'Message Axes');
            
            callback = {@changeView, obj};
            obj.VerticalScrollID = uicontrol('Parent', parentID, 'Visible', 'off', 'Style', 'slider', ...
                'Tag', 'Vertical Scroll', 'Callback', callback);
            obj.HorizontalScrollID = uicontrol('Parent', parentID, 'Visible', 'off', 'Style', 'slider', ...
                'Tag', 'Horizontal Scroll', 'Callback', callback);
            
            function changeView(src, evt, obj)
                obj.OnScroll(src);
            end
        end
        
        function SetNoticeMessage(obj, message, type)
            cla(obj.MessageAxesID);
            if isempty(message)
                if ~isempty(obj.NoticeMessage)
                    obj.NoticeMessage = '';
                    drawnow;
                end
                return;
            end
            obj.NoticeMessage = message;
            if nargin < 3
                type = siman.UIHelper.MESSAGE_INFO;
            end
            background = [.9 .9 .9];
            switch type
                case siman.UIHelper.MESSAGE_INFO
                    background = get(obj.Parent.PlotPanelID, 'BackgroundColor');
                case siman.UIHelper.MESSAGE_ERROR
                    background = [1 .8 .8];
            end
            text(.5, .5, message, 'Parent', obj.MessageAxesID, ...
                'HorizontalAlignment', 'center', 'FontSize', 16, 'BackgroundColor', background);
            drawnow;
        end
        
        function ResetColorbar(obj)
            axes(obj.AxesID);
            if obj.ShowColorbar
                obj.ColorbarID = colorbar('peer', obj.AxesID);
            else
                colorbar('off');
            end
        end
        
        function AddROIType(obj, type)
            obj.InteractionState = 'adding roi';
            roi = siman.ROI(type, obj);
            roi.InitROIVisual();
            obj.AddROI(roi);
        end
        
        function features_hit = FindFeaturesIn(obj, area)
            source = obj.RawDataSource;
            features_hit = siman.List;
            if ~isempty(source)
                list = source.FeatureList;
                for i = 1:list.Count
                    feature = list.ElementAt(i);
                    if feature.HitTest(area)
                        features_hit.Add(feature);
                    end
                end
            end
        end
        
        function hit = HitTest(obj, point)
            pos = get(obj.AxesID, 'Position');
            if point(1) >= pos(1) && point(1) <= (pos(1) + pos(3) - 1)
                if point(2) >= pos(2) && point(2) <= (pos(2) + pos(4) - 1)
                    hit = true;
                    return;
                end
            end
            hit = false;
        end
        
        function OnButtonDown(obj)
            id = obj.AxesID;
            point = get(id, 'CurrentPoint');
            point = [point(1) point(3)];
            obj.ButtonDownPoint = point;
            obj.ButtonDownTime = now;
            h = obj.SelectionRectID;
            if ~isempty(h)
                obj.SelectionRectID = [];
                if ishandle(h)
                    delete(h);
                end
            end
        end
        
        function OnMouseMotion(obj)
            h = obj.SelectionRectID;
            axesID = obj.AxesID;
            if isempty(h)
                t = elapsedsec(obj.ButtonDownTime, now);
                if t > .2
                    point1 = obj.ButtonDownPoint;
                    point2 = get(axesID, 'CurrentPoint');
                    point2 = [point2(1) point2(3)];
                    [~, ~, area] = getvalidrect(point1, point2);
                    if area(3) == 0
                        area(3) = .00001;
                    end
                    if area(4) == 0
                        area(4) = .00001;
                    end
                    
                    obj.SelectionRectID = rectangle('Position', area, 'EdgeColor', [.8, .8, 1], 'Parent', axesID);
                end
            else
                point1 = obj.ButtonDownPoint;
                point2 = get(axesID, 'CurrentPoint');
                point2 = [point2(1) point2(3)];
                [~, ~, area] = getvalidrect(point1, point2);
                if area(3) == 0
                    area(3) = .00001;
                end
                if area(4) == 0
                    area(4) = .00001;
                end
                set(h, 'Position', area);
            end
        end
        
        function OnButtonUp(obj)
            if isempty(obj.ButtonDownPoint)
                return;
            end
            if obj.IsDataOutOfDate % Single click causes redraw if image is out of date
                obj.QueueRecreate();
                return;
            end
            h = obj.SelectionRectID;
            if ~isempty(h)
                obj.SelectionRectID = [];
                if ishandle(h)
                    delete(h);
                end
            end
            point1 = obj.ButtonDownPoint;
            obj.ButtonDownPoint = [];
            point2 = get(obj.AxesID, 'CurrentPoint');
            point2 = [point2(1) point2(3)];
            [point1, point2] = getvalidrect(point1, point2); % convert points to lower-left, upper-right
            area = [point1; point2];
            [x, y]  = obj.ConvertPointsToPlot(area(:,1), area(:,2));
            z = obj.FrameIndex;
            area = [x y [z; z]];
            hitFeatures = obj.FindFeaturesIn(area);
            source = obj.RawDataSource;
            if ~isempty(source)
                source.SelectFeatures(hitFeatures, 'select');
            end
        end
        
        function OnPropertyChanged(obj, src, evt)
            switch evt.PropertyName
                case 'ShowColorbar'
                    obj.ResetColorbar();
                    obj.QueueRedraw();
                case 'LastProcessingDefName'
                case 'Direction'
                    obj.ResetDirection();
                    obj.QueueRedraw();
                    obj.QueueRedrawCell();
                    obj.QueueRedrawFeatures();
                case 'Zoom'
                    if (obj == obj.Parent.CurrentPlot)
                        obj.Parent.RefreshZoomControl();  % Bad form to do here, but works for now
                    end
                    obj.QueueRedraw();
                case {'Colormap', 'ColorLimMode', ...
                        'ColorLim_Lower', 'ColorLim_Upper', ...
                        'ColorPercentLim_Lower',  'ColorPercentLim_Upper', ...
                        'ColorLimActual_Lower',  'ColorLimActual_Upper'}
                    obj.QueueResetColor();
                otherwise
                    obj.QueueRedraw();
            end
            
            switch evt.PropertyName
                case 'ColorLimMode'
                    notify(obj, 'PropertyLayoutChanged');
            end
        end
        
        function ResetDirection(obj)
            id = obj.AxesID;
            switch obj.Direction
                case 'top-bottom'
                    set(id, 'YDir', 'reverse', 'XDir', 'normal');
                case 'bottom-top'
                    set(id, 'YDir', 'normal', 'XDir', 'normal');
                case 'left-right'
                    set(id, 'YDir', 'normal', 'XDir', 'normal');
                case 'right-left'
                    set(id, 'YDir', 'normal', 'XDir', 'reverse');
            end
        end
        
        function enabling = GetEnabling(obj, id, prop, context)
            switch prop
                otherwise
                    enabling = true;
            end
        end
        
        function SetDataSource(obj, source)
            if isequal(source, obj.RawDataSource)
                return
            end
            cla(obj.AxesID);
            if ~isempty(obj.SourceListeners)
                delete(obj.SourceListeners)
            end
            obj.RawDataSource = source;
            if ~isempty(obj.RawDataSource)
                listeners = addlistener(obj.RawDataSource, 'CellMaskUpdated', @obj.OnCellUpdated);
                listeners(2) = addlistener(obj.RawDataSource, 'FeaturesChange', @obj.OnFeaturesChanged);
                listeners(3) = addlistener(obj.RawDataSource, 'FeatureChange', @obj.OnFeaturesChanged);
                obj.SourceListeners = listeners;
            end
            if isempty(obj.CurrentProcessingDef)
                obj.SetDefinition('raw');
            else
                obj.SetDefinition(obj.CurrentProcessingDef.Name);
            end
            obj.ResetFrameIndicator();
            obj.ResetFullDataSize();
        end
        
        function SetDefinition(obj, name)
            mgr = siman.ImageManager.Instance;
            try
                def = mgr.FindDefinition(name);
            catch
                def = mgr.FindDefinition('raw');
            end
            if isempty(def) && isempty(obj.CurrentProcessingDef)
                def = mgr.FindDefinition('raw');
            end
            if (isempty(def))
                siman.Debug.ReportProblem('Could not find any processing definition');
            end
            if (length(def) ~= length(obj.CurrentProcessingDef)) || (def ~= obj.CurrentProcessingDef)
                obj.CurrentProcessingDef = def;
            end
        end
        
        function SetBounds(obj, bounds)
            id = obj.AxesID;
            set(id, 'OuterPosition', bounds);
            obj.QueueRedraw();
        end
        
        function OnCellUpdated(obj, src, evnt)
            obj.OnDataOutOfDate();
            obj.QueueRedrawCell();
        end
        
        function OnFeaturesChanged(obj, src, evnt)
            obj.QueueRedrawFeatures();
        end
        
        function OnDefinitionChanged(obj, src, evnt)
            obj.OnDataOutOfDate();
        end
        
        function OnSliderMoved(obj, src, evnt)
            obj.QueueRedraw();
        end
        
        function OnDataOutOfDate(obj)
            obj.IsDataOutOfDate = true;
            obj.ResetAxes();
        end
        
        function OnCursorMotion(obj, src, evt)
            id = obj.AxesID;
            point = get(id, 'CurrentPoint');
            xlim = get(id, 'XLim');
            ylim = get(id, 'YLim');
            xPoint = round(point(1,1));
            yPoint = round(point(1,2));
            if xPoint >= xlim(1) && xPoint <= xlim(2) && yPoint >= ylim(1) && yPoint <= ylim(2)
                str = ['x:' num2str(xPoint) ', y:' num2str(yPoint)];
                imID = obj.ImageID;
                if ~isempty(imID) && ishandle(imID)
                    data = get(imID, 'CData');
                    yRange = get(imID, 'YData');
                    xRange = get(imID, 'XData');
                    xPoint = xPoint - round(xRange(1));
                    yPoint = yPoint - round(yRange(1));
                    dim = size(data);
                    if xPoint >= 1 && xPoint <= dim(2) && yPoint >= 1 && yPoint <= dim(1)
                        if length(dim) == 3
                            str = [str ', RGB: (' num2str(data(yPoint, xPoint, 1)) ', ' ...
                                num2str(data(yPoint, xPoint, 2)) ', ' num2str(data(yPoint, xPoint, 3)) ')'];
                        else
                            str = [str ', val: ' num2str(data(yPoint, xPoint))];
                        end
                    else
                        str = '';
                    end
                end
                obj.Parent.SetIndicatorText(str);
            end
        end
        
        function OnScroll(obj, id)
            type = get(id, 'Tag');
            val = get(id, 'Value');
            maxVal = get(id, 'Max');
            viewOffset = obj.ViewWindowOffset;
            direction = obj.Direction;
            switch type
                case 'Horizontal Scroll'
                    switch direction
                        case {'top-bottom', 'bottom-top', 'left-right'}
                            viewOffset(1) = val;
                        case 'right-left'
                            viewOffset(1) = maxVal-val;
                    end
                case 'Vertical Scroll'
                    switch direction
                        case {'bottom-top', 'left-right', 'right-left'}
                            viewOffset(2) = val;
                        case 'top-bottom'
                            viewOffset(2) = maxVal-val;
                    end
            end
            obj.ViewWindowOffset = viewOffset;
            siman.ImageManager.Instance.ApplyChanges();
        end
        
        function ResetAxes(obj)
            if obj.IsDataOutOfDate
                set(obj.AxesID, 'Visible', 'on', 'Box', 'on', 'XTickLabel', {}, 'YTickLabel', {}, ...
                    'XColor', 'yellow', 'YColor', 'yellow', 'Color', 'yellow', 'LineWidth', 3, ...
                    'Layer', 'top');
            else
                set(obj.AxesID, 'Visible', 'off');
            end
        end
        
        function ResetFullDataSize(obj)
            if isempty(obj.CurrentProcessingDef) || isempty(obj.RawDataSource)
                obj.FullDataSize = [];
            else
                obj.FullDataSize = obj.CurrentProcessingDef.GetOutputSize(obj.RawDataSource);
            end
        end
        
        function full_size = GetFullImageSize(obj)
            fullDataSize = obj.FullDataSize;
            if isempty(fullDataSize)
                ResetFullDataSize(obj);
                fullDataSize = obj.FullDataSize;
            end
            if isempty(fullDataSize)
                full_size = [];
                return;
            end
            if ismember(obj.Direction, {'left-right', 'right-left'})
                full_size = fullDataSize([2 1]); % rotate 90 degrees
            else
                full_size = fullDataSize;
            end
        end
        
        function ShowFeature(obj, feature)
            area = obj.GetVisibleDataArea();
            if feature.IsInView(area)
                return;
            end
            centeredPoint = feature.CenteredPoint;
            if (numel(centeredPoint) > 2) && centeredPoint(3) ~= obj.FrameIndex
                obj.FrameIndex = centeredPoint(3);
            else
                obj.QueueRedrawFeatures();
            end
        end
        
        function area = GetVisibleDataArea(obj)
            xRange = obj.GetDataXRange();
            yRange = obj.GetDataYRange();
            area = [xRange; yRange];
            z = obj.FrameIndex;
            area = [area [z; z]];
        end
        
        function range = GetDataXRange(obj)
            if isempty(obj.ViewImageXRange)
                range = [];
                return;
            end
            if ismember(obj.Direction, {'left-right', 'right-left'})
                range = obj.ViewImageYRange; % rotate 90 degrees
            else
                range = obj.ViewImageXRange;
            end
        end
        
        function range = GetDataYRange(obj)
            if isempty(obj.ViewImageYRange)
                range = [];
                return;
            end
            if ismember(obj.Direction, {'left-right', 'right-left'})
                range = obj.ViewImageXRange; % rotate 90 degrees
            else
                range = obj.ViewImageYRange;
            end
        end
        
        function ResetSliderLayout(obj)
            vertID = obj.VerticalScrollID;
            horizID = obj.HorizontalScrollID;
            if obj.HasError
                set([vertID horizID], 'Visible', 'off');
                return;
            end
            
            viewOffset = obj.ViewWindowOffset;
            zoom = obj.Zoom/100;
            %zoom = 2;
            axes = obj.AxesID;
            axesPos = get(axes, 'Position');
            direction = obj.Direction;
            fullImageSize = obj.GetFullImageSize();
            fullImageWidth = fullImageSize(1);
            fullImageHeight = fullImageSize(2);
            
            axesWidth = axesPos(3);
            axesHeight = axesPos(4);
            
            fullPixelWidth = fullImageWidth*zoom;
            if axesWidth > fullPixelWidth
                set(horizID, 'Visible', 'off');
            else
                windowPercent = axesWidth/fullPixelWidth;
                windowSize = windowPercent * fullImageWidth;
                maxVal = fullImageWidth-windowSize;
                bigStep = windowPercent/(1-windowPercent);
                smallStep = min(1, bigStep/10);
                viewOffset(1) = min(viewOffset(1), maxVal);
                switch direction
                    case 'left-right'
                        val = viewOffset(1);
                    case 'right-left'
                        val = maxVal-viewOffset(1);
                    case 'top-bottom'
                        val = viewOffset(1);
                    case 'bottom-top'
                        val = viewOffset(1);
                end
                pos = [axesPos(1) axesPos(2)-12 axesWidth 12];
                set(horizID, 'Position', pos, 'Visible', 'on', ...
                    'Min', 0, 'Max', maxVal, 'SliderStep', [smallStep bigStep], 'Value', val);
            end
            
            fullPixelHeight = fullImageHeight*zoom;
            if axesHeight > fullPixelHeight
                set(vertID, 'Visible', 'off');
            else
                windowPercent = axesHeight/fullPixelHeight;
                windowSize = windowPercent * fullImageHeight;
                maxVal = fullImageHeight-windowSize;
                bigStep = windowPercent/(1-windowPercent);
                smallStep = min(1, bigStep/10);
                viewOffset(2) = min(viewOffset(2), maxVal);
                switch direction
                    case 'left-right'
                        val = viewOffset(2);
                    case 'right-left'
                        val = viewOffset(2);
                    case 'top-bottom'
                        val = maxVal-viewOffset(2);
                    case 'bottom-top'
                        val = viewOffset(2);
                end
                pos = [axesPos(1)+axesWidth axesPos(2) 12 axesHeight];
                pos(3) = max(1, pos(3));
                pos(4) = max(1, pos(4));
                smallStep = max(0, smallStep);
                bigStep = max(bigStep, smallStep);
                set(vertID, 'Position', pos, 'Visible', 'on', ...
                    'Min', 0, 'Max', maxVal, 'SliderStep', [smallStep bigStep], 'Value', val);
            end
            obj.ResetViewWindow();
        end
        
        function ResetViewWindow(obj)
            viewOffset = obj.ViewWindowOffset;
            zoom = obj.Zoom/100;
            %zoom = 2;
            axesID = obj.AxesID;
            axesPos = get(axesID, 'Position');
            direction = obj.Direction;
            fullImageSize = obj.GetFullImageSize();
            fullImageWidth = fullImageSize(1);
            fullImageHeight = fullImageSize(2);
            
            %obj.ViewImageXRange = [0 dim(1)] + viewOffset(1);
            %obj.ViewImageYRange = [0 dim(2)] + viewOffset(2);
            
            axesPixelWidth = axesPos(3);
            axesPixelHeight = axesPos(4);
            
            maxImageDisplayWidth = axesPixelWidth/zoom;
            if maxImageDisplayWidth > fullImageWidth  % Image does not go outside visible area
                obj.ViewImageXRange = [1 fullImageWidth];
                xLim = [1 maxImageDisplayWidth];
                extraSpace = maxImageDisplayWidth - fullImageWidth;
                xLim = xLim - extraSpace/2; % centered
            else % zoomed
                xLim = [1 maxImageDisplayWidth] + viewOffset(1);
                if (xLim(2) > fullImageWidth) % prevent moving offscreen
                    xLim = xLim - (xLim(2)-fullImageWidth);
                end
                obj.ViewImageXRange = round(xLim);
            end
            
            maxImageDisplayHeight = axesPixelHeight/zoom;
            if maxImageDisplayHeight > fullImageHeight  % Image does not go outside visible area
                obj.ViewImageYRange = [1 fullImageHeight];
                yLim = [1 maxImageDisplayHeight];
                extraSpace = maxImageDisplayHeight - fullImageHeight;
                yLim = yLim - extraSpace/2; % centered
            else % zoomed
                yLim = [1 maxImageDisplayHeight] + viewOffset(2);
                if (yLim(2) > fullImageHeight) % prevent moving offscreen
                    yLim = yLim - (yLim(2)-fullImageHeight);
                end
                obj.ViewImageYRange = round(yLim);
            end
            yLim(2) = max(yLim(2), yLim(1) + .00001);
            set(axesID, 'XLim', xLim, 'YLim', yLim);
        end
        
        function ChangeFrame(obj, offset)
            dim = obj.CurrentProcessingDef.GetOutputSize(obj.RawDataSource);
            if numel(dim) < 3
                obj.FrameIndex = 1;
                return
            end
            
            index = obj.FrameIndex;
            if isempty(index)
                index = 1;
            end
            index = index + offset;
            if index < 1
                index = 1;
            elseif index > dim(3)
                index = dim(3);
            end
            obj.FrameIndex = index;
        end
        
        function ResetFrameIndicator(obj)
            if ~isempty(obj.Parent)
                if isempty(obj.FrameCount) || obj.FrameCount <= 1
                    str = '';
                else
                    str = ['Frame ' num2str(obj.FrameIndex) ' of ' num2str(obj.FrameCount)];
                end
                obj.Parent.SetFrameText(str);
            end
        end
        
        function [xOut, yOut, zOut] = ConvertPointsToPlot(obj, x, y, z)
            if nargin > 3
                zOut = z;
            end
            if obj.IsRotated
                xOut = y;
                yOut = x;
            else
                xOut = x;
                yOut = y;
            end
        end
        
        function delete(obj)
            delete(obj.MouseListener);
            obj.MouseListener = [];
        end
    end
end