classdef InventoryPanel < siman.ToolPanel
    % DisplayWindow
    properties (SetObservable)
    end
    properties (Transient)
        SliderID
        AxesID
        PlotsPanel
        InventoryListener
    end
    methods
        function obj = InventoryPanel(parent)
            obj@siman.ToolPanel(parent);
        end
        
        function InitDefaults(obj)
            obj.Name = 'Image Inventory';
            obj.DockedLocation = 'east';
        end
        
        function InitPanel(obj)
            InitPanel@siman.ToolPanel(obj);

            panelID = obj.PanelID;
            obj.AxesID = axes('Parent', panelID, 'YDir', 'reverse', 'Visible', 'off', ...
                'DataAspectRatio', [1 1 1], 'PlotBoxAspectRatioMode', 'auto', ...
                'Tag', 'Inventory Axes', 'Units', 'pixels');
            
            import siman.*;
            mgr = ImageManager.Instance;
            obj.InventoryListener = addlistener(mgr, 'InventoryChanged', @obj.OnInventoryChanged);
            obj.PlotsPanel = obj.Parent.GetPanel('PlotsPanel');
        end
        
        function OnInventoryChanged(obj, src, evt)
            obj.QueueRedraw();
        end
        
        function Redraw(obj)
            import siman.*;
            mgr = ImageManager.Instance;
            
            panelID = obj.PanelID;
            panelPos = get(panelID, 'Position');
            
            ax = obj.AxesID; cla(ax);
            list = mgr.GetInventoryList();
            count = list.Count;
            if count == 0
                return
            end
            
            cmap = colormap;
            dim = size(cmap);
            cmapSize = dim(1);
            
            offsets = zeros(count, 1);
            spacing = 20;
            labelOffset = 16;
            mouseCallback = {@inventorySelect, obj};
            
            minX = 0;
            maxY = 0;
            for i = 1:count
                imObj = list.ElementAt(i);
                thumb = double(imObj.GetThumbnail());
                dim = size(thumb);
                if dim(1) > 2*dim(2)
                    thumb = thumb(1:2*dim(2),:,:);
                end
                thumb = imresize(thumb, [NaN 100]);
                name = imObj.Name;
                dim = size(thumb);
                if i < count
                    offsets(i+1) = offsets(i) + spacing + dim(1);
                end
                
                xdata = [1:dim(2)] - dim(2)/2;
                ydata = [1:dim(1)] + offsets(i);
                minX = min(minX, - dim(2)/2);
                maxY = ydata(end) + 10;
                if ~imObj.IsRGB
                    lim = prctile(thumb(:), [5 95]);
                    k = cmapSize/(lim(2)-lim(1));
                    thumb = (thumb - lim(1))*k;
                    if isfloat(thumb)
                        thumb = thumb + 1; % Correct for different treatment of ints and floats as indexes
                    end
                end
                    
                image('Parent', ax, 'XData', xdata, 'YData', ydata, 'CData', thumb, ...
                    'CDataMapping', 'direct', 'EraseMode', 'background', ...
                    'ButtonDownFcn', mouseCallback, 'Tag', imObj.Name, 'UserData', imObj);
                text(0, offsets(i) + dim(1) + labelOffset, name, ...
                    'Parent', ax, 'HorizontalAlignment', 'center');
            end
            
            set(ax, 'XLim', [minX-10 -minX+10], 'YLim', [1 maxY]);
            
            function inventorySelect(src, evt, obj)
                type = get(gcbf, 'SelectionType');
                source = get(src, 'UserData');
                switch type
                    case 'open'
                        obj.PlotsPanel.SetDataSources(source);
                end
                mgr = siman.ImageManager.Instance;
                mgr.ApplyChanges();
            end
        end
                
        function delete(obj)
            if ~isempty(obj.InventoryListener)
                delete(obj.InventoryListener);
                obj.InventoryListener = [];
            end
        end

    end
end