classdef ObservableCollection < matlab.mixin.Copyable
    % Must be a subclass of handle
    properties
        Collection = {};
        DeleteElementsOnClear = false;
        DeleteElementsOnDelete = false;
        NotifyOnItemChange = false;
        Count = -1;
        IsMulticlass = false;
    end
    properties (Transient)
        ElementListeners
        IsAdding = false;
        AllocationBatchSize = 100;
    end
    
    events
        CollectionChange
        ItemChange
    end
    
    methods % Properties
        function count = get.Count(obj)
            count = obj.Count;
            if count == -1
                count = length(obj.Collection);
            end
        end
        
        function set.Collection(obj, value)
            obj.Collection = value;
            if ~obj.IsAdding % the set is triggered by just making the collection bigger, and in
                             % that case we only want to manually add the one listener
                obj.InitElementListeners();
            end
        end
        function set.NotifyOnItemChange(obj, value)
            obj.NotifyOnItemChange = value;
            obj.InitElementListeners()
        end
    end
    
    methods
        function obj = ObservableCollection()
        end
        
        function InitElementListeners(obj)
            if ~obj.NotifyOnItemChange
                return
            end
            if ~isempty(obj.ElementListeners)
                delete(obj.ElementListeners);
                obj.ElementListeners = [];
            end
            if obj.Count == 0
                obj.ElementListeners = [];
                return;
            end
            
            if obj.IsMulticlass
                listeners = [];
                for i = 1:obj.Count
                    element = obj.ElementAt(i);
                    if isempty(listeners)
                        listeners = addlistener(element, 'PropertyChanged', @obj.OnElementChanged);
                    else
                        listeners(end+1) = addlistener(element, 'PropertyChanged', @obj.OnElementChanged);
                    end
                end
            else
                listeners = addlistener([obj.Collection{1:obj.Count}], 'PropertyChanged', @obj.OnElementChanged);
            end
            obj.ElementListeners = listeners;
        end
        
        function OnElementChanged(obj, src, evnt)
            notify(obj, 'ItemChange');
        end
            
        function AddElementListener(obj, item)
            if obj.NotifyOnItemChange
                if isa(item, 'siman.Notifier')
                    if isempty(obj.ElementListeners)
                        obj.ElementListeners = addlistener(item, 'PropertyChanged', @obj.OnElementChanged);
                    else
                        obj.ElementListeners(end + 1) = addlistener(item, 'PropertyChanged', @obj.OnElementChanged);
                    end
                end
            end
        end
        
        function Insert(obj, index, item)
            obj.IsAdding = true;
            if isempty(obj.Collection)
                obj.Count = 0;
            end
            if length(obj.Collection) < index
                collection = cell(1, numel(obj.Collection) + obj.AllocationBatchSize);
                collection(1:numel(obj.Collection)) = obj.Collection;
                obj.Collection = collection;
            end
            if (obj.Count >= index)
                obj.Collection(index+1 : obj.Count+1) = obj.Collection(index : obj.Count);
            end
            obj.Collection{index} = item;
            obj.Count = obj.Count + 1;
            obj.FireCollectionChange('add', item);
            obj.AddElementListener(item);
            obj.IsAdding = false;
        end
        
        function Add(obj, item)
            if isempty(obj.Collection)
                obj.Insert(1, item);
            else
                obj.Insert(obj.Count+1, item);
            end
        end
        
        function AppendList(obj, list)
            obj.IsAdding = true;
            count = obj.Count;
            if count == 0
                collection = list.Collection;
                count = list.Count;
            else
                collection = obj.Collection;
                collection(count + 1:count + list.Count) = list.Collection(1:list.Count);
                count = count + list.Count;
            end
            obj.Collection = collection;
            obj.Count = count;
            obj.FireCollectionChange('append list', list);
            obj.IsAdding = false;
            obj.InitElementListeners();
        end
        
        function Remove(obj, item)
            index = obj.IndexOf(item);
            if isempty(index)
                return;
            end
            count = obj.Count;
            obj.Count = count - 1;
            obj.Collection(index:count-1) = obj.Collection(index+1:count);
            obj.Collection{count} = [];
            obj.InitElementListeners();
            obj.FireCollectionChange('remove', item);
        end
        
        function RemoveIndexes(obj, indexes)
            if isempty(indexes)
                return;
            end
            %removedObjects = obj.Collection(indexes);
            obj.Count = obj.Count - numel(indexes);
            obj.Collection(indexes) = [];
            obj.InitElementListeners();
            obj.FireCollectionChange('removeindexes', indexes);
        end
        
        function item = ElementAt(obj, index)
            if obj.Count < index
                error('Siman:IndexOutOfBounds', 'Attempt to access collection out of bounds.');
            end
            item = obj.Collection{index};
        end
        
        function Swap(obj, index, index2)
            if obj.Count < index || index < 1 || obj.Count < index2 || index2 < 1
                error('Siman:IndexOutOfBounds', 'Attempt to access collection out of bounds.');
            end
            temp = obj.Collection{index};
            obj.Collection{index} = obj.Collection{index2};
            obj.Collection{index2} = temp;
            obj.FireCollectionChange('order', index);
        end
        
        function Replace(obj, index, element)
            if obj.Count < index || index < 1
                error('Siman:IndexOutOfBounds', 'Attempt to access collection out of bounds.');
            end
            obj.Collection{index} = element;
            obj.InitElementListeners();
            obj.FireCollectionChange('replace', index);
        end
        
        function index = IndexOf(obj, item)
            count = obj.Count;
            index = [];
            if isempty(item)
                return;
            end
            for i = 1:count
                if ~isempty(obj.Collection{i}) && obj.Collection{i} == item
                    index = i;
                    return;
                end
            end
        end
        
        function found = IsMember(obj, item)
            found = ~isempty(obj.IndexOf(item));
        end
        
        function Clear(obj)
            delete(obj.ElementListeners);
            obj.ElementListeners = [];
            if obj.DeleteElementsOnClear
                for i = 1:length(obj.Collection)
                    item = obj.Collection{i};
                    if ~isempty(item) && isobject(item)
                        delete(item);
                    end
                end
            end
            obj.Count = 0;
            obj.Collection = {};
            obj.FireCollectionChange('remove all');
        end
        
        function array = ToCellArray(obj)
            if obj.Count == 0
                array = {};
            else
                array = obj.Collection(1:obj.Count);
            end
        end
        
        function array = ToArray(obj)
            if obj.Count == 0
                array = [];
            else
                array = [obj.Collection{1:obj.Count}];
            end
        end
        
        function delete(obj)
            if obj.DeleteElementsOnDelete
                for i = 1:length(obj.Collection)
                    item = obj.Collection{i};
                    if ~isempty(item) && isobject(item)
                        delete(item);
                    end
                end
            end
            obj.Collection = [];
            delete(obj.ElementListeners);
            obj.ElementListeners = [];
        end
        
        % Removed since subsref overloading also overloads method calling
        % (can't figure out how to get around this)
%         function result = subsref(obj, format)
%             %subsref - Overload to allow direct subscripting of collections
%             if ~strcmp(format(1).type, '()')
%                 error('Siman:BadCollectionSyntax', 'Bad indexing for collection.  Only () supported.')
%             end
%             
%             % obj(:).fieldname syntax (returns a cell array of values and errors if
%             % not all objects have that field)
%             if ischar(format(1).subs) && strcmp(format(1).subs, ':')
%                 if length(format) == 2
%                     error('Siman:BadCollectionSyntax', 'Bad indexing for collection.  Only (:).(field) supported.')
%                 end
%                 if ~strcmp(format(2).type, '.')
%                     error('Siman:BadCollectionSyntax', 'Bad indexing for collection.  Only (:).(field) supported.')
%                 end
%                 fieldname = format(2).subs{1};
%                 count = obj.Count;
%                 result = cell(count);
%                 for i = 1:count
%                     element = obj.ElementAt(i);
%                     if ~isprop(element, fieldname)
%                         error('Siman:InvalidCollectionElementProp', ['Collection element does not have property:' fieldname])
%                     end
%                     result{i} = element.(fieldname);
%                 end
%             end
%             result = obj.ElementAt(format(1).subs{1});
%             if length(format) > 1
%                 result = subsref(result, format(2:end));
%             end
%         end
        
        function FireCollectionChange(obj, type, varargin)
            if nargin > 2
                notify(obj, 'CollectionChange', siman.CollectionChangeEventData(type, varargin{:}));
            else
                notify(obj, 'CollectionChange', siman.CollectionChangeEventData(type));
            end
        end
    end
    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@matlab.mixin.Copyable(obj);
            % Make a deep copy of the collection
            for i = 1:length(cpObj.Collection)
                if ~isempty(cpObj.Collection{i})
                    cpObj.Collection{i} = copy(cpObj.Collection{i});
                end
            end
            obj.InitElementListeners();
        end
    end
end
