classdef PlotsPanel < siman.ToolPanel
    % PlotsPanel
    properties
        Plots
        CurrentPlot
    end
    properties (SetObservable)
        BackgroundColor = [1 1 1];
    end
    properties (Transient)
        SelectedPlots
        
        MessageAxesID
        CursorIndicatorID
        FrameIndicatorID
        ZoomControlID
        PlotPanelID
        
        ButtonDownPlot
    end
    properties (Constant)
        ZoomOptions = {'25%', '50%', '75%', '100%', '200%', '400%', 'custom...'};
    end
   
    events
        ChangedCurrentPlot
    end
    
    methods % Properties
        function set.CurrentPlot(obj, value)
            obj.CurrentPlot = value;
            obj.RefreshZoomControl();
            notify(obj, 'ChangedCurrentPlot');
        end
        function list = get.SelectedPlots(obj)
            list = siman.List;
            if ~isempty(obj.CurrentPlot) && ~isempty(obj.CurrentPlot.RawDataSource)
                list.Add(obj.CurrentPlot)
            end
        end
    end
    methods
        % Constructor
        function obj = PlotsPanel(parent)
            obj@siman.ToolPanel(parent);
            %obj.Name = 'Plots';
            obj.Plots = siman.List;
            obj.DockedLocation = 'center';
        end
        
        function InitPanel(obj)
            InitPanel@siman.ToolPanel(obj);

            panelID = obj.PanelID;
            controlParams.DataObject = obj;
            
            set(panelID, 'BorderType', 'none');
            obj.PlotPanelID = uipanel(panelID, 'BorderType', 'none', ...
                'Units', 'pixels', 'Title', '', ...
                'BackgroundColor', obj.BackgroundColor);
            obj.CursorIndicatorID = uicontrol(panelID,'Style','text',...
                'FontSize', 10, ...
                'BackgroundColor', obj.Parent.Color, ...
                'HorizontalAlignment', 'right', ...
                'Tag', 'Indicator Text Control');
            obj.FrameIndicatorID = uicontrol(panelID,'Style','text',...
                'FontSize', 10, ...
                'BackgroundColor', obj.Parent.Color, ...
                'HorizontalAlignment', 'left', ...
                'Tag', 'Frame Text Control');
            
            controlParams.Width = 80;
            controlParams.OffsetType = 'lower left';
            controlParams.ParentPanel = obj.PlotPanelID;
            controlParams.Value = find(strcmp(obj.ZoomOptions, '100%'));
            controlParams.Controller = obj;
            controlParams.Callback = @obj.ControlChanged;
            controlParams.RefreshCallback = @obj.UpdateControl;
            controlParams.ListOptions = obj.ZoomOptions;
            [~, h] = siman.UIHelper.MakePopup('Zoom', [1 1], controlParams);
            obj.ZoomControlID = h;
            
            obj.MessageAxesID = axes('Parent', panelID, 'Visible', 'off', 'Color', 'none', ...
                'XLim', [0 1], 'YLim', [0 1], 'Tag', 'Message Axes');
        end
        
        function SetIndicatorText(obj, text)
            indicatorID = obj.CursorIndicatorID;
            set(indicatorID, 'String', text);
        end
        
        function SetFrameText(obj, text)
            indicatorID = obj.FrameIndicatorID;
            set(indicatorID, 'String', text);
        end
        
        function plot = FindHitPlot(obj, point)
            pos = get(obj.PlotPanelID, 'Position');
            point(1) = point(1) - pos(1) + 1;
            point(2) = point(2) - pos(2) + 1;
            
            for i = 1:obj.Plots.Count
                plot = obj.Plots.ElementAt(i);
                if plot.HitTest(point)
                    return
                end
            end
            plot = [];
        end
        
        function OnButtonDown(obj, point) % override parent
            plot = FindHitPlot(obj, point);
            if ~isempty(plot)
                plot.OnButtonDown();
                obj.ButtonDownPlot = plot;
            end
        end
        
       function OnMouseMotion(obj, point) % override parent
            plot = obj.ButtonDownPlot;
            if ~isempty(plot)
                plot.OnMouseMotion();
            end
        end
        
       function OnButtonUp(obj, point) % override parent
            plot = obj.ButtonDownPlot;
            if ~isempty(plot)
                obj.ButtonDownPlot = [];
                plot.OnButtonUp();
            end
        end
        
        function RefreshPlotControls(obj)
            obj.RefreshZoomControl();
        end
        
        function SetZoom(obj, str_value)
            if isempty(obj.CurrentPlot)
                return
            end
            if strcmp(str_value, 'custom...')
                warndlg('Not implemented');
                val = obj.CurrentPlot.Zoom;
                return;
            else
                val = str2double(str_value(1:end-1));
            end
            obj.CurrentPlot.Zoom = val;
        end
        
        function SetDataSources(obj, sources, append)
            if nargin < 3
                append = false;
            end
            count = length(sources);
            plots = siman.ImagePlot(count);
            for i = 1:count
                if obj.Plots.Count >= i
                    plots(i) = obj.Plots.ElementAt(i);
                else
                    plots(i) = siman.ImagePlot(obj);
                    obj.Plots.Add(plots(i));
                    obj.QueueLayout();
                    obj.QueueRedraw();
                end
                plots(i).SetDataSource(sources(i));
            end
            obj.CurrentPlot = plots(1);
        end
        
        function ApplyChanges(obj)
            ApplyChanges@siman.ToolPanel(obj);
            
            list = obj.Plots;
            if isempty(list)
                return
            end
            count = list.Count;
            for i = 1:count
                p = list.ElementAt(i);
                p.ApplyChanges();
            end
        end
        
        function tf = HasPendingChanges(obj)
            tf = HasPendingChanges@siman.ToolPanel(obj);
            if tf
                return;
            end
            list = obj.Plots;
            if isempty(list)
                return
            end
            count = list.Count;
            for i = 1:count
                p = list.ElementAt(i);
                tf = p.HasPendingChanges();
                if tf
                    return;
                end
            end
        end
        
        function Redraw(obj)
            list = obj.Plots;
            cla(obj.MessageAxesID);
            if isempty(list) || list.Count == 0
                text(.5, .5, 'No image selected', 'Parent', obj.MessageAxesID, ...
                    'HorizontalAlignment', 'center', 'FontSize', 16, 'Color', [.8 .8 .8], ...
                    'FontAngle', 'italic');
                return
            end
        end
       
        function UpdateControl(obj, id) % Handle manual refreshing
            tag = get(id, 'Tag');
            switch tag
                case 'Zoom'
                    obj.RefreshZoomControl();
                    enabling = true;
                otherwise
                    enabling = 'off';
                    siman.Debug.ReportProblem(['Unknown command to manual UpdateControl: ' Tag]);
            end
            if ~ischar(enabling)
                if enabling
                    enabling = 'on';
                else
                    enabling = 'off';
                end
            end
            set(id, 'Enable', enabling);
        end
        
        function RefreshZoomControl(obj)
            controlID = obj.ZoomControlID;
            if isempty(obj.CurrentPlot)
                return;
            end
            list = obj.ZoomOptions;
            currentValue = obj.CurrentPlot.Zoom;
            if isnumeric(currentValue)
                currentValue = [num2str(round(currentValue)) '%'];
            end
            index = find(strcmp(currentValue, list));
            if isempty(index)
                list{end + 1} = currentValue;
                index = numel(list);
            end
            set(controlID, 'String', list, 'Value', index);
        end
        
        function ControlChanged(obj, control, data_obj, prop)
            switch prop
                case 'Zoom'
                    strings = get(control, 'String');
                    value = get(control, 'Value');
                    obj.SetZoom(strings{value});
            end
        end
        
        function handled = OnKeyPress(obj, event)
            if ~isempty(obj.CurrentPlot)
                handled = obj.CurrentPlot.OnKeyPress(event);
            else
                handled = false;
            end
        end
        
        function handled = OnCommand(obj, command)
            handled = false;
            if ~isempty(obj.CurrentPlot)
                handled = obj.CurrentPlot.OnCommand(command);
            end
        end
    end
end



