classdef (ConstructOnLoad) ProcessingOperation < siman.SimanObject
    % Must be a subclass of handle
    properties (SetObservable, GetObservable)
        Type = 'subtract background';
        Skip = false;
        Stop = false;
    end
    properties (Transient)
        ErrorMessage = 'Not validated'
        IsValid = false
        MorphKernelListener
        BaselineKernelListener
        LastResult
    end
    properties (Dependent)
        Active
        OpProperties
    end
    
    properties (SetObservable, GetObservable)
        BackgroundValue = 0;
        MedianFilterType = 'equal sides';
        MedianFilterShape = 'square';
        MedianFilterWidth = 3;
        MedianFilterXWidth = 3;
        MedianFilterYWidth = 3;
        MorphType = 'dilate';
        MorphThreshold = 2;
        MorphThresholdPercent = .5;
        MorphThresholdActual; % assigned by pre-processing
        MorphThresholdType = 'percent';
        MorphIterations = 1;
        BaselineFormat = 'linescan line'; % linescan line, 2-d, constant
        BaselineCalcMethod = 'minimum region'; % minimum region, manual region, average
        BaselineNormMethod = 'divide'; % divide, subtract, divide - 1
        BaselineLinescanMedianFilterWidth = 3;
        BaselineMinRegionFilterWidth = 9;
        BaselineMinimumRegionWidth = 50;
        BaselineManualRegionStart = 1;
        BaselineManualRegionStop = Inf;
        BaselineSlidingRegionWidth = 10;
        BaselineRollingRadius = 5;
        BaselineResultAction = 'calculate only';
        ThresholdType = 'fixed value';
        ThresholdValue = 3;
        SwapDimensions = 'y,z';
        CollapseDimension = 'z';
        ConvertCaMethod = 'resting';
        ConvertCaKd = 1200;
        ConvertCaRestingConc = 100;
        ConvertCaFMax = 100;
        ConvertCaRf = 100;
    end
    
    properties
        MorphKernel;
        BaselineKernel;
    end
    
    properties (Constant)
        MedianFilterTypeOptions = {'equal sides'};
        MedianFilterShapeOptions = {'square'};
        MorphThresholdTypeOptions = {'percent', 'absolute'};
        BaselineFormatOptions = {'linescan line', '2-d'};
        BaselineCalcMethodOptions = {'minimum region', 'manual region', 'sliding region', 'rolling ball'};
        BaselineNormMethodOptions = {'divide', 'subtract', 'divide - 1'};
        BaselineResultActionOptions = {'calculate only', 'baseline image', 'std dev image'};
        MorphTypeOptions = {'dilate', 'erode', 'open', 'close', 'dilate with threshold', 'erode with threshold'};
        ThresholdTypeOptions = {'fixed value', 'baseline st dev', 'baseline percent'};
        SwapDimensionsOptions = {'y,z'};
        CollapseDimensionOptions = {'z'};
        ConvertCaMethodOptions = {'resting', 'fmin-fmax'};
    end
    
    events
        OperationChange
        OperationLayoutChange
    end
    
    methods (Static = true)
        function types = Types(obj)
            types = { ...
                'subtract background', ...
                'median filter', ...
                'calculate baseline', ...
                'baseline normalization', ...
                'threshold', ...
                'morph', ...
                'swap dimensions', ...
                'collapse dimension', ...
                'convert to Ca', ...
                'restore original', ...
                };
        end
    end
    
    methods % Properties
        function set.MorphKernel(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.MorphKernel = [];
                obj.MorphKernelListener = [];
                return
            end
            if ~isempty(obj.MorphKernel)
                delete(obj.MorphKernel);
                if ~isempty(obj.MorphKernelListener)
                    delete(obj.MorphKernelListener)
                end
            end
            obj.MorphKernel = value;
            if ~isempty(value)
                obj.MorphKernelListener = addlistener(obj.MorphKernel, 'PropertyChanged', @obj.OnMorphKernelChanged);
                obj.MorphKernelListener(2) = addlistener(obj.MorphKernel, 'LayoutChanged', @obj.OnMorphKernelLayoutChanged);
            end
        end
        function set.BaselineKernel(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.BaselineKernel = [];
                obj.BaselineKernelListener = [];
                return
            end
            if ~isempty(obj.BaselineKernel)
                delete(obj.BaselineKernel);
                if ~isempty(obj.BaselineKernelListener)
                    delete(obj.BaselineKernelListener)
                end
            end
            obj.BaselineKernel = value;
            if ~isempty(value)
                obj.BaselineKernelListener = addlistener(obj.BaselineKernel, 'PropertyChanged', @obj.OnBaselineKernelChanged);
                obj.BaselineKernelListener(2) = addlistener(obj.BaselineKernel, 'LayoutChanged', @obj.OnBaselineKernelLayoutChanged);
            end
        end
        function result = get.Active(obj)
            result = ~obj.Skip;
        end
        function set.Active(obj, val)
            obj.Skip = ~val;
        end
        function set.ErrorMessage(obj, val)
            obj.ErrorMessage = val;
            obj.FireOperationChange();
        end
        function props = get.OpProperties(obj)
            switch obj.Type
                case 'subtract background'
                    props = {'BackgroundValue'};
                case 'median filter'
                    props = {'MedianFilterShape', 'MedianFilterWidth'};
                case 'calculate baseline'
                    props = {'BaselineFormat', 'BaselineMethod', 'BaselineNormMethod'};
                case 'baseline normalization'
                    props = {'BaselineFormat', 'BaselineMethod', 'BaselineNormMethod'};
                case 'threshold'
                    props = {'ThresholdUnits', 'ThresholdValue'};
                case 'morph'
                    props = {'MorphType', 'MorhKernel', 'MorphThreshold', 'MorphThresholdType', 'MorphIterations'};
                case 'swap dimensions'
                    props = {'SwapDimensions'};
                case 'collapse dimension'
                    props = {'CollapseDimension'};
                case 'convert to Ca'
                    props = {'ConvertToCaMethod', 'ConvertToCaKd', 'ConvertToCaRestingConc'};
                otherwise
                    props = {};
            end
        end
    end
    
    methods
        % Constructor
        function obj = ProcessingOperation()
            obj@siman.SimanObject();
            obj.Skip = false;
            obj.Stop = false;
            
            obj.InitListeners();
            obj.InitConfiguration();
        end
        
        function InitListeners(obj)
            obj.SelfListener = addlistener(obj, 'PropertyChanged', @obj.OnPropertyChanged);
        end
        
        function InitConfiguration(obj)
            if strcmp(obj.Type, 'morph')
                if isempty(obj.MorphKernel)
                    obj.MorphKernel = siman.FilterKernel;
                end
            elseif ~isempty(obj.MorphKernel)
                delete(obj.MorphKernel);
                delete(obj.MorphKernelListener);
                obj.MorphKernel = [];
            end
            if strcmp(obj.Type, 'calculate baseline')
                if isempty(obj.BaselineKernel)
                    obj.BaselineKernel = siman.FilterKernel;
                end
            elseif ~isempty(obj.BaselineKernel)
                delete(obj.BaselineKernel);
                delete(obj.BaselineKernelListener);
                obj.BaselineKernel = [];
            end
        end
        
        % Handle enabling of controls tied to a local property
        function enabling = GetEnabling(obj, id, prop, context)
            enabling = true;
        end
        
        function OnMorphKernelChanged(obj, src, event)
            if strcmp(obj.MorphThresholdType, 'percent')
                obj.CalculateMorphThreshold();
            end
            obj.FireOperationChange();
        end
        
        function OnBaselineKernelChanged(obj, src, event)
            obj.FireOperationChange();
        end
        
        function OnMorphKernelLayoutChanged(obj, src, event)
            obj.FireOperationLayoutChange();
        end
        
        function OnBaselineKernelLayoutChanged(obj, src, event)
            obj.FireOperationLayoutChange();
        end
        
        function OnPropertyChanged(obj, src, evnt)
            switch evnt.PropertyName
                case 'Type'
                    obj.InitConfiguration();
                    obj.FireOperationChange();
                otherwise
                    obj.FireOperationChange();
            end
            if ismember(evnt.PropertyName, {'MorphThreshold', 'MorphThresholdPercent', 'MorphThresholdType'})
                obj.CalculateMorphThreshold();
            end
            % Indicate need for layout change
            % (Should test individual properties, but this is robust and works 95% of the time at least)
            if ischar(obj.(evnt.PropertyName))
                obj.FireOperationLayoutChange();
            end
        end
        
        function OnOperationChanged(obj)
            obj.ClearStoredResults();
        end
        
        function stored = GetStoredResults(obj)
            stored = obj.LastResult;
        end
        
        function ClearStoredResults(obj)
            obj.LastResult = [];
        end
        
        function FireOperationChange(obj)
            notify(obj, 'OperationChange');
        end
        
        function FireOperationLayoutChange(obj)
            notify(obj, 'OperationLayoutChange');
        end
        
        function delete(obj)
            if ~isempty(obj.MorphKernel)
                delete(obj.MorphKernel);
                delete(obj.MorphKernelListener);
                obj.MorphKernel = [];
            end
            if ~isempty(obj.BaselineKernel)
                delete(obj.BaselineKernel);
                delete(obj.BaselineKernelListener);
                obj.BaselineKernel = [];
            end
        end
    end
    
    methods % Custom operation methods
        function CalculateMorphThreshold(obj)
            if ~ismember(obj.MorphType, {'dilate with threshold', 'erode with threshold'})
                return
            end
            switch obj.MorphThresholdType
                case 'absolute'
                    threshold = obj.MorphThreshold;
                case 'percent'
                    if isempty(obj.MorphKernel) || ~isvalid(obj.MorphKernel)
                        return;
                    else
                        percent = obj.MorphThresholdPercent;
                        if percent > 1
                            percent = percent/100;
                        end
                        kernelPointCount = obj.MorphKernel.KernelElementCount;
                        threshold = round(percent * kernelPointCount);
                    end
            end
            obj.MorphThresholdActual = threshold;
        end
    end

    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@siman.SimanObject(obj);
            cpObj.InitListeners();
            % Make a deep copy of the definitions
            if ~isempty(obj.MorphKernel)
                cpObj.MorphKernel = [];
                cpObj.MorphKernel = copy(obj.MorphKernel);
            end
            if ~isempty(obj.BaselineKernel)
                cpObj.BaselineKernel = [];
                cpObj.BaselineKernel = copy(obj.BaselineKernel);
            end
        end
    end
end







