function [list, message] = Detect(obj, target)

method = obj.DetectionMethod;
switch method
    case 'mean position'
        [list, message] = DetectByMeanPosition(obj, target);
    otherwise
        list = siman.List;
        message = ['Unknown detection method: ' obj.DetectionMethod];
end
end

function [list, message] = DetectByMeanPosition(obj, target)
    list = siman.List;
    if isempty(target)
        message = 'No image target';
        return;
    end
    message = '';
    
    unitsDef = obj.GetUnitsDefinition();
    if isempty(unitsDef)
        message = ['Bad units definition name: ' obj.UnitsDefinitionName];
        return;
    end
    
    sparkletList = target.GetFeatureType('sparklet');
    sparkletCount = sparkletList.Count;
    if sparkletCount == 0
        return;
    end
    
    try
        unitData = unitsDef.ProcessSource(target);
    catch
        message = 'Error during creation of image for units';
        return
    end
    dim = size(unitData);
    height = dim(1);
    width = dim(2);
    
    sparkletPositions = zeros(sparkletCount, 3);
    peaks = zeros(sparkletCount, 1);
    for i = 1:sparkletCount
        sparklet = sparkletList.ElementAt(i);
        sparkletPositions(i,1) = sparklet.PeakXIndex;
        sparkletPositions(i,2) = sparklet.PeakYIndex;
        sparkletPositions(i,3) = sparklet.PeakZIndex;
        peaks(i) = sparklet.PeakAmplitude;
    end
    
    distances = zeros(sparkletCount, sparkletCount);
    for i = 1:sparkletCount
        for j = 1:sparkletCount
            xDist = sparkletPositions(i,1) - sparkletPositions(j,1);
            yDist = sparkletPositions(i,2) - sparkletPositions(j,2);
            dist = sqrt(xDist^2 + yDist^2);
            distances(i,j) = dist;
            distances(j,i) = dist;
        end
    end
    
    clusterTolerance = obj.ClusterTolerance;
    areaKernel = obj.AreaKernel;
    
    currentSite = 1;
    siteAssignments = zeros(1, sparkletCount);
    sitePositions = zeros(sparkletCount, 2); % Probably bigger than needed, but avoids warnings
    done = false;
    while ~done
        [~, biggestRemainingSparklet] = max(peaks);
        x = sparkletPositions(biggestRemainingSparklet, 1);
        y = sparkletPositions(biggestRemainingSparklet, 2);
        cluster = find(distances(:, biggestRemainingSparklet) < clusterTolerance);
        if numel(cluster) > 1 % Need to look for new center
            x = round(mean(sparkletPositions(cluster, 1)));
            y = round(mean(sparkletPositions(cluster, 2)));
            
            xDist = sparkletPositions(:,1) - x;
            yDist = sparkletPositions(:,2) - y;
            dist2 = sqrt(xDist.^2 + yDist.^2);
            cluster = find(dist2 < clusterTolerance);
        end
        
        distances(:, cluster) = Inf;
        distances(cluster, :) = Inf;
        peaks(cluster) = -Inf;
        sparkletPositions(cluster, :) = Inf;
        siteAssignments(cluster) = currentSite;
        sitePositions(currentSite, 1) = x;
        sitePositions(currentSite, 2) = y;
        
        done = ~any(siteAssignments == 0);
        currentSite = currentSite + 1;
    end
    
    siteCount = currentSite - 1;
    thumbCrosssectionRadius = round(obj.ThumbnailWidth/2);
    
    % Create feature objects        
    for i = 1:siteCount
        indexes = find(siteAssignments == i);
        sparkletCount = numel(indexes);
        siteSparkletList = siman.List(sparkletCount);
        frameCount = 0;
        zIndexes = [];
        for j = 1:sparkletCount
            sparklet = sparkletList.ElementAt(indexes(j));
            siteSparkletList.Add(sparklet);
            frameCount = frameCount + sparklet.FrameCount;
            zIndexes = union(zIndexes, sparklet.LowerZBound:sparklet.UpperZBound);
        end
      
        feature = siman.SparkletSite();
        feature.Name = ['SparkletSite' num2str(i)];
        feature.Index = i;
        feature.Sparklets = siteSparkletList;
        feature.SparkletCount = siteSparkletList.Count;
        feature.RegionZIndexes = sort(zIndexes);
        
        % search for z-trace that has the largest peak (best indicator of center)
        kernel = areaKernel.StructuringElement.getnhood;
        kernelSize = size(kernel);
        [xOffsets, yOffsets] = areaKernel.GetKernelOffsets();
        centerX = sitePositions(i, 1);
        centerY = sitePositions(i, 2);
        maxAmplitude = -Inf;
        xRange = bounded(floor([centerX-clusterTolerance/2 centerX+clusterTolerance/2]), 1, width, floor(kernelSize(2)/2));
        yRange = bounded(floor([centerY-clusterTolerance/2 centerY+clusterTolerance/2]), 1, height, floor(kernelSize(1)/2));
        for x = xRange(1):xRange(2)
            for y = yRange(1):yRange(2)
                trace = get_z_trace(unitData, [x, y], kernel);
                [val, index] = max(trace);
                if val > maxAmplitude
                    maxAmplitude = val;
                    siteTrace = trace;
                    maxFrame = index;
                    sitePositions(i, 1) = x;
                    sitePositions(i, 2) = y;
                end
            end
        end
        centerX = sitePositions(i, 1);
        centerY = sitePositions(i, 2);
        
        feature.PeakAmplitude = maxAmplitude;
        feature.Area = numel(find(kernel));
        feature.FrameCount = frameCount;
        
        feature.PeakXIndex = sitePositions(i, 1);
        feature.PeakYIndex = sitePositions(i, 2);
        feature.PeakZIndex = maxFrame;
        feature.LowerXBound = centerX + min(xOffsets);
        feature.UpperXBound = centerX + max(xOffsets);
        feature.LowerYBound = centerY + min(yOffsets);
        feature.UpperYBound = centerY + max(yOffsets);
        
        % Calculate some nice stored items (waveform through peak and image thumbnail)
        startX = max(1, feature.PeakXIndex - thumbCrosssectionRadius);
        stopX = min(width, feature.PeakXIndex + thumbCrosssectionRadius);
        startY = max(1, feature.PeakYIndex - thumbCrosssectionRadius);
        stopY = min(height, feature.PeakYIndex + thumbCrosssectionRadius);
        thumbnail = unitData(startY:stopY, startX:stopX, 1);
        
        feature.Thumbnail = thumbnail;
        feature.Waveform = siteTrace;
        
        blankFrame = zeros([height, width]);
        areaIndexes = centerY + yOffsets + (centerX + xOffsets - 1) * height;
        blankFrame(areaIndexes) = 1;
        [~, highestIndex] = min(areaIndexes);
        [xIndexes, yIndexes] = xyindexes(areaIndexes(highestIndex), [height, width]);
        [topCol, topRow] = xyindexes(areaIndexes(highestIndex), [height, width]);
        contour = bwtraceboundary(blankFrame, [topRow, topCol], 'NE');
        feature.Outlines = {contour};
        feature.RegionXIndexes = xIndexes;
        feature.RegionYIndexes = yIndexes;
        
        feature.CalculateExtraProperties(obj);
        list.Add(feature);
    end
    target.AddFeatures(list);
    disp(['Detected ' num2str(list.Count) ' sparklet sites']);
end




