function [list, message] = Detect(obj, target)

method = obj.DetectionMethod;
switch method
    case 'detection image'
        [list, message] = DetectStandard(obj, target);
    otherwise
        list = siman.List;
        message = ['Unknown detection method: ' obj.DetectionMethod];
end
end

function [list, message] = DetectStandard(obj, target)
    list = siman.List;
    if isempty(target)
        message = 'No image target';
        return;
    end
    message = '';
    
    unitsDef = obj.GetUnitsDefinition();
    if isempty(unitsDef)
        message = ['Bad units definition name: ' obj.UnitsDefinitionName];
        return;
    end
    
    detectionDef = obj.DetectionDefinition;
    units = detectionDef.GetOutputUnits();
    if ~strcmp(units, 'binary')
        message = 'Detection image is not binary';
        return
    end
    
    try
        data = detectionDef.ProcessSource(target);
    catch ex
        message = 'Error during creation of detection image';
        return;
    end
    
    connectivityType = str2double(obj.ConnectivityType);
    result = bwconncomp(data, connectivityType);
    
    rawCount = result.NumObjects;
    if result.NumObjects == 0
        return
    end
    
    % Filter out features that are too big or too small
    rawIndexList = result.PixelIdxList;
    validFeatures = cell(rawCount, 2);
    minSize = obj.MinSize;
    maxSize = obj.MaxSize;
    validCount = 0;
    for i = 1:rawCount
        len = numel(rawIndexList{i});
        if len > minSize && len < maxSize
            validCount = validCount + 1;
            validFeatures{validCount} = rawIndexList{i};
        end
    end

    if validCount == 0
        return
    end
    
    % We have features, so get the image where the peak can be taken from (different units)
    try
        unitData = unitsDef.ProcessSource(target);
    catch
        message = 'Error during creation of image for units';
        return
    end
    if ~isequal(size(unitData), size(data))
        message = 'Unequal sizes between detection and unit images';
        return
    end
    
    dim = size(data);
    width = dim(2);
    height = dim(1);
    thumbYStartOffset = obj.ThumbnailStartOffset;
    thumbYStopOffset = obj.ThumbnailStopOffset;
    thumbXRadius = round(obj.ThumbnailWidth/2);
    traceXRadius = round(obj.TraceWidth/2);
    traceLength = thumbYStopOffset - thumbYStartOffset + 1;
    
    % Create feature objects        
    for i = 1:validCount
        indexes = validFeatures{i};
        values = unitData(indexes);
        [peakValue, index] = max(values);
        peakIndex = indexes(index);
        [xIndexes, yIndexes] = xyindexes(indexes, [height width]);
      
        feature = siman.Spark();
        feature.Name = ['Spark' num2str(i)];
        feature.Index = i;
        feature.PeakIndex = peakIndex;
        feature.PeakXIndex = xIndexes(index);
        feature.PeakYIndex = yIndexes(index);
        feature.PeakAmplitude = peakValue;
        feature.Area = numel(indexes);
        
        feature.LowerXBound = min(xIndexes);
        feature.UpperXBound = max(xIndexes);
        feature.LowerYBound = min(yIndexes);
        feature.UpperYBound = max(yIndexes);
        
        % Calculate some nice stored items (waveform through peak and image thumbnail)
        trace = zeros(traceLength, 1)*nan;
        startY = max(1, feature.PeakYIndex + thumbYStartOffset);
        stopY = min(height, feature.PeakYIndex + thumbYStopOffset);
        if (feature.PeakYIndex + thumbYStartOffset < 1)
            startY_trace = 1 - (feature.PeakYIndex + thumbYStartOffset);
        else
            startY_trace = 1;
        end
        stopY_trace = startY_trace + (stopY-startY+1) - 1;
        startX = max(1, feature.PeakXIndex - traceXRadius);
        stopX = min(width, feature.PeakXIndex + traceXRadius);
        trace(startY_trace:stopY_trace) = mean(unitData(startY:stopY, startX:stopX, 1), 2);
        startX = max(1, feature.PeakXIndex - thumbXRadius);
        stopX = min(width, feature.PeakXIndex + thumbXRadius);
        thumbnail = unitData(startY:stopY, startX:stopX, 1);
        
        feature.WaveformStartIndex = startY;
        feature.WaveformStopIndex = stopY;
        feature.WaveformStartOffset = thumbYStartOffset;
        feature.WaveformStopOffset = thumbYStopOffset;
        feature.Thumbnail = thumbnail;
        feature.Waveform = trace;
        feature.RegionIndexes = indexes;
        feature.RegionXIndexes = xIndexes;
        feature.RegionYIndexes = yIndexes;
        
        [~, highestIndex] = min(indexes);
        topCol = xIndexes(highestIndex);
        topRow = yIndexes(highestIndex);
        contour = bwtraceboundary(data, [topRow, topCol], 'E');
        feature.Outlines = {contour};
        
        feature.CalculateExtraProperties(obj);
        list.Add(feature);
    end
    target.AddFeatures(list);
end




