classdef (ConstructOnLoad) SparksMethod < siman.SimanObject
    % SparksMethod.
    properties (Constant)
        FeatureType = 'spark';
        DetectionDefName = 'spark detection';
        UnitsDefName = 'spark units';
        
        DetectionMethodOptions = {'detection image'};
        ConnectivityTypeOptions = {'4', '8', '6', '18', '26'};
    end
    properties
        DetectionDefinition
        UnitsDefinition
    end
    properties (Dependent)
        HasDetectionDefinition
        HasUnitsDefinition
    end
    properties (Transient)
        DetectionDefListener
        UnitsDefListener
    end
    properties (GetObservable, SetObservable) % Registered properties
        UnitConversion_Space = 1;
        UnitConversion_Time = 1;
        DetectionMethod = 'detection image';
        UnitsDefinitionName = 'spark units';
        ConnectivityType = '4';
        MinSize = 0;
        MaxSize = Inf;
        ThumbnailStartOffset = -10;
        ThumbnailStopOffset = 100;
        ThumbnailWidth = 50;
        TraceWidth = 1;
    end
    
    events
        MethodChange
        MethodLayoutChange
    end
    
    methods % Properties
        function set.DetectionDefinition(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.DetectionDefinition = [];
                obj.DetectionDefListener = [];
                return
            end
            if ~isempty(obj.DetectionDefinition) && isvalid(obj.DetectionDefinition)
                delete(obj.DetectionDefinition);
                if ~isempty(obj.DetectionDefListener)
                    delete(obj.DetectionDefListener)
                end
            end
            obj.DetectionDefinition = value;
            if ~isempty(value)
                obj.DetectionDefListener = addlistener(obj.DetectionDefinition, 'DefinitionChange', @obj.OnDefinitionChanged);
            end
        end
        function set.UnitsDefinition(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.UnitsDefinition = [];
                obj.UnitsDefListener = [];
                return
            end
            if ~isempty(obj.UnitsDefinition) && isvalid(obj.UnitsDefinition)
                delete(obj.UnitsDefinition);
                if ~isempty(obj.UnitsDefListener)
                    delete(obj.UnitsDefListener)
                end
            end
            obj.UnitsDefinition = value;
            if ~isempty(value)
                obj.UnitsDefListener = addlistener(obj.UnitsDefinition, 'DefinitionChange', @obj.OnDefinitionChanged);
            end
        end
        function tf = get.HasDetectionDefinition(obj)
            tf = ~isempty(obj.DetectionDefName);
        end
        function tf = get.HasUnitsDefinition(obj)
            tf = ~isempty(obj.UnitsDefName);
        end
    end
    
    methods
        % Constructor
        function obj = SparksMethod()
            obj@siman.SimanObject();
            
            obj.InitListeners();
        end
        
        function InitListeners(obj)
            obj.SelfListener = addlistener(obj, 'PropertyChanged', @obj.OnPropertyChanged);
        end
        
        function OnPropertyChanged(obj, src, evnt)
            switch evnt.PropertyName
                otherwise
                    notify(obj, 'MethodChange');
            end
            % Indicate need for layout change
            switch evnt.PropertyName
                case 'DetectionMethod'
                    notify(obj, 'MethodLayoutChange');
            end
        end
        
        function OnDefinitionChanged(obj, src, evnt)
            notify(obj, 'MethodChange');
        end
        
       % Handle enabling of controls tied to a local property
        function enabling = GetEnabling(obj, id, prop, context)
            enabling = true;
        end
        
        function def = GetDefaultDetectionDefinition(obj)
            def = siman.ProcessingDefinition;
            def.Name = obj.DetectionDefName;
            def.IsSystemDefinition = true;
            op = def.AddOperation('subtract background');
            op = def.AddOperation('median filter');
            op = def.AddOperation('calculate baseline');
            op = def.AddOperation('threshold');
            op.ThresholdType = 'baseline st dev';
            op = def.AddOperation('morph');
            op.MorphType = 'erode';
            op.MorphKernel.Width = 5;
            op = def.AddOperation('morph');
            op.MorphType = 'dilate';
        end
        
        function def = GetDefaultUnitsDefinition(obj)
            def = siman.ProcessingDefinition;
            def.Name = obj.UnitsDefName;
            def.IsSystemDefinition = true;
            op = def.AddOperation('subtract background');
            op = def.AddOperation('median filter');
            op = def.AddOperation('calculate baseline');
            op = def.AddOperation('baseline normalization');
        end
              
        function def = GetUnitsDefinition(obj)
            mgr = siman.ImageManager.Instance;
            try
                def = mgr.FindDefinition(obj.UnitsDefinitionName);
            catch
                def = [];
            end
        end
        
        function ApplyChanges(obj)
            obj.DetectionDefinition.ApplyChanges();
            obj.UnitsDefinition.ApplyChanges();
        end
        
        function delete(obj)
            if ~isempty(obj.DetectionDefinition)
                delete(obj.DetectionDefinition);
                delete(obj.DetectionDefListener);
                obj.DetectionDefinition = [];
            end
            if ~isempty(obj.UnitsDefinition)
                delete(obj.UnitsDefinition);
                delete(obj.UnitsDefListener);
                obj.UnitsDefinition = [];
            end
        end
        
    end
    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@siman.SimanObject(obj);
            cpObj.InitListeners();
            % Make a deep copy of the definitions
            if ~isempty(obj.DetectionDefinition)
                cpObj.DetectionDefinition = [];
                cpObj.DetectionDefinition = copy(obj.DetectionDefinition);
            end
            if ~isempty(obj.UnitsDefinition)
                cpObj.UnitsDefinition = [];
                cpObj.UnitsDefinition = copy(obj.UnitsDefinition);
            end
        end
    end
end





