classdef ToolPanel < siman.SimanObject
    % DisplayWindow
    properties
        Name
        Parent
        Stretchable = false;
    end
    properties (SetObservable, GetObservable)
        Size = [200, 200];  % One or both values used
        Location = [0, 0];  % Only used for undocked panels
        DockedLocation = 'east';
    end
    properties (Transient)
        PanelID
        NeedsLayout = true;
        NeedsRedraw = true;
        
        PlotListeners
        BasePlotListeners
        CurrentAttachedPlot
        PanelDataSource
    end
    
    events
        MouseMotion
    end
    
    methods % Properties
        function set.Name(obj, value)
            obj.Name = value;
            id = obj.PanelID;
            if ishandle(id)
                set(id, 'Title', value);
            end
        end
        function set.CurrentAttachedPlot(obj, plot)
            if isempty(plot) && isempty(obj.CurrentAttachedPlot)
                return;
            end
            if length(plot) ~= length(obj.CurrentAttachedPlot) || plot ~= obj.CurrentAttachedPlot
                if ~isempty(obj.PlotListeners)
                    delete(obj.PlotListeners);
                    obj.PlotListeners = [];
                end
                obj.CurrentAttachedPlot = plot;
                if ~isempty(plot)
                    obj.PlotListeners = addlistener(obj.CurrentAttachedPlot, 'DefinitionSwitched', @obj.OnPlotDefinitionSwitched);
                    obj.PlotListeners(2) = addlistener(obj.CurrentAttachedPlot, 'SourceSwitched', @obj.OnPlotSourceSwitched);
                    obj.PlotListeners(3) = addlistener(obj.CurrentAttachedPlot, 'PropertyLayoutChanged', @obj.OnPlotPropertyLayoutChanged);
                    obj.PlotListeners(4) = addlistener(obj.CurrentAttachedPlot, 'PropertyChanged', @obj.OnPlotPropertyChanged);
                end
                obj.OnAttachedPlotChanged('plot');
            end
        end
    end
    
    methods
        % Constructor
        function obj = ToolPanel(parent)
            obj@siman.SimanObject();
            obj.Parent = parent;
            obj.BasePlotListeners = addlistener(parent, 'MouseMotion', @obj.OnCursorMotion);
            obj.BasePlotListeners(2) = addlistener(parent, 'CurrentPlotChanged', @obj.OnCurrentPlotChanged);
            obj.InitDefaults();
            obj.InitPanel();
            obj.CurrentAttachedPlot = parent.CurrentPlot;
        end
        
        function InitDefaults(obj)
            % hook function that allows subclasses to set property defaults
            % before initialization
        end
        
        function InitPanel(obj)
            figID = obj.Parent.FigureID;
            loc = obj.Location;
            sz = obj.Size;
            pos = [loc sz];
            obj.PanelID = uipanel(figID, 'BorderType', 'beveledout', ...
                'FontSize', siman.UIHelper.FONT_SIZE + 2, ...
                'Units', 'pixels', 'Position', pos, 'Title', obj.Name, ...
                'BackgroundColor', obj.Parent.Color);
        end
        
        function SetBounds(obj, pos)
            panelID = obj.PanelID;
            set(panelID, 'Position', pos);
            obj.OnPanelResized();
        end
        
        function hit = HitTest(obj, point)
            pos = get(obj.PanelID, 'Position');
            if point(1) >= pos(1) && point(1) <= (pos(1) + pos(3) - 1)
                if point(2) >= pos(2) && point(2) <= (pos(2) + pos(4) - 1)
                    hit = true;
                    return;
                end
            end
            hit = false;
        end
        
        function OnPanelResized(obj)
            obj.QueueLayout();
        end
        
        function OnButtonDown(obj, point)
        end
        
        function OnCursorMotion(obj, src, evt)
            notify(obj, 'MouseMotion');
        end
        
        function OnMouseMotion(obj, point)
            % stub.  no default action.
        end
        
        function OnButtonUp(obj, point)
        end
        
        function OnCurrentPlotChanged(obj, src, evt)
            obj.CurrentAttachedPlot = obj.Parent.CurrentPlot;
            obj.OnAttachedPlotChanged('plot');
        end
        
        function OnPlotSourceSwitched(obj, src, evt)
            obj.OnAttachedPlotChanged('source');
        end
        
        function OnPlotDefinitionSwitched(obj, src, evt)
            obj.OnAttachedPlotChanged('definition');
        end
        
        function OnPlotPropertyChanged(obj, src, evt)
            % No default action;
        end
        
        function OnPlotPropertyLayoutChanged(obj, src, evt)
            % No default action;
        end
        
        function OnAttachedPlotChanged(obj, type)
            % Stub function for any descendent to respond to a type of change in the currently
            % selected plot.  Change "type" can be 'plot', 'source', or 'definition'
        end
              
        function SetPanelDataSource(obj, source)
            obj.PanelDataSource = source;
            obj.QueueRedraw();
        end
        
        function source = GetPanelDataSource(obj)
            source = obj.PanelDataSource;
        end
        
        function ids = GetControlIDs(obj)
            ids = findobj(obj.PanelID);
            ids(1) = [];
            if numel(ids) <= 1
                return;
            end
            parents = cell2mat(get(ids, 'Parent'));
            type = cellstr(get(parents, 'Type'));
            ids(strcmp('axes', type)) = [];
        end
        
        % Main function for telling controls to refresh
        % (UIHelper will determine if standard refresh, command refresh, or custom refresh)
        function RefreshControls(obj)
            fig = obj.Parent.FigureID;
            if strcmp(get(fig, 'BeingDeleted'), 'on')
                return;
            end
            controls = obj.GetControlIDs();
            source = obj.GetPanelDataSource();
            types = cellstr(get(controls, 'Type'));
            if isempty(source)
                set(controls(strcmp(types, 'uicontrol')), 'Enable', 'off');
                return
            end
            for i = 1:length(controls)
                id = controls(i);
                isUIControl = strcmp(types{i}, 'uicontrol');
                if isUIControl
                    siman.UIHelper.RefreshControl(id);
                end
            end
        end
        
        % Stub function to catch missing enable handlers that fall through (of standard refresh callback)
        function enabling = GetEnabling(obj, id, prop, context)
            switch prop
                otherwise
                    siman.Debug.ReportProblem(['Unknown control to refresh: ' get(id, 'Tag')]);
                    enabling = false;
            end
        end
        
        % Stub function to catch missing custom control change handlers
        function ControlChanged(obj, control, data_obj, prop)
            siman.Debug.ReportProblem(['No custom control change code for ' prop]);
        end
        
        % Stub function to catch missing command refresh handlers
        function enabling = ResetCommandEnabling(obj, cmd, context)
            switch cmd.Name
                case {'ToolPanel.Close'}
                    enabling = true;
                otherwise
                    siman.Debug.ReportProblem(['No command enable code for ' cmd.Name]);
                    enabling = false;
            end
        end
        
        % Stub function to catch missing command handlers
        function OnCommand(obj, cmd, id, context)
            switch cmd.Name
                case 'ToolPanel.Close'
                otherwise
                    siman.Debug.ReportProblem(['No command execution code for ' get(id, 'Tag')]);
            end
        end
        
        function QueueLayout(obj)
            obj.NeedsLayout = true;
            obj.QueueRedraw();
        end
        
        function QueueRedraw(obj)
            obj.NeedsRedraw = true;
        end
        
        function Redraw(obj)
            obj.RefreshControls();
        end
        
        function ApplyChanges(obj)
            if obj.NeedsLayout
                obj.ResetLayout();
                obj.NeedsLayout = false;
            end
            if obj.NeedsRedraw
                obj.Redraw();
                obj.NeedsRedraw = false;
            end
        end
        
        function tf = HasPendingChanges(obj)
            tf = obj.NeedsLayout || obj.NeedsRedraw;
        end
        
        function handled = OnKeyPress(obj, event)
            handled = false; % No default action
        end
        
        function delete(obj)
            delete(obj.PlotListeners);
            obj.PlotListeners = [];
            delete(obj.BasePlotListeners);
            obj.BasePlotListeners = [];
            
            id = obj.PanelID;
            if ~isempty(id) && ishandle(id)
                set(id, 'Units', 'pixels');
                pos = get(id, 'Position');
                obj.Location = pos([1 2]);
                obj.Size = pos([3 4]);
                delete(id);
            end
        end
    end
    methods (Abstract)
        ResetLayout(obj);
    end
end