classdef (ConstructOnLoad) TransientsMethod < siman.SimanObject
    % TransientsDefinition.
    properties (Constant)
        FeatureType = 'transient';
        DetectionDefName = 'transient detection';
        UnitsDefName = 'transient units';
        
        DetectionMethodOptions = {'derivative'};
        DerivativeThresholdTypeOptions = {'image units', 'percentile'};
    end
    properties (SetObservable, GetObservable)
        DetectionDefinition
        UnitsDefinition
    end
    properties (Dependent)
        HasDetectionDefinition
        HasUnitsDefinition
    end
    properties (Transient)
        DetectionDefListener
        UnitsDefListener
    end
    properties (GetObservable, SetObservable) % Registered properties
        UnitConversion_Space = 1;
        UnitConversion_Time = 1;
        DetectionMethod = 'derivative';
        UnitsDefinitionName = 'transient units';
        DerivativeThresholdType = 'percentile';
        DerivativeThreshold_DataUnits = 10;
        DerivativeThreshold_Percentile = 99;
        DerivativeThreshold_Actual;
        DetectionMedianFilterWidth = 5;
        WaveformStartOffset = -10;
        WaveformStopOffset = 100;
        RefractoryLimit = 50;
        UnitsMedianFilterWidth = 3;
    end
    
    events
        MethodChange
        MethodLayoutChange
    end
    
    methods % Properties
        function set.DetectionDefinition(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.DetectionDefinition = [];
                obj.DetectionDefListener = [];
                return
            end
            if ~isempty(obj.DetectionDefinition) && isvalid(obj.DetectionDefinition)
                delete(obj.DetectionDefinition);
                if ~isempty(obj.DetectionDefListener)
                    delete(obj.DetectionDefListener)
                end
            end
            obj.DetectionDefinition = value;
            if ~isempty(value)
                obj.DetectionDefListener = addlistener(obj.DetectionDefinition, 'DefinitionChange', @obj.OnDefinitionChanged);
            end
        end
        function set.UnitsDefinition(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.UnitsDefinition = [];
                obj.UnitsDefListener = [];
                return
            end
            if ~isempty(obj.UnitsDefinition) && isvalid(obj.UnitsDefinition)
                delete(obj.UnitsDefinition);
                if ~isempty(obj.UnitsDefListener)
                    delete(obj.UnitsDefListener)
                end
            end
            obj.UnitsDefinition = value;
            if ~isempty(value)
                obj.UnitsDefListener = addlistener(obj.UnitsDefinition, 'DefinitionChange', @obj.OnDefinitionChanged);
            end
        end
        function tf = get.HasDetectionDefinition(obj)
            tf = ~isempty(obj.DetectionDefName);
        end
        function tf = get.HasUnitsDefinition(obj)
            tf = ~isempty(obj.UnitsDefName);
        end
    end
    
    methods
        % Constructor
        function obj = TransientsMethod()
            obj@siman.SimanObject();
            
            obj.InitListeners();
        end
        
        function InitListeners(obj)
            obj.SelfListener = addlistener(obj, 'PropertyChanged', @obj.OnPropertyChanged);
        end
        
        function OnPropertyChanged(obj, src, evnt)
            switch evnt.PropertyName
                otherwise
                    notify(obj, 'MethodChange');
            end
            % Indicate need for layout change
            switch evnt.PropertyName
                case {'DetectionMethod', 'DerivativeThresholdType'}
                    notify(obj, 'MethodLayoutChange');
            end
        end
        
        function OnDefinitionChanged(obj, src, evnt)
            notify(obj, 'MethodChange');
        end
        
       % Handle enabling of controls tied to a local property
        function enabling = GetEnabling(obj, id, prop, context)
            enabling = true;
        end
        
        function def = GetDefaultDetectionDefinition(obj)
            def = siman.ProcessingDefinition;
            def.Name = obj.DetectionDefName;
            def.IsSystemDefinition = true;
        end
            
        function def = GetDefaultUnitsDefinition(obj)
            def = siman.ProcessingDefinition;
            def.Name = obj.UnitsDefName;
            def.IsSystemDefinition = true;
        end
        
        function def = GetUnitsDefinition(obj)
            mgr = siman.ImageManager.Instance;
            try
                def = mgr.FindDefinition(obj.UnitsDefinitionName);
            catch
                def = [];
            end
        end
        
        function ApplyChanges(obj)
            obj.DetectionDefinition.ApplyChanges();
            obj.UnitsDefinition.ApplyChanges();
        end
        
        function delete(obj)
            if ~isempty(obj.DetectionDefinition)
                if isvalid(obj.DetectionDefinition)
                    delete(obj.DetectionDefinition);
                end
                delete(obj.DetectionDefListener);
                obj.DetectionDefinition = [];
            end
            if ~isempty(obj.UnitsDefinition)
                if isvalid(obj.UnitsDefinition)
                    delete(obj.UnitsDefinition);
                end
                delete(obj.UnitsDefListener);
                obj.UnitsDefinition = [];
            end
        end
    end
    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@siman.SimanObject(obj);
            cpObj.InitListeners();
            % Make a deep copy of the definitions
            if ~isempty(obj.DetectionDefinition)
                cpObj.DetectionDefinition = [];
                cpObj.DetectionDefinition = copy(obj.DetectionDefinition);
            end
            if ~isempty(obj.UnitsDefinition)
                cpObj.UnitsDefinition = [];
                cpObj.UnitsDefinition = copy(obj.UnitsDefinition);
            end
        end
    end
end





