classdef DataCache < handle
    % DataCache - A singleton class that manages large data sets that objects
    % would like to store for convenience but which can be recreated if
    % needed.  Objects can register a large data matrix with the cache, and
    % the cache will keep it around as long as memory is available. Objects
    % can simply request the data be returned, and if it has been deleted
    % they must recreate it.
    properties (Constant)
        Instance = siman.DataCache;
    end
    properties (Access = private)
        OwnerObjs = {};
        IDs = [];
        CachedData = {};
        LastID = 0;
    end
    methods
        function obj = DataCache()
        end
    end
    methods (Static = true)
        function [value, found] = GetCachedData(id)
            import siman.DataCache;
            dc = DataCache.Instance;
            value = [];
            found = false;
            if ~isnumeric(id) || length(id) ~= 1
                return;
            end
            index = find(dc.IDs == id, 1);
            if isempty(index)
                return;
            end
            found = true;
            value = dc.CachedData{index};
        end
        function [newID, replaced] = SetCachedData(data, id, owner)
            import siman.DataCache;
            dc = DataCache.Instance;
            replaced = false;
            if nargin > 2 && ~isempty(id)
                if ~isnumeric(id) || length(id) ~= 1
                    error('Siman:BadCacheID', 'Cache id must be a single numeric value.');
                end
                newID = id;
                index = find(dc.IDs == id, 1);
                if ~isempty(index)
                    replaced = true;
                    dc.CachedData{index} = data;
                    return
                end
                dc.IDs(end + 1) = newID;
            else
                newID = dc.LastID + 1;
                dc.LastID = newID;
                dc.IDs(end + 1) = newID;
            end
            index = length(dc.IDs);
            dc.OwnerObjs{index} = owner;
            dc.CachedData{index} = data;
        end
        function RemoveCachedData(id)
            if ~isnumeric(id) || length(id) ~= 1
                return;
            end
            cache = siman.DataCache.Instance;
            if isempty(cache) || isempty(cache.IDs)
                return
            end
            index = find(cache.IDs == id, 1);
            if isempty(index)
                return;
            end
        
            siman.DataCache.Instance.IDs(index) = [];
            siman.DataCache.Instance.CachedData(index) = [];
            siman.DataCache.Instance.OwnerObjs(index) = [];
        end
        function found = DataIsCached(id)
            found = ~isempty(id) && ~isempty(find(siman.DataCache.Instance.IDs == id, 1));
        end
        function FreeMemory(memory_size)
            ids = siman.DataCache.Instance.IDs;
            data = siman.DataCache.Instance.CachedData;
            owners = siman.DataCache.Instance.OwnerObjs;
            startingMem = memory('data');
            if memory_size > startingMem
                siman.DataCache.Instance.CachedData = {};
                siman.DataCache.Instance.IDs = [];
                siman.DataCache.Instance.OwnerObjs = {};
            end
            newSize = memory('data');
            while length(data) > 1 && (startingMem - newSize < memory_size)
                data(1) = [];
                ids(1) = [];
                owners(1) = [];
                newSize = memory('data');
            end
            siman.DataCache.Instance.IDs = ids;
            siman.DataCache.Instance.CachedData = data;
            siman.DataCache.Instance.OwnerObjs = owners;
        end
        function CompressCache()
            ids = siman.DataCache.Instance.IDs;
            owners = siman.DataCache.Instance.OwnerObjs;
            data = siman.DataCache.Instance.CachedData;
            invalid = logical(size(ids));
            for i = 1:length(ids)
                if ~isempty(owners{i})
                    invalid(i) = ~isvalid(owners{i});
                end
            end
            ids(invalid) = [];
            data(invalid) = [];
            owners(invalid) = [];
            siman.DataCache.Instance.IDs = ids;
            siman.DataCache.Instance.CachedData = data;
            siman.DataCache.Instance.OwnerObjs = owners;
        end
        function ClearCache()
            siman.DataCache.Instance.IDs = [];
            siman.DataCache.Instance.CachedData = {};
            siman.DataCache.Instance.OwnerObjs = {};
        end     
    end
end





