classdef FileIOHelper < handle
    % UIHelper - utility functions for creating manual ui controls
    
    properties (Constant)
        Instance = siman.FileIOHelper;
    end
    
    properties (Constant)
        SIMAN_FILE_VERSION = 1.0;
    end
    
    properties
        SimanPath
        IsLoading = false;  % Convenience field for noting when object are being created by a load
    end
       
    methods
        function obj = FileIOHelper() % Singleton constructor
            fullpath = mfilename('fullpath');
            localPath = [filesep '+siman' filesep 'FileIOHelper'];
            basePath = strrep(fullpath, localPath, '');
            obj.SimanPath = basePath;
        end
    end
    
    methods (Static)
        function obj = GetHelper()
            obj = siman.FileIOHelper.Instance;
        end
        
        function siman_path = GetSimanPath()
            siman_path = siman.FileIOHelper.Instance.SimanPath;
        end
        
        function delimiter = GetLineDelimiter()
            delimiter = '\n';
            %delimiter = '\r\n';
        end
        
        function [ok, message] = SaveSimanFile(siman_obj, source, prop_stem)
            defaultPath = '';
            if nargin > 2
                pathProp = [prop_stem 'Path'];
                if isempty(source) || ~isprop(source, pathProp)
                    siman.Debug.ReportProblem(['Bad default file stem: ' prop_stem]);
                else
                    defaultPath = source.(pathProp);
                end
            end
            [filename, path] = uiputfile({'*.mat', 'SImAn files'}, ...
                'Choose SImAn Files', defaultPath);
            if isequal(filename, 0)
                ok = 0;
                message = '';
                return
            end
            fullPath = fullfile(path, filename);
            if nargin > 2
                source.(pathProp) = fullPath;
            end
            [ok, message] = siman.FileIOHelper.WriteSimanFile(fullPath, siman_obj);
        end
        
        function [success, message] = WriteSimanFile(filename, siman_obj)
            success = false;
            if nargin < 1 || isempty(siman_obj)
                siman.Debug.ReportProblem('Bad object to write to file.');
                return;
            end
            try
                version = siman.FileIOHelper.SIMAN_FILE_VERSION;
                if isempty(version)
                    siman.Debug.ReportProblem('Missing system version number');
                end
                save(filename, 'siman_obj', 'version');
            catch exception
                siman.Debug.ReportError(exception, 'Error while saving file');
                message = ['Error saving file: ' exception.message];
                return;
            end
            message = '';
            success = true;
        end
        
        function [ok, message, obj] = OpenSimanFile(source, prop_stem)
            defaultPath = '';
            if nargin > 1
                pathProp = [prop_stem 'Path'];
                if isempty(source) || ~isprop(source, pathProp)
                    siman.Debug.ReportProblem(['Bad default file stem: ' prop_stem]);
                else
                    defaultPath = source.(pathProp);
                end
            end
            [filename, path] = uigetfile({'*.mat', 'SImAn files'}, ...
                'Choose SImAn Files', defaultPath);
            if isequal(filename, 0)
                ok = 0;
                message = '';
                return
            end
            fullPath = fullfile(path, filename);
            if nargin > 1
                source.(pathProp) = fullPath;
            end
            [ok, message, obj] = siman.FileIOHelper.LoadSimanFile(fullPath);
        end
        
        function [success, message, obj] = LoadSimanFile(filename)
            success = false;
            if ~exist(filename, 'file')
                message = 'File does not exist';
                errordlg(message, 'Error Loading', 'modal');
                return;
            end
            try
                load(filename);
            catch exception
                siman.Debug.ReportError(exception, 'Error while loading file');
                message = ['Error loading file: ' exception.message];
                errordlg(message, 'Error Loading', 'modal');
                return;
            end
            if exist('version', 'var')
                %
            end
            if ~exist('siman_obj', 'var')
                message = 'Could not find objects in file.';
                return
            end
            obj = siman_obj;
            success = true;
            message = '';
        end
        
        function [ok, message] = SaveImageFile(im, source, prop_stem)
            defaultPath = '';
            if nargin > 2
                pathProp = [prop_stem 'Path'];
                if isempty(source) || ~isprop(source, pathProp)
                    siman.Debug.ReportProblem(['Bad default file stem: ' prop_stem]);
                else
                    defaultPath = source.(pathProp);
                end
            end
            [filename, path] = uiputfile({'*.tiff', 'Tiff files'}, ...
                'Select output file', defaultPath);
            if isequal(filename, 0)
                ok = 0;
                message = '';
                return
            end
            fullPath = fullfile(path, filename);
            if nargin > 2
                source.(pathProp) = fullPath;
            end
            [ok, message] = siman.FileIOHelper.WriteImageFile(fullPath, im);
        end
        
        function [success, message] = WriteImageFile(filename, im)
            success = false;
            if nargin < 1 || isempty(im)
                siman.Debug.ReportProblem('No image data to write to file.');
                return;
            end
            try
                if ~isstruct(im)
                    imwrite(im, filename, 'tiff');
                else
                    data = im.Data;
                    dim = size3d(data);
                    frameCount = dim(3);
                    if isfield(im, 'Colormap')
                        imwrite(data(:,:,1), im.Colormap, filename, 'tif');
                        for i = 2:frameCount
                            imwrite(data(:,:,i), im.Colormap, filename, 'tif', 'WriteMode', 'append');
                        end
                    else
                        imwrite(data(:,:,1), filename, 'tif');
                        for i = 2:frameCount
                            imwrite(data(:,:,i), filename, 'tif', 'WriteMode', 'append');
                        end
                    end
                end
            catch exception
                siman.Debug.ReportError(exception, 'Error while saving image file');
                message = ['Error saving image file: ' exception.message];
                return;
            end
            message = '';
            success = true;
        end
        
        function full_path = ChooseOutputFile(filetypes, source, prop_stem, title)
            defaultPath = '';
            if nargin > 2 && ~isempty(prop_stem)
                pathProp = [prop_stem 'Path'];
                if isempty(source) || ~isprop(source, pathProp)
                    siman.Debug.ReportProblem(['Bad default file stem: ' prop_stem]);
                else
                    defaultPath = source.(pathProp);
                end
            end
            if nargin < 4
                title = 'Select output file';
            end
            [filename, path] = uiputfile(filetypes, title, defaultPath);
            if isequal(filename, 0)
                full_path = '';
                return
            end
            full_path = fullfile(path, filename);
            if nargin > 2
                source.(pathProp) = full_path;
            end
        end
        
        function folder_name = ChooseOutputPath(source, prop_stem, title)
            defaultPath = '';
            if nargin > 1 && ~isempty(prop_stem)
                pathProp = [prop_stem 'Path'];
                if isempty(source) || ~isprop(source, pathProp)
                    siman.Debug.ReportProblem(['Bad default path stem: ' prop_stem]);
                else
                    defaultPath = source.(pathProp);
                end
                if ~exist(defaultPath, 'dir') % If filename, convert to path only
                    if isempty(defaultPath)
                        defaultPath = pwd;
                    else
                        defaultPath = fileparts(defaultPath);
                    end
                end
            end
            if nargin < 3
                title = 'Select output folder';
            end
            folder_name = uigetdir(defaultPath, title);
            if isequal(folder_name, 0)
                folder_name = '';
                return
            end
            if nargin > 1
                source.(pathProp) = folder_name;
            end
        end
        
        
        function full_path = ChooseInputFile(filetypes, source, prop_stem, title)
            defaultPath = '';
            if nargin > 2 && ~isempty(prop_stem)
                pathProp = [prop_stem 'Path'];
                if isempty(source) || ~isprop(source, pathProp)
                    siman.Debug.ReportProblem(['Bad default file stem: ' prop_stem]);
                else
                    defaultPath = source.(pathProp);
                end
            end
            if nargin < 4
                title = 'Select output file';
            end
            [filename, path] = uigetfile(filetypes, title, defaultPath);
            if isequal(filename, 0)
                full_path = '';
                return
            end
            full_path = fullfile(path, filename);
            if nargin > 2
                source.(pathProp) = full_path;
            end
        end
        
    end
end





