classdef List < matlab.mixin.Copyable
    % Must be a subclass of handle
    properties
        Collection = {};
        DeleteElementsOnClear = false;
        DeleteElementsOnDelete = false;
        Count = 0;
    end
    properties (Transient)
        AllocationBatchSize = 50;
    end
    
    methods % Properties
    end
    
    methods
        function obj = List(count) % Known size list can be pre-allocated
            if nargin > 0
                obj.Collection = cell(1, count);
            end
        end
        
        function Insert(obj, index, item)
            if length(obj.Collection) < index
                collection = cell(1, numel(obj.Collection) + obj.AllocationBatchSize);
                collection(1:numel(obj.Collection)) = obj.Collection;
                obj.Collection = collection;
            end
            if (obj.Count >= index)
                obj.Collection(index+1 : obj.Count+1) = obj.Collection(index : obj.Count);
            end
            obj.Collection{index} = item;
            obj.Count = obj.Count + 1;
        end
        
        function AppendList(obj, list)
            count = obj.Count;
            if count == 0
                collection = list.Collection;
                count = list.Count;
            else
                collection = obj.Collection;
                collection(count + 1:count + list.Count) = list.Collection(1:list.Count);
                count = count + list.Count;
            end
            obj.Collection = collection;
            obj.Count = count;
        end
        
        function Add(obj, item)
            if isempty(obj.Collection)
                obj.Insert(1, item);
            else
                obj.Insert(obj.Count+1, item);
            end
        end
        
        function Remove(obj, item)
            index = obj.IndexOf(item);
            if isempty(index)
                return;
            end
            count = obj.Count;
            for i = index+1:count
                obj.Collection(i-1) = obj.Collection(i);
            end
            obj.Collection{count} = [];
            obj.Count = count - 1;
        end
        
        function item = ElementAt(obj, index)
            if obj.Count < index
                error('Siman:IndexOutOfBounds', 'Attempt to access collection out of bounds.');
            end
            item = obj.Collection{index};
        end
        
        function Swap(obj, index, index2)
            if obj.Count < index || index < 1 || obj.Count < index2 || index2 < 1
                error('Siman:IndexOutOfBounds', 'Attempt to access collection out of bounds.');
            end
            temp = obj.Collection{index};
            obj.Collection{index} = obj.Collection{index2};
            obj.Collection{index2} = temp;
        end
        
        function Replace(obj, index, element)
            if obj.Count < index || index < 1
                error('Siman:IndexOutOfBounds', 'Attempt to access collection out of bounds.');
            end
            obj.Collection{index} = element;
        end
        
        function index = IndexOf(obj, item)
            count = obj.Count;
            index = [];
            if isempty(item)
                return;
            end
            for i = 1:count
                if ~isempty(obj.Collection{i}) && obj.Collection{i} == item
                    index = i;
                    return;
                end
            end
        end
        
        function found = IsMember(obj, item)
            found = ~isempty(obj.IndexOf(item));
        end
        
        function Clear(obj)
            if obj.DeleteElementOnClear
                for i = 1:length(obj.Collection)
                    item = obj.Collection{i};
                    if ~isempty(item) && isobject(item)
                        delete(item);
                    end
                end
            end
            obj.Collection = {};
            obj.Count = 0;
        end
        
        function array = ToCellArray(obj)
            if obj.Count == 0
                array = {};
            else
                array = obj.Collection(1:obj.Count);
            end
        end
        
        function array = ToArray(obj)
            if obj.Count == 0
                array = [];
            else
                array = [obj.Collection{1:obj.Count}];
            end
        end
        
        function delete(obj)
            if obj.DeleteElementsOnDelete
                for i = 1:length(obj.Collection)
                    item = obj.Collection{i};
                    if ~isempty(item) && isobject(item)
                        delete(item);
                    end
                end
            end
            obj.Collection = [];
        end
        
        
        function FireCollectionChange(obj, type, varargin)
            if nargin > 2
                notify(obj, 'CollectionChange', siman.CollectionChangeEventData(type, varargin{:}));
            else
                notify(obj, 'CollectionChange', siman.CollectionChangeEventData(type));
            end
        end
    end
    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@matlab.mixin.Copyable(obj);
            % Make a deep copy of the collection
            for i = 1:length(cpObj.Collection)
                if ~isempty(cpObj.Collection{i})
                    cpObj.Collection{i} = copy(cpObj.Collection{i});
                end
            end
        end
    end
end
