classdef UIHelper < handle
    % UIHelper - utility functions for creating manual ui controls
    
    properties (Constant)
        TEXT_HEIGHT = 17;
        VERTICAL_SPACING = 8;
        LABEL_SPACING = 2;
        PANEL_MARGIN = 8;
        PANEL_TOP_OFFSET = 28;
        TOOLBAR_BUTTON_HEIGHT = 30;
        TOOLBAR_BUTTON_WIDTH = 30;
        TOOLBAR_BUTTON_SPACER = 8;
        BUTTON_HEIGHT = 20;
        BUTTON_WIDTH = 60;
        FONT_SIZE = 10;
        LABEL_LETTER_SPACING = 7;
    end
    properties (Constant)
        MESSAGE_ERROR = 1;
        MESSAGE_INFO = 2;
    end
        
    
    methods (Static)
        function delimiter = GetLineDelimiter()
            if ispc
                delimiter = '\r\n';
            else
                delimiter = '\n';
            end
        end
        function output_str = WriteLine(str)
            output_str = sprintf('%s\n', str);
        end
        function [new_location, h] = MakeLabeledTextbox(prop, label, location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitControlChangeParams(params, prop, false);
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.Width, params.TextHeight);
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnControlChanged;
            
            % Make label
            userData.IsLabel = true;
            pos = [location(1) location(2) 100 params.TextHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'text', 'String', label, ...
                'FontSize', params.FontSize, ...
                'Tag', [prop 'Label'], 'Position', pos, 'HorizontalAlignment', 'left', ...
                'BackgroundColor', params.BackgroundColor, 'UserData', userData);
            %extent = get(h, 'Extent');
            %pos(3) = extent(3); % crop label to exact length
            pos(3) = siman.UIHelper.LABEL_LETTER_SPACING * length(label);
            set(h, 'Position', pos);
            userData.LabelID = h;
            
            % Make control
            userData.IsLabel = false;
            pos = [pos(1)+pos(3)+params.LabelHorizontalSpacing pos(2) params.Width params.TextHeight+2];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'edit', ...
                'FontSize', params.FontSize, ...
                'Tag', prop, 'Position', pos, 'Callback', callback, 'HorizontalAlignment', 'left', ...
                'UserData', userData);
            siman.UIHelper.RefreshControl(h);
            
            % Return new position for next control
            new_location = location;
            new_location(2) = pos(2)-params.VerticalSpacing;
        end
        
        function [new_location, h] = MakeLabeledIndicator(prop, label, location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitControlChangeParams(params, prop, false);
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.Width, params.TextHeight);
            
            userData = params.UserData;
            
            % Make label
            userData.IsLabel = true;
            pos = [location(1) location(2) 100 params.TextHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'text', 'String', label, ...
                'FontSize', params.FontSize, ...
                'Tag', [prop 'Label'], 'Position', pos, 'HorizontalAlignment', 'left', ...
                'BackgroundColor', params.BackgroundColor, 'UserData', userData);
            extent = get(h, 'Extent');
            pos(3) = extent(3); % crop label to exact length
            set(h, 'Position', pos);
            userData.LabelID = h;
            
            % Make control
            userData.IsLabel = false;
            pos = [pos(1)+extent(3)+params.LabelHorizontalSpacing pos(2) params.Width params.TextHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'text', ...
                'FontSize', params.FontSize, ...
                'Tag', prop, 'Position', pos, 'HorizontalAlignment', 'left', ...
                'UserData', userData, 'BackgroundColor', params.BackgroundColor);
            siman.UIHelper.RefreshControl(h);
            
            % Return new position for next control
            new_location = location;
            new_location(2) = pos(2)-params.VerticalSpacing;
        end
        
        function [new_location, h] = MakeIndicator(prop, location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitControlChangeParams(params, prop, false);
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.Width, params.TextHeight);
            
            userData = params.UserData;
            
            % Make control
            userData.IsLabel = false;
            pos = [location(1) location(2) params.Width params.TextHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'text', ...
                'FontSize', params.FontSize, ...
                'Tag', prop, 'Position', pos, 'HorizontalAlignment', 'left', ...
                'UserData', userData, 'BackgroundColor', params.BackgroundColor, ...
                'ForegroundColor', params.ForegroundColor);
            siman.UIHelper.RefreshControl(h);
            
            % Return new position for next control
            new_location = location;
            new_location(2) = pos(2)-params.VerticalSpacing;
        end
        
        function [new_location, h] = MakeCheckbox(prop, label, location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitControlChangeParams(params, prop, false);
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.Width, params.TextHeight);
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnControlChanged;
            
            % Make control
            userData.IsLabel = false;
            pos = [location(1) location(2) params.Width params.TextHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'checkbox', ...
                'FontSize', params.FontSize, 'String', label, 'Callback', callback, ...
                'Tag', prop, 'Position', pos, 'HorizontalAlignment', 'left', ...
                'UserData', userData, 'BackgroundColor', params.BackgroundColor, ...
                'ForegroundColor', params.ForegroundColor);
            siman.UIHelper.RefreshControl(h);
            
            % Return new position for next control
            new_location = location;
            new_location(2) = pos(2)-params.VerticalSpacing;
        end
        
        function [new_location, h] = MakeLabeledListbox(prop, label, location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitControlChangeParams(params, prop, true);
            labelSpacing = 3;
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.Width, params.TextHeight+params.Height+labelSpacing);
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnControlChanged;
            
            if ~isfield(params, 'Muliselect')
                params.MultiSelect = false;
            end
            
            % Make label
            userData.IsLabel = true;
            pos = [location(1)+3 location(2)+params.Height+labelSpacing 100 params.TextHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'text', 'String', label, ...
                'FontSize', params.FontSize, ...
                'Tag', [prop 'Label'], 'Position', pos, 'HorizontalAlignment', 'left', ...
                'BackgroundColor', params.BackgroundColor, 'UserData', userData);
            extent = get(h, 'Extent');
            pos(3) = extent(3); % crop label to exact length
            set(h, 'Position', pos);
            userData.LabelID = h;
            
            % Make control
            userData.IsLabel = false;
            pos = [location(1) location(2) params.Width params.Height];
            pos(3) = max(2, pos(3));
            pos(4) = max(2, pos(4));
            options = params.ListOptions;
            if isfield(params, 'Value')
                val = params.Value;
            elseif isprop(params.DataObject, prop)
                val = find(strcmp(params.DataObject.(prop), options));
            else
                val = 1;
            end
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'listbox', ...
                'FontSize', params.FontSize, ...
                'Tag', prop, 'Position', pos, 'Callback', callback, 'HorizontalAlignment', 'left', ...
                'String', params.ListOptions, 'Value', val, 'UserData', userData);
            if params.MultiSelect
                set(h, 'Min', 0, 'Max', 2);
            end
            
            % Return new position for next control
            new_location = location;
            new_location(2) = pos(2)-params.VerticalSpacing;
        end
        
        function [new_location, h] = MakeLabeledPopup(prop, label, location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitControlChangeParams(params, prop, true);
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.Width, params.TextHeight);
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnControlChanged;
            
            % Make label
            userData.IsLabel = true;
            pos = [location(1) location(2) 100 params.TextHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'text', 'String', label, ...
                'FontSize', params.FontSize, ...
                'Tag', [prop 'Label'], 'Position', pos, 'HorizontalAlignment', 'left', ...
                'BackgroundColor', params.BackgroundColor, 'UserData', userData);
            %extent = get(h, 'Extent');
            %pos(3) = extent(3); % crop label to exact length            
            pos(3) = siman.UIHelper.LABEL_LETTER_SPACING * length(label);
            set(h, 'Position', pos);
            userData.LabelID = h;
            
            % Make control
            userData.IsLabel = false;
            pos = [pos(1)+pos(3)+params.LabelHorizontalSpacing pos(2)+3 params.Width params.TextHeight+2];
            options = params.ListOptions;
            if isprop(params.DataObject, prop)
                val = find(strcmp(params.DataObject.(prop), options));
            else
                val = 1;
            end
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'popupmenu', ...
                'FontSize', params.FontSize, ...
                'Tag', prop, 'Position', pos, 'Callback', callback, 'HorizontalAlignment', 'left', ...
                'String', params.ListOptions, 'Value', val, 'UserData', userData);
            siman.UIHelper.RefreshControl(h);
            
            % Return new position for next control
            new_location = location;
            new_location(2) = pos(2)-params.VerticalSpacing-3; %(extra 3 for popup size)
        end
        
        function [new_location, h] = MakePopup(prop, location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitControlChangeParams(params, prop, true);
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.Width, params.TextHeight);
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnControlChanged;
            
            % Make control
            userData.IsLabel = false;
            options = params.ListOptions;
            if isprop(params.DataObject, prop)
                val = find(strcmp(params.DataObject.(prop), options));
            else
                val = 1;
            end
            pos = [location(1) location(2) 100 params.TextHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'popupmenu', ...
                'FontSize', params.FontSize, ...
                'Tag', prop, 'Position', pos, 'Callback', callback, 'HorizontalAlignment', 'left', ...
                'String', options, 'Value', val, 'UserData', userData);
            siman.UIHelper.RefreshControl(h);
            
            % Return new position for next control
            new_location = location;
            new_location(2) = pos(2)-params.VerticalSpacing;
        end
        
        function [new_location, h] = MakeRadioButtonList(prop, labels, location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitControlChangeParams(params, prop, true);
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.Width, params.TextHeight);
            
            if ~isfield(params, 'Orientation')
                params.Orientation = 'vertical';
            end
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnControlChanged;
            
            % Make control
            userData.IsLabel = false;
            options = params.ListOptions;
            count = length(options);
            if length(labels) ~= count
                siman.Debug.ReportProblem(['Bad radio button labels: ' prop]);
            end
            if isprop(params.DataObject, prop)
                val = find(strcmp(params.DataObject.(prop), options));
            else
                val = 0;
            end
            h = zeros(count, 1);
            optionLocation = round(location);
            for i = 1:count
                pos = [optionLocation(1) optionLocation(2) 200 params.TextHeight];
                userData.Option = options{i};
                h(i) = uicontrol('Parent', params.ParentPanel, 'Style', 'radiobutton', ...
                    'FontSize', params.FontSize, ...
                    'Tag', params.Tag, 'Position', pos, 'Callback', callback, 'HorizontalAlignment', 'left', ...
                    'String', labels{i}, 'Value', isequal(val, i), 'UserData', userData, ...
                    'BackgroundColor', params.BackgroundColor);
                switch params.Orientation
                    case 'vertical'
                        optionLocation(2) = optionLocation(2) + params.TextHeight + ...
                            params.VerticalSpacing;
                    case 'horizontal'
                        extent = get(h(i), 'Extent');
                        optionLocation(1) = optionLocation(1) + extent(3) + 25;
                    otherwise
                        siman.Debug.ReportProblem(['Bad radio button orientation: ' prop]);
                end
                siman.UIHelper.RefreshControl(h(i));
            end
            
            % Return new position for next control
            new_location = location;
            switch params.Orientation
                case 'vertical'
                    new_location(2) = pos(2)-params.VerticalSpacing;
                case 'horizontal'
                    new_location(2) = pos(2)-params.VerticalSpacing;
            end
        end
        
        function [new_location, h] = MakeLabel(label, location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.Width, params.TextHeight);
            
            % Make label
            pos = [location(1) location(2) 100 params.TextHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'text', ...
                'FontSize', params.FontSize, ...
                'Tag', 'Notification', 'Position', pos, 'HorizontalAlignment', 'left', ...
                'BackgroundColor', params.BackgroundColor, 'ForegroundColor', params.ForegroundColor, ...
                'String', label);
            extent = get(h, 'Extent');
            pos(3) = extent(3); % crop label to exact length
            set(h, 'Position', pos);
            
            % Return new position for next control
            new_location = location;
            new_location(2) = pos(2)-params.VerticalSpacing;
        end
        
        function [new_location, h] = MakeButton(location, label, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitButtonParams(params);
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.ButtonWidth, params.ButtonHeight);
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnButtonPressed;
            
            pos = [location(1), location(2), params.ButtonWidth, params.ButtonHeight];
            h = uicontrol('Parent', params.ParentPanel, 'Style', 'pushbutton', ...
                'FontSize', params.FontSize, ...
                'Tag', params.Tag, 'Position', pos, 'HorizontalAlignment', 'center', ...
                'BackgroundColor', params.BackgroundColor, 'Callback', callback, ...
                'UserData', userData, 'Tooltip', params.Tooltip, 'String', label);
            siman.UIHelper.RefreshControl(h);
            
            % Return new position for next control
            new_location = location;
            new_location(1) = pos(1)+pos(3);
        end
        
        function [new_location, h] = MakeToolbarButton(location, params)
            params = siman.UIHelper.InitDefaultParams(params); % validate and fill in missing values
            params = siman.UIHelper.InitButtonParams(params);
            location = siman.UIHelper.ConvertLocation(location, params, ...
                params.ToolbarButtonWidth, params.ToolbarButtonHeight);
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnButtonPressed;
            
            pos = [location(1), location(2), params.ToolbarButtonWidth, params.ToolbarButtonHeight];
            if params.IsToggle
                style = 'togglebutton';
            else
                style = 'pushbutton';
            end
            h = uicontrol('Parent', params.ParentPanel, 'Style', style, ...
                'FontSize', params.FontSize, ...
                'Tag', params.Tag, 'Position', pos, 'HorizontalAlignment', 'center', ...
                'BackgroundColor', params.BackgroundColor, 'Callback', callback, ...
                'UserData', userData, 'Tooltip', params.Tooltip);
            if strcmp(params.LabelType, 'string')
                set(h, 'String', params.Label);
            end
            siman.UIHelper.RefreshControl(h);
            
            % Return new position for next control
            new_location = location;
            if ismember(params.LocationType, {'upper left', 'lower left'})
                new_location(1) = pos(1)+pos(3);
            end
            if ismember(params.LocationType, {'upper left', 'upper right'})
                new_location(2) = new_location(2) + params.ToolbarButtonHeight;
            end
        end
        
        function h = MakeMenuItem(label, command, separator, params, accelerator)
            params.CommandName = command;
            params = siman.UIHelper.InitMenuParams(params, false, separator);
            if nargin > 4
                params.Accelerator = accelerator;
            end
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnButtonPressed;
            
            h = uimenu(params.ParentMenu, 'Label', label, 'Tag', params.Tag, ...
                'Callback', callback, 'UserData', userData, 'Accelerator', params.Accelerator, ...
                'Separator', params.Separator);
        end
        
        function h = MakeParentMenu(label, tag, separator, params)
            params = siman.UIHelper.InitMenuParams(params, true, separator);
            
            userData = params.UserData;
            callback = @siman.UIHelper.OnButtonPressed;
            
            h = uimenu(params.ParentMenu, 'Label', label, 'Tag', tag, ...
                'Callback', callback, 'UserData', userData, 'Accelerator', params.Accelerator, ...
                'Separator', params.Separator);
        end
        
        function params_new = InitDefaultParams(params)
            params_new = params;
            
            validParent = isfield(params, 'ParentPanel') || isfield(params, 'ParentMenu');
            siman.Debug.Assert(validParent, 'UI:BadParameter', 'Missing parent panel or menu for creating control');
            if (~isfield(params, 'BackgroundColor'))
                params_new.BackgroundColor = get(params.ParentPanel, 'BackgroundColor');
            end
            if (~isfield(params, 'ForegroundColor'))
                params_new.ForegroundColor = [0 0 0];
            end
            if (~isfield(params, 'Width'))
                params_new.Width = 50;
            end
            if (~isfield(params, 'Height'))
                params_new.Height = 25;
            end
            if (~isfield(params, 'TextHeight'))
                params_new.TextHeight = siman.UIHelper.TEXT_HEIGHT;
            end
            if (~isfield(params, 'FontSize'))
                params_new.FontSize = siman.UIHelper.FONT_SIZE;
            end
            if (~isfield(params, 'ToolbarButtonWidth'))
                params_new.ToolbarButtonWidth = siman.UIHelper.TOOLBAR_BUTTON_HEIGHT;
            end
            if (~isfield(params, 'ToolbarButtonHeight'))
                params_new.ToolbarButtonHeight = siman.UIHelper.TOOLBAR_BUTTON_HEIGHT;
            end
            if (~isfield(params, 'ButtonWidth'))
                params_new.ButtonWidth = siman.UIHelper.BUTTON_HEIGHT;
            end
            if (~isfield(params, 'ButtonHeight'))
                params_new.ButtonHeight = siman.UIHelper.BUTTON_HEIGHT;
            end
            if (~isfield(params, 'VerticalSpacing'))
                params_new.VerticalSpacing = siman.UIHelper.VERTICAL_SPACING;
            end
            if (~isfield(params, 'LabelHorizontalSpacing'))
                params_new.LabelHorizontalSpacing = siman.UIHelper.LABEL_SPACING;
            end
            if (~isfield(params, 'LocationType'))
                params_new.LocationType = 'upper left';
            end
            if (~isfield(params, 'Tooltip'))
                params_new.Tooltip = '';
            end
            if (~isfield(params, 'UserData'))
                params_new.UserData = struct('IsManaged', true);
            else
                isValidStruct = isempty(params.UserData) || isstruct(params.UserData);
                siman.Debug.Assert(isValidStruct, 'UI:BadParameter', 'Bad internal user data for creating control');
                if ~isfield(params.UserData, 'IsManaged')
                    udata = params.UserData;
                    udata.IsManaged = true;
                    params_new.UserData = udata;
                end
            end
        end
        
        function location_new = ConvertLocation(location, params, width, height)
            switch params.LocationType
                case 'lower left'
                    location_new = location;
                case 'upper left'
                    location_new = [location(1) location(2)-height];
                case 'lower right'
                    location_new = [location(1)-width location(2)];
                case 'upper right'
                    location_new = [location(1)-width location(2)-height];
                otherwise
                    siman.Debug.ReportProblem(['Unknown location type given to UIHelper: ' params.LocationType]);
                    location_new = [1 1];
            end
        end
        
        function params_new = InitControlChangeParams(params, prop, init_list)
            params_new = params;
            if (nargin < 3)
                init_list = false;
            end
            userData = params.UserData;
            if (~isfield(params, 'Callback') || isempty(params.Callback))
                params_new.Callback = @siman.UIHelper.HandleGenericControlChange;
                siman.Debug.Assert(isfield(params, 'DataObject'), 'UI:BadParameter', ['Missing data object for creating control: ' prop]);
                dataObj = params.DataObject;
                userData.DataObject = dataObj;
                if ~isempty(dataObj)
                    siman.Debug.Assert(isprop(dataObj, prop), 'UI:BadParameter', ['Bad source property for creating control: ' prop]);
                end
                siman.Debug.Assert(isfield(params, 'Controller'), 'UI:BadParameter', ['Missing controller for creating control: ' prop]);
                userData.Controller = params.Controller;
                if init_list
                    if ~isfield(params, 'ListOptions') || isempty(params.ListOptions)
                        siman.Debug.Assert(isprop(dataObj, [prop 'Options']), 'UI:BadParameter', ['No option list property for creating list control: ' prop]);
                        params_new.ListOptions = dataObj.([prop 'Options']);
                    end
                end
            else
                if init_list
                    if ~isfield(params, 'ListOptions') || isempty(params.ListOptions)
                        siman.Debug.Fail('UI:BadParameter', ['Missing option list for creating list control: ' prop]);
                    end
                end
            end
            userData.Property = prop;
            userData.Callback = params_new.Callback;
            if (~isfield(params, 'RefreshCallback') || isempty(params.RefreshCallback))
                params_new.RefreshCallback = @siman.UIHelper.RefreshGenericControl;
            end
            userData.RefreshCallback = params_new.RefreshCallback;
            if (~isfield(params, 'Context') || isempty(params.Context))
                params_new.Context = [];
            end
            userData.Context = params_new.Context;
            if (~isfield(params, 'Tag'))
                params_new.Tag = prop;
            end
            params_new.UserData = userData;
        end
        
        function params_new = InitButtonParams(params)
            params_new = params;
            userData = params.UserData;
            if (~isfield(params, 'Callback') || isempty(params.Callback)) % Let controls assign own callback if desired
                if ~isfield(params, 'Command') || isempty(params.Command) % If command object missing, look one up
                    if ~isfield(params, 'CommandName') || isempty(params.CommandName)
                        siman.Debug.ReportProblem('Missing command name for creating button');
                        params_new.Callback = @(src, event)warndlg('Not implemented');
                    else
                        params_new.Command = siman.ImageManager.Instance.FindCommand(params.CommandName);
                        if isempty(params_new.Command)
                            siman.Debug.ReportProblem(['Unknown command name for button: ' params.CommandName]);
                            params_new.Callback = @(src, event)warndlg('Not implemented');
                        end
                    end
                end
                if isfield(params_new, 'Command') && ~isempty(params_new.Command)
                    userData.Command = params_new.Command;
                    params_new.Callback = @siman.UIHelper.HandleGenericButtonPress;
                    params_new.Tooltip = params_new.Command.Tooltip;
                    params_new.Label = params_new.Command.Label;
                    if (~isfield(params, 'Tag'))
                        params_new.Tag = userData.Command.Name;
                    end
                else
                    params_new.Callback = @(src, event)warndlg('Not implemented');
                end
            end
            userData.Callback = params_new.Callback;
            if (~isfield(params, 'RefreshCallback') || isempty(params.RefreshCallback))
                params_new.RefreshCallback = @siman.UIHelper.RefreshGenericControl;
            end
            userData.RefreshCallback = params_new.RefreshCallback;
            if (~isfield(params, 'Context') || isempty(params.Context))
                params_new.Context = [];
            end
            userData.Context = params_new.Context;
            params_new.UserData = userData;
            if (~isfield(params, 'LabelType'))
                params_new.LabelType = 'string';
                if (~isfield(params_new, 'Label'))
                    params_new.Label = '?';
                end
            end
            if (~isfield(params, 'IsToggle'))
                params_new.IsToggle = false;
            end
            if (~isfield(params_new, 'Tag'))
                params_new.Tag = '(unknown button)';
            end
        end
        function params_new = InitMenuParams(params, is_list, has_separator)
            params_new = params;
            validParent = isfield(params, 'ParentMenu');
            siman.Debug.Assert(validParent, 'UI:BadParameter', 'Missing parent panel or menu for creating control');
            if (~isfield(params, 'UserData'))
                params_new.UserData = struct('IsManaged', true);
            else
                isValidStruct = isempty(params.UserData) || isstruct(params.UserData);
                siman.Debug.Assert(isValidStruct, 'UI:BadParameter', 'Bad internal user data for creating control');
                if ~isfield(params.UserData, 'IsManaged')
                    udata = params.UserData;
                    udata.IsManaged = true;
                    params_new.UserData = udata;
                end
            end
            userData = params_new.UserData;
            if (~isfield(params, 'Callback') || isempty(params.Callback)) % Let controls assign own callback if desired
                if is_list
                    params_new.Callback = @siman.UIHelper.RefreshGenericMenuList;
                else
                    if ~isfield(params, 'Command') || isempty(params.Command) % If command object missing, look one up
                        if ~isfield(params, 'CommandName') || isempty(params.CommandName)
                            siman.Debug.ReportProblem('Missing command name for creating button');
                            params_new.Callback = @(src, event)warndlg('Not implemented');
                        else
                            params_new.Command = siman.ImageManager.Instance.FindCommand(params.CommandName);
                            if isempty(params_new.Command)
                                siman.Debug.ReportProblem(['Unknown command name for button: ' params.CommandName]);
                                params_new.Callback = @(src, event)warndlg('Not implemented');
                            end
                        end
                    end
                    if isfield(params_new, 'Command') && ~isempty(params_new.Command)
                        userData.Command = params_new.Command;
                        params_new.Callback = @siman.UIHelper.HandleGenericButtonPress;
                        params_new.Tooltip = params_new.Command.Tooltip;
                        params_new.Label = params_new.Command.Label;
                        if (~isfield(params, 'Tag'))
                            params_new.Tag = userData.Command.Name;
                        end
                    else
                        params_new.Callback = @(src, event)warndlg('Not implemented');
                    end
                end
            end
            userData.Callback = params_new.Callback;
            if (~isfield(params, 'RefreshCallback') || isempty(params.RefreshCallback))
                if is_list
                    params_new.RefreshCallback = @siman.UIHelper.RefreshGenericMenuList;
                else
                    params_new.RefreshCallback = @siman.UIHelper.RefreshGenericControl;
                end
            end
            userData.RefreshCallback = params_new.RefreshCallback;
            if (~isfield(params, 'Context') || isempty(params.Context))
                params_new.Context = [];
            end
            userData.Context = params_new.Context;
            params_new.UserData = userData;
            if (~isfield(params_new, 'Tag'))
                params_new.Tag = '(unknown menu)';
            end
            if ~isfield(params, 'Accelerator')
                params_new.Accelerator = '';
            end
            if isempty(has_separator)
                params_new.Separator = 'off';
            else
                params_new.Separator = has_separator;
            end
            if ~ischar(params_new.Separator)
                if has_separator
                    params_new.Separator = 'on';
                else
                    params_new.Separator = 'off';
                end
            end
        end
        
        % Callbacks (all controls go through these routines for centralized processing)
        function OnControlChanged(src, ~)
            userData = get(src, 'UserData');
            if ~isfield(userData, 'Callback')
                siman.Debug.ReportProblem(['No callback function for control change: ' get(src, 'Tag')]);
                return;
            end
            callback = userData.Callback;
            if ~isa(callback, 'function_handle')
                siman.Debug.ReportProblem(['Invalid callback function argument for control change: ' get(src, 'Tag')]);
                return;
            end
            if isfield(userData, 'DataObject')
                dataObject = userData.DataObject;
            else
                dataObject = [];
            end
            if isfield(userData, 'Property')
                prop = userData.Property;
            else
                prop = [];
            end
            try
                feval(callback, src, dataObject, prop);
            catch ex
                siman.Debug.ReportError(ex, ['Error during control change callback: ' get(src, 'Tag')]);
                return;
            end
            
            try
                mgr = siman.ImageManager.Instance;
                mgr.ApplyChanges();
            catch ex
                siman.Debug.ReportError(ex, ['Error during applying control change callback: ' get(src, 'Tag')]);
                return;
            end
        end
        function OnButtonPressed(src, ~)
            userData = get(src, 'UserData');
            if ~isfield(userData, 'Callback')
                siman.Debug.ReportProblem(['No callback function for button press: ' get(src, 'Tag')]);
                return;
            end
            callback = userData.Callback;
            if ~isa(callback, 'function_handle')
                siman.Debug.ReportProblem(['Invalid callback function argument for button press: ' get(src, 'Tag')]);
                return;
            end
            try
                feval(callback, src);
            catch ex
                siman.Debug.ReportError(ex, ['Error during button press callback: ' get(src, 'Tag')]);
                return;
            end
            
            try
                mgr = siman.ImageManager.Instance;
                mgr.ApplyChanges();
            catch ex
                siman.Debug.ReportError(ex, ['Error during applying button press callback; ' get(src, 'Tag')]);
                return;
            end
        end
        function OnMenuOpened(id)
            userData = get(id, 'UserData');
            if ~isfield(userData, 'RefreshCallback')
                siman.Debug.ReportProblem(['No callback function for control refresh: ' get(id, 'Tag')]);
                return;
            end
            callback = userData.RefreshCallback;
            if ~isa(callback, 'function_handle')
                siman.Debug.ReportProblem(['Invalid callback function argument for control refresh: ' get(id, 'Tag')]);
                return;
            end
            try
                feval(callback, id);
            catch ex
                siman.Debug.ReportError(ex, ['Error during control refresh callback: ' get(id, 'Tag')]);
                return;
            end
        end
        function RefreshControl(id)
            userData = get(id, 'UserData');
            if ~isfield(userData, 'RefreshCallback')
                if strcmp(get(id, 'Style'), 'text') % allow generic labels with no callback
                    return
                end
                siman.Debug.ReportProblem(['No callback function for control refresh: ' get(id, 'Tag')]);
                return;
            end
            callback = userData.RefreshCallback;
            if ~isa(callback, 'function_handle')
                siman.Debug.ReportProblem(['Invalid callback function argument for control refresh: ' get(id, 'Tag')]);
                return;
            end
            try
                feval(callback, id);
            catch ex
                siman.Debug.ReportError(ex, ['Error during control refresh callback: ' get(id, 'Tag')]);
                return;
            end
        end
        
        % Control processing
        function HandleGenericControlChange(id, data_obj, prop)
            if isempty(data_obj)
                siman.Debug.ReportProblem('Missing data object for control change');
                return;
            end
            if isempty(prop)
                siman.Debug.ReportProblem('Missing property name for control change');
                return;
            end
            if ~isprop(data_obj, prop)
                siman.Debug.ReportProblem(['No property to assign value called ' prop]);
                return;
            end
            siman.UIHelper.AssignControlValue(id, data_obj, prop);
        end
        
        function AssignControlValue(id, data_obj, prop)
            style = get(id, 'Style');
            switch style
                case 'edit'
                    value = data_obj.(prop);
                    if isnumeric(value)
                        try
                            propVal = str2double(get(id, 'String'));
                        catch ex
                            warnmodaldlg(['Invalid number: ' ex]);
                            return;
                        end
                    else
                        propVal = get(id, 'String');
                    end
                    data_obj.(prop) = propVal;
                case {'popupmenu', 'listbox'}
                    options = get(id, 'String');
                    val = get(id, 'Value');
                    str = options{val};
                    data_obj.(prop) = str;
                case {'checkbox', 'togglebutton'}
                    val = get(id, 'Value');
                    data_obj.(prop) = val;
                case {'radiobutton'}
                    val = get(id, 'Value');
                    if (val)
                        uData = get(id, 'UserData');
                        data_obj.(prop) = uData.Option;
                    end
            end
        end
        
        function RefreshGenericControl(id)
            userData = get(id, 'UserData');
            if ~isfield(userData, 'Context')
                context = [];
            else
                context = userData.Context;
            end
            type = get(id, 'Type');
            
            % Test for command button (and all generic menu items)
            if isfield(userData, 'Command') && ~isempty(userData.Command)
                cmd = userData.Command;
                try
                    cmd.RefreshEnabling(context);
                catch ex
                    siman.Debug.ReportError(ex, 'Error during command button refresh callback');
                    return;
                end
                enabling = cmd.Enabled;
                if ~ischar(enabling)
                    if enabling
                        enabling = 'on';
                    else
                        enabling = 'off';
                    end
                end
                set(id, 'Enable', enabling);
                
                if ~strcmp(type, 'uimenu')
                    return
                end
                
                try
                    cmd.RefreshMenuCheck(context);
                catch ex
                    siman.Debug.ReportError(ex, 'Error during menu check refresh callback');
                    return;
                end
                checked = cmd.Checked;
                if ~ischar(checked)
                    if checked
                        checked = 'on';
                    else
                        checked = 'off';
                    end
                end
                set(id, 'Checked', checked);
                return
            end
            
            % It's a regular control (represents the property of an object)
            if ~isfield(userData, 'Controller')
                siman.Debug.ReportProblem('Missing controller object for control refresh');
                return;
            end
            controller = userData.Controller;
            
            enabling = controller.GetEnabling(id, userData.Property, context);
            if ~ischar(enabling)
                if enabling
                    enabling = 'on';
                else
                    enabling = 'off';
                end
            end
            set(id, 'Enable', enabling);
            if strcmp(enabling, 'off')
                %TODO set some default disabled control values
            end
            
            
            if ~isfield(userData, 'IsLabel')
                isLabel = false;
            else
                isLabel = userData.IsLabel;
            end
            type = get(id, 'Type');
            isUIControl = strcmp(type, 'uicontrol');
            if ~isUIControl || isLabel
                return; % allow automatic enabling of things like axes which don't have values
            end
            
            if ~isfield(userData, 'DataObject')
                siman.Debug.ReportProblem('Missing data object for control refresh');
                return;
            end
            data_obj = userData.DataObject;
            if ~isfield(userData, 'Property');
                siman.Debug.ReportProblem('Missing property name for control refresh');
                return;
            end
            prop = userData.Property;
            
            siman.UIHelper.UpdateGenericControlValue(id, data_obj, prop)
        end
        
        function RefreshGenericMenuList(id)
            menus = get(id, 'Children');
            for i = 1:numel(menus)
                siman.UIHelper.RefreshControl(menus(i));
            end
        end
        
        function UpdateGenericControlValue(id, data_obj, prop)
            style = get(id, 'Style');
            switch style
                case {'edit', 'text'}
                    if isempty(data_obj)
                        str = '';
                    else
                        value = data_obj.(prop);
                        if isnumeric(value)
                            str = num2str(value);
                        else
                            str = value;
                        end
                    end
                    set(id, 'String', str);
                case {'popupmenu', 'listbox'}
                    color = [0,0,0];
                    if isempty(data_obj)
                        val = 1;
                    else
                        options = get(id, 'String');
                        str = data_obj.(prop);
                        if ~ischar(str)
                            val = [];
                        else
                            val = find(strcmp(str, options));
                        end
                    end
                    if isempty(val)
                        val = 1;
                        color = [1,0,0];
                    end
                    set(id, 'Value', val,'ForegroundColor', color);
                case {'checkbox', 'togglebutton', 'slider'}
                    if isempty(data_obj)
                        val = false;
                    else
                        val = data_obj.(prop);
                    end
                    set(id, 'Value', val);
                case {'radiobutton'}
                    if isempty(data_obj)
                        val = false;
                    else
                        uData = get(id, 'UserData');
                        val = strcmp(uData.Option, data_obj.(prop));
                    end
                    set(id, 'Value', val);
            end
        end
        
        function HandleGenericButtonPress(id)
            userData = get(id, 'UserData');
            if ~isfield(userData, 'Command');
                siman.Debug.ReportProblem('Missing command object for button press');
                return;
            end
            command = userData.Command;
            if ~isfield(userData, 'Context')
                context = [];
            else
                context = userData.Context;
            end
            command.Run(id, context); %TODO allow optional param within UserData
        end
        
        function GetBuiltinButtonImage(name)
            switch (name)
                case 'copy'
            end
        end
    end
end





