# Source_code
# P. Remigi
# 13/04/2017

# This document contains the R code to run simulations decribed in the manuscript "Spatio-temporal control of mutualism in legumes helps spread symbiotic nitrogen fixation" from Daubech B. et al 2017 

# This document is divided in 2 parts:
# - Part 1 (l.11 to l.248): the functions used to perform the simulations described in the manuscript
# - Part 2 (l.250 to l. 558): code used to fit model parameters to experimental data
##############################################################################################################################################
########################################################################
# PART 1: 
########################################################################
# This part is divided in 2 sections:
# 1. Parameters
# 2. Functions 
#    This section contains 3 sub-sections: Function 'nodulation', Function 'lambda' and Function 'cycleNif'.

########################################################################
# 1. Parameters
########################################################################
# Carrying capcity: max number of bacteria per nodule
K<-1.4e8

# Growth parameters:
r<-1.95 # 'r' is the maximum growth rate of bacteria
s<-1.65 #'s' is the reduction in growth rate due to plant sanctions
ds<-17 # 'ds' is the day at which sanctions begin

# Nodulation parameters
lmax<-0.44 
a1<-0.03 
a2<-0.006 
a3<-2 

# Maximum number of nodules per plant (upper limit, way above the number of nodules counted in experiments; this value is only useful for modeling purpose)
Nn<-100

########################################################################
# 2. Functions
########################################################################
## 2.1 Function 'nodulation'
########################################################################
#this function simulates nodulation and bacterial growth in the nodules for each pool of plants, for a given length of cycle

nodulation<-function(Pfix, pool, days){
  # Pfix= Initial fraction of Fix+ bacteria
  # pool= number of plants per pool
  # days= length of each cycle
  
  ###  2.1.1 First define the matrices that will store the data
  
  # 'pop' is the main one, keeping population size of all nodules
      # every line is an individual nodule: we create Nn*pool lines
      # columns are: plant ID, nodule ID, nodule occupancy (0: empty, 1: fix-, 2: fix+), day of nodule formation, number of bacteria per nodule, index of nitrogen fixation
      # each plant is identified with a number 'pID' ranging from 1 to the value of 'pool' 
      pid<-rep(1, Nn)
      for (k in 2:pool){pid<-c(pid, rep(k, Nn))}
      pID<-pid
  
      nID<-rep(1:Nn, pool) #nodule ID
      occ<-rep(0, Nn*pool) #who is in the nodule: 0=empty, 1=fix-, 2=fix+
      day<-rep(0, Nn*pool) #day of nodule formation
      age<-rep(0, Nn*pool) #age of nodule
      cell<-rep(0, Nn*pool) #how many cells are in this nodule
  pop<-data.frame(pID=pID, nID=nID, occ=occ, day=day, age=age, cell=cell)
  
  # 'plant' is a dataframe that will store for each plant, the total number of nif and of nodules:
      Ind<-seq(1:pool) #ID of the plant
      Nod<-rep(0, pool)  #Nod is the number of nodules on this plant
      NewNod<-rep(0, pool) #NewNod is the number of nodules formed the previous day
      Totcell<-rep(0, pool) #Totcell is the total number of bacteria hosted by the plant
      FixNod<-rep(0,pool) #FixNod is the number of nodules hosting Fix+ bacteria
  plant<-data.frame(Ind=Ind, Nod=Nod, NewNod=NewNod, Totcell=Totcell, FixNod=FixNod)
  
  storepop<-list() #'storepop' is a list that will store all 'pop' data frames from 1 cycle
  storeplant<-list() # same for 'plant'
  
  # define here plant sanctions (as they are a function of the age of each nodule)
      d<-rep(0, days)
      if (days>ds) d[ds:days]<-s
  sanction<-matrix(rep(0, 2*days), ncol=days, nrow=2)
  sanction[1,]<-d
  
  ### 2.1.2. Start the nodulation process
  # here we simulate nodulation and bacterial growth
  
  ## nodulation: new nodules are formed every day. 
  #The number of new nodules is determined according to a Poisson law. The coefficient lambda decreases as nitrogen-fixing nodules are formed: this process is called 'auto-regulation of nodulation'.
  
  for (l in 1:days){
    
    FixIndex<-rep(0, pool) #FixIndex is a vector where we will store for each plant a value that will determine the formation of new nodules
    # calculate the lamdba coefficient for each plant:
    for (j in 1:pool){FixIndex[j]<-lambda(lmax, a1, a2, a3, l, plant$FixNod[j])}
   
    # determine the number of new nodules for each plant and assess the new nodules to the plants and decide if these nodules colonized by a fix+ bacterium:
    for (j in 1:pool){
      n<-rpois(1, lambda=FixIndex[j])
      if (n!=0){
        plant$NewNod[j]<-n # store the number of new nodules
        fixation<-rep(1,n) #by default, assume that nodules are founded by Fix- bacteria
        # get random numbers; if these numbers are smaller than Pfix, then it means that a Fix+ bacteria is formed:
        test<-runif(n)
        fixation[which(test<Pfix)]<-2
        
        #now add these numbers into the 'pop' matrix
        kk<-plant$Nod[j]
        ind1<-(j-1)*Nn+kk+1
        ind2<-(j-1)*Nn+kk+n
        pop$occ[ind1:ind2]<-fixation
        pop$day[ind1:ind2]<-l
        pop$age[ind1:ind2]<-1
        pop$cell[ind1:ind2]<-1
      }
      else{plant$NewNod[j]<-0}
      #end loop on the plants to calculate new nodules
    }
    plant$Nod<-plant$Nod+plant$NewNod # update the total number of nodules on that given plant
    
    ##  multiplication of bacteria within nodules
    colNod<-which(pop$cell!=0) #find nodules that were colonized
    # create a temporary vector 'dd' that will store the sanctions for non-empty nodules according to their age and fixation status:
    dd<-rep(0, length(colNod))
    for (i in 1:length(dd)){ dd[i]<-sanction[pop$occ[colNod[i]] , pop$age[colNod[i]] ]  }
    
    # calculate bacterial growth using a logistic map with a time step of 0.5 day
    newCell<- (r-dd) * pop$cell[colNod] * (1-pop$cell[colNod]/K)
    newCell2<- (r-dd) * newCell * (1-newCell/K)
    pop$cell[colNod]<-newCell2 #update the 'pop' matrix
    pop$age[colNod]<-pop$age[colNod]+1 #update the age of the nodules
    
    #### summarize the information in 'plant'
    for (j in 1:pool){
      plantpop<-subset(pop,pID==j)
      plant$Totcell[j]<-sum(plantpop$cell)
      plant$FixNod[j]<-length(which(plantpop$occ==2))
    }
    
    storepop[[l]]<-pop #store 'pop' in 'storepop'
    storeplant[[l]]<-plant #store 'plant' in 'storeplant'
  } #end day loop
  return(list(storepop=storepop, storeplant=storeplant))
  
} # end function 'nodulation'

########################################################################
## 2.2 Function 'lambda'
########################################################################
#this function calculates the parameter of the Poisson distribution that will calculate the number of new nodules

lambda<-function(lmax, a1, a2, a3, time,n){
  # lmax, a1, a2, a3 are the 'nodulation' parameters estimated from experimental data 
  #time is the day post-infection 
  #n is the number of nodules already present on a given plant
  decay<-a2*(time-a3)
  if (decay<0){decay<-0}
  out<-lmax-a1*n-decay
  if (out<0){out<-0}
  return(out)
}

########################################################################
# 2.3.Function 'cycleNif'
########################################################################
# This is the main function that allows to perform simulations on successive inoculation-nodulation cycles
# This function calls the 2 functions defined above: 'nodulation' and 'lambda'

cycleNif<-function(Pfixo, Np, pool, cyc, days){
  
  # Pfixo: initial proportion of Fix+ bacteria
  # Np: total number of plants
  # pool: number of plants per pool
  # cyc: number of cycles
  # days: length of each cycle (in days)
  repet<-Np/pool #repet: how many replicate populations will be running? 
  
  
  ### 2.3.1 Perform the nodulation cycles
  
  ## First define some lists that will store the data.
  # at every cycle, we store the last "population" and "plant" matrices for each replicate experiment:
  cycPop<-list()
  cycPlant<-list()
  cycPfix<-list()
  cycTotPop<-list()
  # we will store the final populations/plants for each replicate:
  finalPop<-list()
  finalPlant<-list()
  finalPfix<-list()
  finalPop<-list()
  
  # define a matrix 'tempPlant' that will be used to caculate the proportion of fix+ bacteria at each time point:
      pID<-c(1:pool) #ID of the plant
      Totcell<-rep(0,pool) #total number of bacteria per plant
      Nod<-rep(0,pool) #number of nodules on each plant
      FixNod<-rep(0,pool) # number of Fix+ nodules on each plant
      FixCell<-rep(0,pool) # number of Fix+ cells on each plant
      NoFixCell<-rep(0,pool) # number of Fix- cells on each plant
      percent<-rep(0,pool) # percent of Fix+ cells in the total bacterial population
  tempPlant<-data.frame(pID=pID, Totcell=Totcell, FixNod=FixNod, FixCell=FixCell, NoFixCell=NoFixCell, percent=percent)
  
  ## start the loops; 
  # we first run all the cycles for a subgroup (pool) of plants, then repeat as many times as there are replicate plant pools
  
  for (rr in 1:repet){ # start loop on each independent pool of plants
    Pfix<-Pfixo # define 'Pfix' from the initial value first ('Pfixo'). 'Pfix' will then be updated depending on the result of each cycle
    
    for (c in 1:cyc){ # start the 'cycle' loop
      result<-nodulation(Pfix, pool, days) # Run 'nodulation' function
      
      #store the output matrices:
      cycPop[[c]]<-result[[1]][days][[1]]
      cycPlant[[c]]<-result[[2]][days][[1]]
      
      #update tempPlant:
      tempPlant$Nod<-cycPlant[[c]]$Nod
      tempPlant$FixNod<-cycPlant[[c]]$FixNod
      tempPlant$Totcell<-cycPlant[[c]]$Totcell
      
      #loop on every plant to get the numbers of fixing and non-fixing cells
      for (k in 1:pool){
        temp<-subset(cycPop[[c]], pID==k)
        tempFix<-sum(subset(temp, occ==2)$cell)
        tempNoFix<-sum(subset(temp, occ==1)$cell)
        tempPlant$FixCell[k]<-tempFix
        tempPlant$NoFixCell[k]<-tempNoFix
        tempPlant$percent[k]<-tempFix/(tempFix+tempNoFix)
      }
      
      Pfix<-sum(tempPlant$FixCell)/sum(tempPlant$Totcell) #define the new Pfix value from the output population:
      cycPfix[[c]]<-Pfix # and store this value in the list 'cycPfix'
      cycTotPop[[c]]<-sum(tempPlant$Totcell) #record the total population size in the list 'cycTotPop'
    } # end of the 'cycle' loop 
    
    ## store the results of all cycles for each pool of plants in the main lists:
    finalPop[[rr]]<-cycPop
    finalPlant[[rr]]<-cycPlant
    finalPfix[[rr]]<-cycPfix
    finalPop[[rr]]<-cycTotPop
  } #end the 'repet' loop
  
  return(list(finalPop=finalPop, finalPlant=finalPlant, finalPfix=finalPfix, finalPop=finalPop))
} #end the function cycleNif

#######################################################################
# END OF PART 1
##############################################################################################################################################

##############################################################################################################################################
# PART 2
#######################################################################
# This part of the document contains the code used to fit model parameters to experimental data for the simulations described in the manuscript 

# This part is divided in 4 sections:
# 1. Experimental dataset
# 2. RMSE Function
# 3. Fitting parameters related to nodulation kinetics
# 4. Fitting parameters related to bacterial growth within nodules

# In order for the code below to work, the functions 'lambda' and 'nodulation' need to be defined in the R environment. These functions can be found in Part 1 of this document
# Also need the experimental dataset from Figure supplement 2 found in Source Data 3. Data from each dpi were concatenated and saved as a .csv file

# load package 'dplyr'
library("dplyr", lib.loc="~/Library/R/3.1/library")


#######################################################################
# 1. EXPERIMENTAL DATASET
#######################################################################
# Here we load and re-format the dataset

# load manually the file 'data_nodulation.csv' and rename it
mydir<-getwd()
inoc<-read.csv(file.path(mydir,"/data_inoculation.csv")) #assuming that the data file was saved as 'data_inoculation.csv' in the working directory
names(inoc)[4]<-"bacteria" # rename one column

# Retrieve the number of nodule per plant ('N'), the average number of bacteria per nodule for each plant ('aver') and the total number of bacteria per nodule for each plant ('total')
grouped<-group_by(inoc, strain, DPI, Plant)
growth_data<-summarise(grouped, N=length(bacteria), aver=mean(bacteria), stddev=sd(bacteria), total=sum(bacteria))

# Split the dataset according to bacterial genotype
fix<-subset(growth_data, strain=="Fix+")
nofix<-subset(growth_data, strain=="Fix-")

#######################################################################
# 2. RMSE function
#######################################################################
# The 'rmse' function calculates the root mean square error between observed and predicted values

rmse <- function(obs, pred)
{
  error<-obs-pred
  sqrt(mean(error^2))
}

#######################################################################
# 3. NODULATION KINETICS
#######################################################################
# Here we find sets of parameters that allow the 'nodulation' function to reproduce experimental results with good accuracy
# We fit the equation describing parameter λ, which is the following: λ(t, nod+t) = lmax - a1 × nod+t - a2 × (t- a3). 
# In the equation above, (lmax, a1, a2, a3) are the parameters to fit; t is time post-inoculation (in days) and nod+t is the number of Fix+ nodules present on a given plant at time t
# We first fit 3 nodulation parameters (lmax, a2, a3) on the Fix- nodulation data, then a 4th parameter (a1: autoregulation of nodulation) on the Fix+ nodulation data


# 3.1. Nodulation with Fix- bacteria (with no autoregulation of nodulation)

# 3.1.1 Prepare the data
obsno<-c(0,aggregate(nofix$N~nofix$DPI, FUN=median)[,2]) # extract the median number of bacteria for non-fixing nodules

# Define the time points and time intervals
time<-c(1,14,21,28,35,42,49)
deltatime<-c(0,14,7,7,7,7,7)

# check the data 
plot(obsno~time)

# 3.1.2 Set the range of parameters to test

a1<-0 # we're working with Fix- so there is no AON for the moment
# explore the following ranges for parameters for lmax, a1, and a3:
lmlist<-seq(0,1,0.01)
a2list<-seq(0,0.5,0.002)
a3list<-c(1:30)

# 3.1.3 Perform simulations with nested loops
# first 'a3list', then on 'lmlist' and finally 'a2list'
# for each value of a3, we store the minimal RMSE value found, as well as the corresponding values for lmax and a2, in the following vectors:
minr<-rep(0,length(a3list))
minlm<-rep(0,length(a3list))
mina2<-rep(0,length(a3list))

for (kk in 1:length(a3list)){
  
  result<-matrix(rep(0,length(lmlist)*length(a2list)), ncol=length(lmlist), nrow=length(a2list)) # will store RMSE values for each value of lmax and a2
  xx<-seq(1,50,1) # vector of time points that will be used to calculate the predicted number of nodules
  
  for (i in 1:length(lmlist)){
    for (j in 1:length(a2list)){
      pred<-rep(0, length(xx)) # vector that will store the predicted number of nodules
      for (k in 2:length(xx)){pred[k]<-pred[k-1]+lambda(lmlist[i], a1, a2list[j], a3list[kk], xx[k], pred[k-1])} # calculates the predicted number of nodules for the current set of parameters
      result[j,i]<-rmse(obsno, pred[time]) # calculate and stores RMSE value for the current set of parameters
    }
  }
  
  minr[kk]<-min(result) # select the minimal RMSE value for the current value of a3
  best<-which(result==min(result), arr.ind = TRUE) #find the position of the minimal RMSE value within the matrix 'result'
  minlm[[kk]]<-lmlist[best[1,2]] # pick the value of lmax that corresponds to the minimal RMSE
  mina2[[kk]]<-a2list[best[1,1]] # pick the value of a2 that corresponds to the minimal RMSE
}

# 3.1.4: Best parameters found
bestr<-min(minr) # best RMSE=1.205779
ind<-which(minr==min(minr))
lmax<-minlm[ind] # lmax=0.44
a2<-mina2[ind] # a2=0.006
a3<-a3list[ind] # a3=2

# 3.1.5: Plot data vs. simulations with best fit
predno<-rep(0, length(xx))
for (k in 2:length(xx)){predno[k]<-predno[k-1]+lambda(lmax, a1, a2, a3, xx[k], pred[k-1])}
plot(obsno~time)
lines(predno~xx)

##################################################
#3.2 now do it for Fix+ with the parameters fitted from Fix-

# 3.2.1 Prepare the data
nodfix<-c(0,aggregate(fix$N~fix$DPI, FUN=median)[,2])
obs<-nodfix
time<-c(1,14,21,28,35,42,49)
deltatime<-c(0,14,7,7,7,7,7)

plot(obs~time)

# 3.2.2 Set the range of parameters
# here we use the values of a, c and d found above
lmax<-0.44
a2<-0.006
a3<-2

#now we want to explore values of b to account for AON
a1list<-seq(0,1,0.01)

# 3.2.3 Perform simulations 
result<-rep(0,length(a1list)) # will store RMSE values for each value of a1

for (i in 1:length(a1list)){
  pred<-rep(0, length(xx))
  for (k in 2:length(xx)){pred[k]<-pred[k-1]+lambda(lmax, a1list[i], a2, a3, xx[k], pred[k-1])}
  result[i]<-rmse(obs, pred[time])
}

# 3.2.4. Best parameters found
best<-which(result==min(result), arr.ind = TRUE) #the minimal RMSE=2.193655
a1<-a1list[best] #here we get a1=0.03

# 3.2.5. show the fit for the best parameters
pred<-rep(0, length(xx))
for (k in 2:length(xx)){pred[k]<-pred[k-1]+lambda(lmax, a1, a2, a3, xx[k], pred[k-1])}

plot(obs~time, ylim=c(0,15))
lines(pred~xx)
boxplot(fix$N~fix$DPI, at=c(14,21,28,35,42,49), add=TRUE)

#######################################################################
# 4. PARAMETRIZE GROWTH CURVES
#######################################################################
# In this section we estimate parameters of the logistic equation used to simulate the growth rate of bacteria within nodules
# Because the distribution of bacterial nodule population sizes depends both on the growth rate of bacteria and on the time at which nodules are formed, we compare experimental measures of nodule populations to the output of the 'nodulation' function that was run with a range of parameter values

# 4.1 Prepare the dataset
# Extract the median number of bacteria per nodule
grouped<-group_by(inoc, strain, DPI) #all plants are averaged
growth_data<-summarise(grouped, N=length(bacteria), med=median(bacteria), stddev=sd(bacteria))

fix<-subset(growth_data, strain=="Fix+")
nofix<-subset(growth_data, strain=="Fix-")

plot(fix$med~fix$DPI, log="y", ylim=c(5e5, 5e8))
lines(nofix$med~nofix$DPI, type="p", pch=19)

#define vectors for the observed data
obs<-c(4, fix$med) # 'obs' will be the observed values for Fix+ bacteria; we assume that there are 4 cells in the nodules at day 1.
obs2<-c(4, nofix$med) # 'obs2' will be the observed values for Fix- bacteria
time<-c(1,fix$DPI)

# 4.2 Set fixed parameters and the range of parameters to test regarding bacterial growth

# the nodulation parameters estimated in the section above
lmax<-0.44
a1<-0.03
a2<-0.006
a3<-2

#Population parameters
pool<-100 # the simulations will be run and averaged over 100 independent plants
days<-49
Nn<-100 # maximum number of nodules per plant

# Range of values explored for growth parameters (s, ds, r and K)
# For all 4 parameters, a wider range was explored first with less replicates
# then parameter range was refined around the best values observed

#First run: set of parameter range, with pool=40 plants
#slist<-seq(1,2,0.05) # list of 's': value of plant sanctions
#daylist<-seq(10,25,1) # list of 'ds': day at which sanctions start
#rlist<-seq(1.2,2.2,0.1) # list of 'r': maximum bacterial growth rate
#Klist<-seq(1e8,2e8,1e7) # list of 'K': nodule carrying capacity

# Second run: parameter range tested with pools of 100 plants:
slist<-seq(1.2, 1.8, 0.05)
daylist<-c(16:21)
rlist<-seq(1.8,2,0.05)
Klist<-seq(1.1e8,1.4e8,0.5e7)

# 4.3 Run simulations

# define matrices to store results
result1<-matrix(rep(0,length(rlist)*length(Klist)), ncol=length(rlist), nrow=length(Klist))
result2<-matrix(rep(0,length(rlist)*length(Klist)), ncol=length(rlist), nrow=length(Klist))
bests<-matrix(rep(0,length(rlist)*length(Klist)), ncol=length(rlist), nrow=length(Klist))
bestday<-matrix(rep(0,length(rlist)*length(Klist)), ncol=length(rlist), nrow=length(Klist))
result<-matrix(rep(0,length(slist)*length(daylist)), ncol=length(slist), nrow=length(daylist)) 

for (i in 1:length(rlist)){
  for (j in 1:length(Klist)){
    r<-rlist[i]
    K<-Klist[j]
    #run the simulations: start with Fix+ bacteria first (no sanctions, only need to fit 'r' and 'K')
    a<-nodulation(1, pool,days)
    #extract nodule data from the simulation and create a vector 'pred' storing the average number of bacteria per nodule
    popfix<-rep(0, days)
    for (k in 1:days){popfix[k]<-sum(a[[2]][k][[1]]$Totcell)/sum(a[[2]][k][[1]]$Nod)} 
    pred<-popfix[time]  
    #do the rmse
    result1[j,i]<-rmse(log(obs), log(pred))
    
    
    # then start nested loops to estimate also 's' and 'ds' for Fix- bacteria
    for (k in 1:length(slist)){
      for (l in 1:length(daylist)){
        s<-slist[k]
        ds<-daylist[l]
        a<-nodulation(0, pool,days) 
        #extract nodule data from the simulation and create a vector 'pred2'
        popnofix<-rep(0, days)
        for (kk in 1:days){popnofix[kk]<-sum(a[[2]][kk][[1]]$Totcell)/sum(a[[2]][kk][[1]]$Nod)}
        pred2<-popnofix[time]  
        #do the rmse
        result[l,k]<-rmse(log(obs2), log(pred2))
        #end the s/day loops  
        print(c(i,j, l, k))
      }
    } 
    result2[j,i]<-min(result)
    best<-which(result==min(result), arr.ind = TRUE)
    bests[j,i]<-slist[best[1,2]]
    bestday[j,i]<-daylist[best[1,1]]
    
    # end the r/K loops  
  }
}
#######################################################################
#results from 1st run
# add the RMSE values from the simulations on Fix- and Fix+ bacteria
result3<-result1+result2
min(result3) # = 0.4575663 
best3<-which(result3==min(result3), arr.ind = TRUE) #  row=3 -> K= 1.2e8; col=9 -> r=2
bests[3,9] # s=1.6
bestday[3,9] # day=17

#results from 2nd run
# add the RMSE values from the simulations on Fix- and Fix+ bacteria
result3<-result1+result2
min(result3) # = 0.4897681
best3<-which(result3==min(result3), arr.ind = TRUE) #  row=7 -> K=1.4e8 ; col=4 -> r=1.95
bests[best3] # s=1.65
bestday[best3] # day=17
################################################################################################
# 4.4 Plot the fit for the best parameters

r<-1.95
K<-1.4e8
s<-1.65
day<-17
pool<-100

inocfix<-subset(inoc, strain=="Fix+")
inocnofix<-subset(inoc, strain=="Fix-")

# start with Fix+
a<-nodulation(1, pool,days)
popfix<-rep(0, days)
for (k in 1:days){popfix[k]<-sum(a[[2]][k][[1]]$Totcell)/sum(a[[2]][k][[1]]$Nod)}
pred<-popfix[time]  

plot(pred~time, type='l', lty=2,log="y", lwd=2, ylim=c(1,1e9), xlab="Days", ylab="Bacteria per nodules (log)")
boxplot(inocfix$bacteria~inocfix$DPI, cex=0.8, boxwex=1, xlim=c(0.5,6.5), at=c(14,21,28,35,42,49), add=TRUE, xaxt='n')
title(main="Fit nodule population Fix+", sub= paste("r=", r, "K= ", K))
lines(obs~time, type='p', pch=19)
lines(pred~time, type='p', pch=19, col="red")

#now for Fix-
a<-nodulation(0, pool,days)
popfix2<-rep(0, days)
for (k in 1:days){popfix2[k]<-sum(a[[2]][k][[1]]$Totcell)/sum(a[[2]][k][[1]]$Nod)}
pred2<-popfix2[time]  

plot(pred2~time, type='l', lty=2, lwd=2, ylim=c(1,2e8),log="y", xlab="Days", ylab="Bacteria per nodules (log)", col="black")
boxplot(inocnofix$bacteria~inocnofix$DPI, cex=0.8, boxwex=1, xlim=c(0.5,6.5), at=c(14,21,28,35,42,49), add=TRUE, xaxt='n')
title(main="Fit nodule population Fix-", sub= paste("r=", r, "K= ", K, "s= ", s, "day= ", day))
lines(obs2~time, type='p', pch=19)
lines(pred2~time, type='p', pch=19, col="red")

#######################################################################
## END
##############################################################################################################################################