#!/usr/bin/python

# functions to do bootstrap analysis of a given set of data

#Developed by: Peter Freddolino, Tavazoie lab, Columbia University
# https://tavazoielab.c2b2.columbia.edu/lab/
#
#Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal with the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
#    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimers.
#    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimers in the documentation and/or other materials provided with the distribution.
#    * Neither the names of the Tavazoie lab, Columbia University, nor the names of its contributors may be used to endorse or promote products derived from this Software without specific prior written permission.
#
#THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS WITH THE SOFTWARE.

import numpy
import numpy.random
import scipy.stats as st

def get_rand_sample(data):
  # Return a random sample, with replacement, from data, the same length as data itself

  newdat = [data[numpy.random.randint(0,len(data))] for i in data]
  return numpy.array(newdat)

def bootstrap_1v(data, function, nsamp=100):
  """
  Baseline function for bootstrapping a single sample

  Give it data and a function, and it returns a dictionary containing statistics of interest
  """

  estimates = []
  retdict = {}

  for i in range(nsamp):
    estimates.append(function(get_rand_sample(data)))

  retdict["stderr"] = numpy.sqrt(numpy.var(estimates))
  retdict["ci95"] = ( st.scoreatpercentile( estimates, 2.5 ) , st.scoreatpercentile( estimates, 97.5) )
  return retdict



def bootstrap_stderr(data, function, nsamp=100):
  """
  Given a set of data and a function, estimate the standard error of the estimate of that function

  function must take only one argument (the data itself)
  """

  datdict = bootstrap_1v(data, function, nsamp)
  return datdict["stderr"]

def bootstrap_mv(data, function, nsamp=100):
  """
  Perform bootstrapping on a multivariate function

  We must be given a list containing observations for each of m variables, and a function taking m arguments
  Then we bootstrap by taking random subpopulations for each variable separately and applying function to each random sample
  """

  retdict = {}
  estimates = []

  for i in range(nsamp):
    func_args = []
    for variable in data:
      func_args.append(get_rand_sample(variable))
    myest = function(*func_args)
    estimates.append(myest)

  retdict["median"] = numpy.median(estimates)
  retdict["distribution"] = estimates
  retdict["stderr"] = numpy.sqrt(numpy.var(estimates))
  retdict["ci95"] = ( st.scoreatpercentile( estimates, 2.5 ) , st.scoreatpercentile( estimates, 97.5) )
  return retdict

