#!/usr/bin/python

# make comparitive gfp and mruby fluorescence plots
# usage:  make_comp_hists.py expfile reffile outfile starving?
# here expfile should be a data table containing flow cytometry values from the condition of interest,
#  and reffile should have data from growth under rich conditions
#  the last argument should be 'true' if the cells in expfile are starving (and thus have not yet begun growth),
#  false otherwise


#Developed by: Peter Freddolino, Tavazoie lab, Columbia University
# https://tavazoielab.c2b2.columbia.edu/lab/
#
#Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal with the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
#    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimers.
#    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimers in the documentation and/or other materials provided with the distribution.
#    * Neither the names of the Tavazoie lab, Columbia University, nor the names of its contributors may be used to endorse or promote products derived from this Software without specific prior written permission.
#
#THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS WITH THE SOFTWARE.

import numpy
import table
import pylab
import sys

expfile,reffile,outfile, isstarving = sys.argv[1:5]

if isstarving == 'true':
  starving_exp = True
else:
  starving_exp = False

exptab = table.read_table(expfile,sep=",",comment=",",header=False,colnames=['index','fsc','ssc','gfp','mruby'],guesstypes=True)
reftab = table.read_table(reffile,sep=",",comment=",",header=False,colnames=['index','fsc','ssc','gfp','mruby'],guesstypes=True)

exptab = exptab.filter('fsc>190.0')
reftab = reftab.filter('fsc>190.0')

exp_fsc = numpy.array(exptab.get_column('fsc'))
exp_ssc = numpy.array(exptab.get_column('ssc'))
exp_gfp = numpy.array(exptab.get_column('gfp'))
exp_mruby = numpy.array(exptab.get_column('mruby'))

ref_fsc = numpy.array(reftab.get_column('fsc'))
ref_ssc = numpy.array(reftab.get_column('ssc'))
ref_gfp = numpy.array(reftab.get_column('gfp'))
ref_mruby = numpy.array(reftab.get_column('mruby'))

# the corrections below were derived empirically from nonfluorescent cells
#  under similar growth conditions
normed_ref_gfp = numpy.fmax(ref_gfp - 12.7,0.0)
normed_ref_mruby = numpy.fmax(ref_mruby - 4.0,0.0)

if starving_exp:
  normed_exp_gfp = numpy.fmax(exp_gfp - (-5.6026163 + exp_fsc*0.0554416 + exp_ssc*0.0393806),0.0)
  normed_exp_mruby = numpy.fmax(exp_mruby - (-1.8894818 + exp_fsc*0.0440239 + exp_ssc*0.0561541),0.0)
else:
  normed_exp_gfp = numpy.fmax(exp_gfp - 32.30624,0.0)
  normed_exp_mruby = numpy.fmax(exp_mruby - 7.37964,0.0)

#binpowers = numpy.arange(15,step=0.3)
#binlocs = 2**binpowers
binlocs = numpy.arange(-8.1,8.2,0.1)

median_ref_gfp = numpy.median(normed_ref_gfp)
median_ref_mruby = numpy.median(normed_ref_mruby)

pylab.figure(figsize=(4,6))
ax = pylab.gca()
#ax.set_xscale('log',nonposx='clip')
#ax.set_yscale('log',nonposy='clip')
print median_ref_gfp
print median_ref_mruby
pylab.hist( numpy.log2(numpy.fmax(normed_ref_gfp/median_ref_gfp,2.0**(-8))), bins=binlocs,histtype='step',color='green',lw=3,ls='dashed',normed=True,cumulative=True,label="GFP (rich)")
pylab.hist( numpy.log2(numpy.fmax(normed_exp_gfp/median_ref_gfp,2.0**(-8))), bins=binlocs,histtype='step',color='green',lw=3,normed=True,cumulative=True,label="GFP (tuned)")
pylab.hist( numpy.log2(numpy.fmax(normed_ref_mruby/median_ref_mruby,2.0**(-8))), bins=binlocs,histtype='step',color='red',lw=3,ls='dashed',normed=True,cumulative=True,label="mRuby (rich)")
pylab.hist( numpy.log2(numpy.fmax(normed_exp_mruby/median_ref_mruby,2.0**(-8))), bins=binlocs,histtype='step',color='red',lw=3,normed=True,cumulative=True,label="mRuby (tuned)")
pylab.ylim( (0,1))
pylab.ylabel("Population CDF")
pylab.legend(loc=2)
pylab.xlabel("log$_2$ fold fluorescence change")
pylab.xlim((-8,8))
pylab.savefig(outfile)
