#!/usr/bin/python

"""Annotate synonymous barcodes on reads."""


import os
import sys
import subprocess
import argparse
import math
import numpy
import Bio.SeqIO 
import pandas
import matplotlib
matplotlib.use('pdf')
import matplotlib.pyplot as plt
import scipy.misc
import HTSeq 


class ArgumentParserNoArgHelp(argparse.ArgumentParser):
    """Like *argparse.ArgumentParser*, but prints help when no arguments."""
    def error(self, message):
        """Prints error message, then help."""
        sys.stderr.write('error: %s\n\n' % message)
        self.print_help()
        sys.exit(2)


def parseArgs():
    """Parse arguments for script."""
    parser = ArgumentParserNoArgHelp(description="Annotate synonymous barcodes "
            "on influenza reads and make some plots.")
    parser.add_argument('--bamfiles', help='BAM file generated by cellranger',
            required=True, nargs='+')
    parser.add_argument('--flugenes', help='FASTA file with flu genes',
            required=True)
    parser.add_argument('--synflugenes', help='FASTA file with synonymously '
            'barcoded flu genes', required=True)
    parser.add_argument('--cellbarcodes', help='Cell barcodes file created '
            'by cellranger', required=True)
    parser.add_argument('--outprefix', help='Prefix for output files.',
            required=True)
    parser.add_argument('--samplename', help='Name of sample, used as title for some plots', required=True)
    return parser


def main():
    """Main body of script."""
    parser = parseArgs()
    args = vars(parser.parse_args())
    bamfiles = args['bamfiles']
    genefile = args['flugenes']
    syngenefile = args['synflugenes']
    cellbarcodesfile = args['cellbarcodes']
    outprefix = args['outprefix']

    print("Annotating synonymously barcoded reads.\n")

    # read genes and synonymous variants
    genes = dict([(seq.name, str(seq.seq)) for seq in Bio.SeqIO.parse(
            genefile, 'fasta')])
    genenames = [seq.name for seq in Bio.SeqIO.parse(genefile, 'fasta')]
    print("Read {0} genes from {1}".format(len(genes), genefile))
    syngenes = dict([(seq.name, str(seq.seq)) for seq in Bio.SeqIO.parse(
            syngenefile, 'fasta')])
    assert set(syngenes.keys()) == set([gene + '-syn' for gene in genes]), (
            "gene names in {0} not the same as in {1} plus '-syn' suffix".format(
            genefile, syngenefile))
    print("Read matching synonymously barcoded genes from {0}".format(syngenefile))
    syngenes = dict([(key.replace('-syn', ''), value) for (key, value) in 
            syngenes.items()])

    # get valid cell barcodes
    with open(cellbarcodesfile) as f:
        cellbarcodes = [line.strip() for line in f if not line.isspace()]
    assert len(cellbarcodes) == len(set(cellbarcodes)), "Non-unique cell barcodes"
    cellbarcodes = set(cellbarcodes)
    print("\nWill examine only reads with one of the {0} cell barcodes in {1}".format(
            len(cellbarcodes), cellbarcodesfile))

    readstats = ['no cell identifier', 'unrecognized cell', 
            'no UMI', 'wt barcode', 'syn barcode', 'uncalled barcode',
            'invalid barcode']
    readstats_d = {}

    # keyed by cell, then gene, then UMI, then list of one of umi_cats
    umi_cats = ['syn', 'wt', 'invalid', 'uncalled']
    cell_d = dict([(cb, dict([(gene, {}) for gene in genenames])) for
            cb in cellbarcodes])

    # process reads in BAM files
    for bamfile in bamfiles:

        if (not bamfile) or bamfile.isspace():
            continue

        print("\nProcessing reads in {0}".format(bamfile))

        for gene in genenames:
        
            print("\nAnalyzing reads that align to {0}.".format(gene))

            samfile = '_{0}.sam'.format(gene)
            with open(samfile, 'w') as f:
                subprocess.check_call(['samtools', 'view', bamfile, 
                        gene], stdout=f)

            shortgene = gene.replace('flu', '')
            seq = genes[gene]
            synseq = syngenes[gene]
            assert len(seq) == len(synseq)
            synbarcode = dict([(r, (wt, mut)) for (r, (wt, mut)) in
                    enumerate(zip(seq, synseq)) if wt != mut])
            print("This gene has {0} synonymous barcoded sites: {1}".format(
                    len(synbarcode), ', '.join(["{0}{1}{2}".format(wt, r + 1, m)
                    for (r, (wt, m)) in sorted(synbarcode.items())])))

            n = dict([(x, 0) for x in readstats])
            for a in HTSeq.SAM_Reader(samfile):

                # see if we keep the alignment
                assert a.iv.chrom == gene
                try:
                    cellbc = a.optional_field('CB')
                except:
                    n['no cell identifier'] += 1
                    continue
                if cellbc not in cellbarcodes:
                    n['unrecognized cell'] += 1
                    continue
                try:
                    umi = a.optional_field('UB')
                    if umi not in cell_d[cellbc][gene]:
                        cell_d[cellbc][gene][umi] = []
                except:
                    n['no UMI'] += 1
                    continue

                # look for synonymous barcode
                synbarcoded = []
                read = a.read if a.iv.strand == '+' else a.read_as_aligned
                i = 0 # read index
                r = a.iv.start # reference index: 0, 1, ...
                for operation in a.cigar:
                    if operation.type == 'M': # aligned
                        for j in range(operation.size):
                            nt = read.seq[i + j]
                            if r in synbarcode:
                                assert seq[r] == synbarcode[r][0]
                                if nt == synbarcode[r][0]:
                                    synbarcoded.append(False)
                                elif nt == synbarcode[r][1]:
                                    synbarcoded.append(True)
                                else:
                                    synbarcoded.append(None)
                            r += 1
                        i += operation.size
                    elif operation.type == 'I': # insertion
                        i += operation.size
                    elif operation.type in ['N', 'D']: # deletion
                        for j in range(operation.size):
                            r += 1   
                    elif operation.type == 'S': # soft clipping of read
                        i += operation.size
                    else: # we don't expect other cigar operations
                        raise RuntimeError("Unexpected cigar operation {0}".format(
                                operation.type))
                if len(synbarcoded) == len(synbarcode):
                    if all([x == True for x in synbarcoded]):
                        n['syn barcode'] += 1
                        cell_d[cellbc][gene][umi].append('syn')
                    elif all([x == False for x in synbarcoded]):
                        n['wt barcode'] += 1
                        cell_d[cellbc][gene][umi].append('wt')
                    else:
                        n['invalid barcode'] += 1
                        cell_d[cellbc][gene][umi].append('invalid')
                else:
                    n['uncalled barcode'] += 1
                    cell_d[cellbc][gene][umi].append('uncalled')
            
            if gene in readstats:
                for (stat, n) in readstats.items():
                    readstats_d[gene] += n
            else:
                readstats_d[gene] = [n[stat] for stat in readstats]

            os.remove(samfile)

    # write and plot read statistics
    readstats_file = outprefix + 'readstats.csv'
    readstats_plot = outprefix + 'readstats.pdf'
    print("\nWriting and plotting read statistics to {0} and {1}".format(
            readstats_file, readstats_plot))
    readstats_df = pandas.DataFrame.from_dict(readstats_d, orient='index')
    readstats_df.columns = readstats
    readstats_df = readstats_df.reindex(genenames)
    readstats_df.to_csv(readstats_file)
    readstats_df.plot.bar(stacked=True, figsize=(7, 3))
    plt.ylabel('number of reads')
    plt.legend(loc='upper left', bbox_to_anchor=(1, 1))
    plt.savefig(readstats_plot, bbox_inches='tight')

    # get umi statistics about gene calls into a dataframe
    cellcounts = dict([("{0}-{1}".format(gene, cat), []) for gene in genenames
            for cat in umi_cats])
    cellcounts['cellbarcode'] = []
    for cb in cellbarcodes:
        cellcounts['cellbarcode'].append(cb)
        for gene in genenames:
            cellgene_d = dict([(cat, 0) for cat in umi_cats])
            for (umi, readlist) in cell_d[cb][gene].items():
                assert readlist
                if all(['uncalled' == x for x in readlist]):
                    cellgene_d['uncalled'] += 1
                else:
                    readlist = [x for x in readlist if x != 'uncalled']
                    if all(['syn' == x for x in readlist]):
                        cellgene_d['syn'] += 1
                    elif all(['wt' == x for x in readlist]):
                        cellgene_d['wt'] += 1
                    else:
                        cellgene_d['invalid'] += 1
            for cat in umi_cats:
                cellcounts["{0}-{1}".format(gene, cat)].append(cellgene_d[cat])
    cellcounts_df = pandas.DataFrame.from_dict(cellcounts)
    cellcounts_df.set_index('cellbarcode', inplace=True, drop=True, 
            verify_integrity=True)
    for gene in genenames:
        cellcounts_df[gene] = cellcounts_df[['{0}-{1}'.format(gene, cat)
                for cat in umi_cats]].sum(axis=1)
    for cat in umi_cats:
        cellcounts_df['flu-{0}'.format(cat)] = cellcounts_df[
                ['{0}-{1}'.format(gene, cat)
                for gene in genenames]].sum(axis=1)
    cellcounts_df['total-flu'] = cellcounts_df[genenames].sum(axis=1)

    # write and plot umi statistics
    umistatsplot = outprefix + 'umistats.pdf'
    umistatsfile = outprefix + 'umistats.csv'
    barcodefracplot = outprefix + 'umibarcodefrac.pdf'
    print("\nWriting and plotting distribution of UMI barcodes across cells to "
          "{0}, {1}, {2}".format(umistatsfile, umistatsplot, barcodefracplot))
    umistats = {}
    for gene in genenames:
        umistats[gene] = dict([(cat, cellcounts_df.sum(axis=0)['{0}-{1}'.format(
                gene, cat)]) for cat in umi_cats])
    umistats = pandas.DataFrame.from_dict(umistats, orient='index')
    umistats = umistats.reindex(genenames)

    umistats[umi_cats].plot.bar(stacked=True, figsize=(5, 2.5))
    plt.ylabel('number of UMIs')
    plt.legend(loc='upper left', bbox_to_anchor=(1, 1))
    plt.title(args['samplename'])
    (ymin, ymax) = plt.gca().get_ylim()
    plt.gca().set_ylim(bottom=0, top=max(ymax, 10))
    plt.savefig(umistatsplot, bbox_inches='tight')

    umistats_all = umistats.sum(axis=0)
    umistats_all.name = 'all'
    umistats = umistats.append(umistats_all)
    umistats['fracsyn'] = umistats['syn'] / umistats[['syn', 'wt']].sum(axis=1)
    umistats.to_csv(umistatsfile)
    ax = umistats[['fracsyn']].plot(figsize=(7, 3), kind='bar')
    for p in ax.patches:
        ax.annotate('{0:.2f}'.format(p.get_height()), 
                (p.get_x(), p.get_height() * 1.05))
    plt.ylabel('fraction barcoded UMIs that are syn')
    plt.legend().set_visible(False)
    plt.ylim(0, 1)
    plt.savefig(barcodefracplot, bbox_inches='tight')

    # cumulative distribution plot
    nfluplot = outprefix + 'nflu.pdf'
    print("\nPlotting distribution of flu reads across cells to "
          "{0}".format(nfluplot))
    nwithflu = cellcounts_df['total-flu'].astype(bool).sum()
    ser = cellcounts_df['total-flu'].sort_values()
    plt.figure(figsize=(4, 3))
    if (0 < ser).any():
        cdf = pandas.Series(numpy.linspace(0., 1., len(ser)), index=ser)
    else:
        cdf = pandas.Series([0, 1, 1], index=[0, 1, 10])
    cdf.plot(drawstyle='steps', logx=True)
    plt.xlabel('number of flu UMIs per cell')
    plt.ylabel('frac cells <= this many')
    plt.xlim(0.8, 1.5 * max(ser))
    if (0 >= ser).all():
        plt.gca().text(0.5, 0.5, 'no cells with flu reads', 
                horizontalalignment='center', 
                verticalalignment='center',
                transform=plt.gca().transAxes)
    plt.title(args['samplename'])
    plt.tight_layout()
    plt.savefig(nfluplot, bbox_inches='tight')

    # plot purity
    purityplot = outprefix + 'purity.pdf'
    print("Plotting purity of virus barcodes to {0}".format(purityplot))
    avgfracsyn = umistats['fracsyn']['all']
    nflu = []
    purity = []
    randompurity = []
    fracsyn = []
    for (cellbc, row) in cellcounts_df.sort_values('total-flu').iterrows():
        nbc = row['flu-syn'] + row['flu-wt']
        if nbc < 2:
            continue
        nflu.append(row['total-flu'])
        fracsyn.append(row['flu-syn'] / float(nbc))
        purity.append(max(row['flu-syn'], row['flu-wt']) / float(nbc))
        if nbc > 500:
            # numerical overflow for large nbc, assign value about this
            randompurity.append(max(avgfracsyn, 1 - avgfracsyn))
        else:
            randompurity.append(sum([scipy.misc.comb(nbc, k) * avgfracsyn**k * 
                    (1 - avgfracsyn)**(nbc - k) * max(k, nbc - k) / float(nbc) for k in 
                    range(0, nbc + 1)]))
    (fig, axes) = plt.subplots(nrows=1, ncols=2, figsize=(7, 3))
    if nflu:
        ncells = len(nflu)
        lognflu = [math.log10(n) for n in nflu]
        synbar = [n * f for (n, f) in zip(lognflu, fracsyn)]
        wtbar = [n * (1 - f) for (n, f) in zip(lognflu, fracsyn)]
        psyn = axes[0].bar(range(ncells), synbar, color='blue', linewidth=0, width=1)
        pwt = axes[0].bar(range(ncells), wtbar, bottom=synbar, color='red', linewidth=0, width=1)
        axes[0].legend((psyn[0], pwt[0]), ('syn', 'wt'), loc='upper left', fontsize=12)
        puritybar = [n * max(f, 1 - f) for (n, f) in zip(lognflu, fracsyn)]
        impuritybar = [n * min(f, 1 - f) for (n, f) in zip(lognflu, fracsyn)] 
        pimpure = axes[1].bar(range(ncells), impuritybar, color='orange', linewidth=0, width=1)
        ppure = axes[1].bar(range(ncells), puritybar, color='green', linewidth=0, width=1,
                bottom=impuritybar)
        axes[1].legend((pimpure[0], ppure[0]), ('minor barcode', 'major barcode'), 
                loc='upper left', fontsize=12)
        for ax in [axes[0], axes[1]]:
            ax.set_ylabel('log10 flu UMIs / cell')
            ax.set_xlabel('cells with >=2 barcoded flu UMIs')
            ax.set_xlim(-0.5, ncells + 0.5)
            ax.set_xticks([])
    else:
        for ax in [axes[0], axes[1]]:
            ax.text(0.5, 0.5, 'no cells with >=2 barcoded flu reads', 
                    horizontalalignment='center', verticalalignment='center')
            ax.set_xticks([])
            ax.set_yticks([])
    plt.suptitle(args['samplename'])
    plt.tight_layout()
    fig.subplots_adjust(top=0.88)
    plt.savefig(purityplot, bbox_inches='tight')

    # write out file giving matrix of values for each cell
    cellannotations = outprefix + 'cellannotations.tsv'
    print("\nWriting cell annotations to {0}".format(cellannotations))
    cellcounts_df[['{0}-{1}'.format(gene, cat) for gene in genenames + ['flu']
            for cat in ['syn', 'wt']]].to_csv(cellannotations, sep='\t')

    print("\nProgram complete.")


if __name__ == '__main__':
    main()
