% Statistical analysis on decoding accuracies of task and object, and plotting of results

addpath('helper_functions')

load data/decoding_task.mat
load data/decoding_obj.mat

disp('Running stats...')

% get the first and last sample of each time period in which we want to run statistical tests
% (these are the time periods where we expect to find anything if any)
t0_task = ceil(101*0.12); % *0.12, because we have 120 samples per second (after downsampling)
tn_task = ceil(3600*0.12);
t0_obj = ceil(2101*0.12);
tn_obj = ceil(3600*0.12);

opt.method = 'signperm';
opt.multcomp = 'cluster';
opt.n_perm = 'all';
opt.memlim = 1024; % use maximally a quarter of your ram
p_task = meg_stats(DA_task(:,t0_task:tn_task)',0.5,opt);
p_obj = meg_stats(DA_obj(:,t0_obj:tn_obj)',0.5,opt);

t = linspace(-100,5000,612);

n_sub = 17;

%% Create figure and set size

f = figure;
ax = gca;

f.Position = [800 1200 1280 768];
ax.Position = [0.05 0.1 0.9 0.85];


%% Plot results

% plot task with SEM
[h1,p1] = boundedline(t,100*mean(DA_task,1),std(100*DA_task,[],1)/sqrt(n_sub),'','alpha');

hold on
% plot obj with SEM
[h2,p2] = boundedline(t,100*mean(DA_obj,1),std(100*DA_obj,[],1)/sqrt(n_sub),'r','alpha');
 
ylim([40 100]) % accuracy range
xlim([-100 5000]) % time range

% add white patch (not in paper)
hp = patch([3500 3500 5000 5000],[41 99 99 41],[1 1 1]);
hp.FaceAlpha = 0.75;
hp.LineStyle = 'none';

% adjust plot properties
h1.Color = [52 96 148]/255;
h1.LineWidth = 2;
h2.LineWidth = 2;

% now add p = nan back in (for the values we are not testing)
h_task = plot_significance(p_task<0.05,size(DA_task,2),t0_task,tn_task);
h_obj = plot_significance(p_obj<0.05,size(DA_obj,2),t0_obj,tn_obj);

% plot those significance values
sig1 = plot(t,99.5*h_task,'b','linewidth',2);
sig2 = plot(t,99*h_obj,'r','linewidth',2);



%% Add lines

% chance
l1 = plot([-100 5000],[50 50]);
l1.Color = [0 0 0];

% task cue onset
l2 = plot([0 0],ylim,'--');
l2.Color = [0.3 0.3 0.3];
% stimulus onset
l3 = plot([2000 2000],ylim,'--');
l3.Color = [0.3 0.3 0.3];
% response screen onset
l4 = plot([3500 3500],ylim,'--');
l4.Color = [0.3 0.3 0.3];

%% Add xlabel and ylabel

lab1 = ylabel('Decoding accuracy (%)');
lab2 = xlabel('Time (ms)');
lab1.Position(1) = lab1.Position(1) * 1.10;
lab2.Position(2) = lab2.Position(2) * 0.95;
set([lab1 lab2],'FontWeight','bold');

%% Add legend for h1 and h2

leg = legend([h1 h2],{'task','object'});

%% Add text descriptions

t1 = text(4500,49,'chance-level');
t2 = text(50,41,'task cue');
t3 = text(2050,41,'object');
t4 = text(3550,41,'response');
% set([t1 t2 t3 t4],'FontAngle','italic')
set([t2 t3 t4],'Rotation',90)

% Change Font Size to 12 and Font Name to Arial
set(findall(f,'-property','FontSize'),'FontSize',16)
set(findall(f,'-property','FontName'),'FontName','Arial');

a = axis; 
plot(a(1)*[1,1],a(3:4),'k','linewidth',0.5) 
plot(a(2)*[1,1],a(3:4),'k','linewidth',0.5) 
plot(a(1:2),a(4)*[1,1],'k','linewidth',0.5) 

