% statistical analysis of temporal generalization of task and plotting of results

addpath('helper_functions')

load data/generalization_task.mat
results = double(results);

if exist('data/generalization_task_perm.mat','file')

    disp('loading pre-calculated permutations...')
    load data/generalization_task_perm.mat
    
else
    
    disp('running permutations (should take maximum 10 minutes)')
    opt.method = 'signperm';
    opt.multcomp = 'cluster';
    opt.n_perm = 5000; % run fewer if you get out of memory
    opt.memlim = 4096; % you may or may not have so much memory (we usually use about 2-3x as much in reality)
    baseline = 0.5;
    
    % reduce to relevant time periods
    t0_task = ceil(101*0.12);
    tn_task = ceil(3600*0.12);
    
    [p, mat] = meg_stats2(results(t0_task:tn_task,t0_task:tn_task,:),baseline,opt);
     
    save data/generalization_task_perm.mat p mat -v7.3
    
end

% add left out values (these values are no longer used for statistical purposes, just for plotting)
p = [ones(12,420); p];
p = [ones(432,12) p];
p(433:612,:) = 1; p(:,433:612) = 1;


%% Plotting

% we actually only use t(2,:) later, but for comparability we leave these values in
t = 1:612; % samples
t(2,:) = floor(linspace(-100,5000,612));
t(2,:) = 10*floor(t(2,:)/10); % time points relative to 0
t(3,:) = round(linspace(0,5100,612)); % time points absolute

cl{1} = {'k--','k','k--','k'}; % line styles
% cl{3} = {'w--','w','w--','w'}; % line styles

% make figure
fh = figure;
fh.Position = [1000 918 1120 840];

% plot generalization results
da = 100*mean(results,3);
ah1 = imagesc(da,[20 80]);

% adjust axis
h = gca;
axis square
h.Position(2) = 1.1*h.Position(2);
h.YDir = 'normal';
h.XTick = floor([13:60:612 612]);
h.XTickLabel = floor(t(2,[13:60:612 612]));
h.YTick = floor([13:60:612 612]);
h.YTickLabel = floor(t(2,[13:60:612 612]));

% plot slight adaptation of redyellowblue colormap that makes details more visible
out = redyellowblue(200);
out(51:end,2) = out(51:end,2).*linspace(1,0.2,50)';
out(51:end,3) = out(51:end,3).*linspace(1,0.2,50)';
colormap(out)
colorbar

    
% plot lines
hold on
plot(repmat(13,1,612),1:612,'k')
plot(1:612,repmat(13,1,612),'k')
t1 = text(13,-25,'task cue on');
t2 = text(253,-25,'object on');
t3 = text(433,-25,'response on');
set([t1 t2 t3],'Rotation',45,'HorizontalAlignment','right')
plot(repmat(253,1,612),1:612,'Color',[0.5 0.5 0.5],'LineWidth',1.5)
plot(1:612,repmat(253,1,612),'Color',[0.5 0.5 0.5],'LineWidth',1.5)
plot(repmat(433,1,612),1:612,'Color',[0.5 0.5 0.5],'LineWidth',1.5)
plot(1:612,repmat(433,1,612),'Color',[0.5 0.5 0.5],'LineWidth',1.5)

% plot title and axis labels
title('Task')
xlabel('Generalization Time')
ylabel('Training Time')

% Change Font Size to 12 and Font Name to Arial
set(findall(fh,'-property','FontSize'),'FontSize',16)
set(findall(fh,'-property','FontName'),'FontName','Arial');

% get significance outline from p values
a = diff([ones(1,612); p; ones(1,612)]<0.05);
b = diff([ones(612,1) p ones(612,1)]<0.05,[],2);
c = double(b(:,1:end-1)|a(1:end-1,:));
a_ = diff([p; ones(1,612)]<0.05);
b_ = diff([p ones(612,1)]<0.05,[],2);
c_ = double(b_|a_);
c = c|c_;

% plot outline on top
h2 = axes;
fhdl = imagesc(1-c);
h2.YDir = 'normal';
axis square
h2.Position = h.Position;
set(fhdl, 'AlphaData', c==1);
axis off
colormap(h2,'gray')        
        
% if you want to save the plot
% print('results_generalization.eps','-painters','-depsc')