% Statistical analysis on decoding accuracies of object given task type, and plotting of results

addpath('helper_functions')

load data/decoding_obj_bytask.mat

disp('Running stats...')

% get the first and last sample of each time period in which we want to run statistical tests
% (these are the time periods where we expect to find anything if any)
t0_obj = ceil(2101*0.12);
tn_obj = ceil(3600*0.12);

opt.method = 'signperm';
opt.multcomp = 'cluster';
opt.n_perm = 'all';
opt.memlim = 1024; % use maximally a quarter of your ram
p = meg_stats(DA_objbytask(t0_obj:tn_obj,:,1) - DA_objbytask(t0_obj:tn_obj,:,2),0,opt);

n_sub = 17;

timerange_ms = [1900 3500];
% timerange_ms = [-100 5000];
timerange = floor(timerange_ms*0.12)+12; timerange(1) = timerange(1)+1;
timevals = timerange(1):timerange(2);
timestr = timerange_ms - timerange_ms(1)-100;
t = linspace(timestr(1),timestr(2),length(timevals));

%% Create figure and set size

f = figure;
ax = gca;

f.Position = [800 800 600 600];
ax.Position = [0.125 0.1 0.8 0.85];

%% Plot results

% get standard error of the difference of the mean
sedm = 100*std(DA_objbytask(timevals,:,1)-DA_objbytask(timevals,:,2),[],2)/sqrt(n_sub);

% % plot obj given task type with SEM
[h1,p1] =  boundedline(t,100*mean(DA_objbytask(timevals,:,1),2),sedm,'alpha');
hold on
[h2,p2] =  boundedline(t,100*mean(DA_objbytask(timevals,:,2),2),sedm,'--','alpha');

ylim([40 90]) % accuracy range
xlim(timestr); % time range

% Adjust plot properties
h1.LineWidth = 1.5;
h2.LineWidth = 1.5;

h1.Color = [200 0 0]/255;
p1.FaceColor = [200 0 0]/255;
h2.Color = [255 80 80]/255;
p2.FaceColor = [255 144 144]/255;

% now add p = nan back in (for the values we are not testing)
h_obj = plot_significance(p<0.05,size(DA_objbytask,1),t0_obj,tn_obj);
h_obj = h_obj(timevals);

% plot those significance values
sig2 = plot(t,89*h_obj,'b','linewidth',2);
sig2.Color = h1.Color;

ax.XTick = 0:250:1500;

%% Add lines

% chance
l1 = plot([-100 5000],[50 50]);
l1.Color = [0 0 0];

% stimulus onset
l2 = plot([0 0],ylim,'--');
l2.Color = [0.3 0.3 0.3];

%% Add xlabel, ylabel and title

title('Object decoding by task type')
lab1 = ylabel('Decoding accuracy (%)');
lab2 = xlabel('Time (ms)');
lab1.Position(1) = lab1.Position(1) * 1.2;
% lab2.Position(2) = lab2.Position(2) * 0.95;
set([lab1 lab2],'FontWeight','bold');

%% Add legend for h1 and h2

leg = legend([h1 h2],{'conceptual','perceptual'});

%% Add text descriptions

t1 = text(1150,47,'chance-level');
t2 = text(50,81,'object on');
set(t2,'Rotation',90)

set(findall(f,'-property','FontSize'),'FontSize',16)
set(findall(f,'-property','FontName'),'FontName','Arial');


a = axis; 
plot(a(1)*[1,1],a(3:4),'k','linewidth',0.5) 
plot(a(2)*[1,1],a(3:4),'k','linewidth',0.5) 
plot(a(1:2),a(4)*[1,1],'k','linewidth',0.5) 

% if you want to save the figure
% print('results_object_bytask.eps','-painters','-depsc')
