function [I_ion,h,m,n,p] = compute_HH_ionic_currents(v,dt,g_Na,g_K,g_M,g_L,phi,E_Na,E_K,E_L,V_T,h,m,n,p)

% This function computes the total ionic current (I_ion) and updates values of
% activation/inactivation variables (h,m,n,p) to all neurons, based on:
%
% - membrane potential vector (v) in current time step 
% - time step (dt)
% - ionic conductance amplitudes of sodium (g_Na), potassium (g_K), slow potassium (g_M), and leak current (g_L)
% - ionic conductance time constants (phi)
% - reversal potentials (E_Na, E_K,E_L)
% - spk threshold, V_T
% - activation/inactivation variables (h,m,n,p) from previous time step

% The dynamics for activation/inactivation variable, x = {n,m,h}, has the following form:
% tau_x*dx/dt + x = x_inf, where tau_x = phi/(alpha_x + beta_x) and x_inf = alpha_x/(alpha_x + beta_x)

%compute rates for activation/inactivation variables
    alpha_n = -0.032*(v-V_T-15)./(exp(-0.2*(v-V_T-15))-1);
    beta_n = 0.5*exp(-(v-V_T-10)/40);

    alpha_m = -0.32*(v-V_T-13)./(exp(-0.25*(v-V_T-13))-1);
    beta_m = 0.28*(v-V_T-40)./(exp(0.2*(v-V_T-40))-1);

    alpha_h = 0.128*exp(-(v-V_T-17)/18);
    beta_h = 4./(exp(-0.2*(v-V_T-40))+1);

    n_inf = alpha_n./(alpha_n + beta_n);
    m_inf = alpha_m./(alpha_m + beta_m);
    h_inf = alpha_h./(alpha_h + beta_h);
    p_inf = 1./(exp(-0.1*(v+35))+1);

    tau_n = phi./(alpha_n + beta_n);
    tau_m = phi./(alpha_m + beta_m);
    tau_h = phi./(alpha_h + beta_h);
    tau_p = phi*4000./(3.3*exp(0.05*(v+35)) + exp(-0.05*(v+35))); 
    
%compute ionic currents
I_Na = g_Na*m_inf.^3.*h.*(v-E_Na);
I_K = g_K*n.^4.*(v-E_K);
I_M = g_M*p.*(v-E_K);
I_L = g_L*(v-E_L);

I_ion = I_Na + I_K + I_M + I_L;

%update activation/inactivation
n = n + (dt./tau_n).*(n_inf - n);
m = m + (dt./tau_m).*(m_inf - m);
h = h + (dt./tau_h).*(h_inf - h);
p = p + (dt./tau_p).*(p_inf - p);