function [G_I_EL,G_I_ER,G_EL_I,G_ER_I,G_I_I] = gridcell_development_1D()

%%%%%%%%%%%
%%%SETUP%%%
%%%%%%%%%%%

    T_long = 14400;     %total sim time (s)
    T = 10;             %sim time blocks (s)
    dt = 1/2000;       %step size of numerical integration (s)
    tau_s = 30/1000;   %neural time constant (s)    

    N_E = 200;          %num. of excitatory cells per pop
    N_I = 80;           %num. of inhibitory cells
    N_tot = N_I+2*N_E;
    n_I = 1:N_I;                %I cell indices
    n_EL = N_I+1:N_I+N_E;       %EL cell indices
    n_ER = N_I+N_E+1:N_I+2*N_E; %ER cell indices

    %External inputs
        beta_vel = 0.9;     %velocity gain
        sigma_LS = 0.01;    %width of gaussian visual input (m)
        beta_LS_E = 10;     %magnitude of gaussian visual input to E pop
        beta_LS_I = 50;     %magnitude of gaussian visual input to I pop
        
        x_prefs_I = (1:N_I)'/N_I; %location preferences of I pop
        x_prefs_E = (1:N_E)'/N_E; %location preferences of I pop
        
    %Suppressive envelope
        periodic = 0; %set to 0 for aperiodic net
        A_I = create_envelope(periodic,N_I)'; 
        A_E = create_envelope(periodic,N_E)';

    %STDP parameters
        l=0.1;          %length of STDP kernel (s)
        tau_STDP = 0.006;   %STDP time constant

        %STDP kernel for excitatory synapses
            K1_pos = 1.2*exp(-(l-dt:-dt:0)/(4*tau_STDP));
            K1_neg = -1*exp(-(l-dt:-dt:0)/(3*tau_STDP));

        %STDP kernel for inhibitory synapses
            K2_pos = 0.5*exp(-(l-dt:-dt:0)/(4*tau_STDP));
            K2_neg = -exp(-(l-dt:-dt:0)/(2*tau_STDP));
            
        %learning rates
            eta = 0.015;        %base learning rate factor 
            eta_II = 7*eta;     %I->I learning rate
            eta_IE = eta;       %E->I learning rate
            eta_EI = 2*eta;     %I->E learning rate

    %Initial weights
        g0 = 1e-3;        %scales initial size of weights
        G_I_EL = g0*(2*rand(N_I,N_E)-1);
        G_I_ER = g0*(2*rand(N_I,N_E)-1);
        G_EL_I = g0*(2*rand(N_E,N_I)-1);
        G_ER_I = g0*(2*rand(N_E,N_I)-1);
        G_I_I = g0*(2*rand(N_I,N_I)-1);
        
    %Sinusoidal animal trajectory for training
        x = (sin((dt:dt:T)*2*pi/(T/2)-pi)+1)/2;
        v = diff(x)/dt; v = [v(1),v];
        
%%%%%%%%%%%%%%
%%%DYNAMICS%%%
%%%%%%%%%%%%%%
        
    for tt = 1:T_long/T %OUTER time loop (T_long sec)

        %initialize variables
            if tt==1
                spk_prev = zeros(N_tot,l/dt);
                s_prev = zeros(N_tot,1);
            end
            spk = zeros(N_tot,T/dt);

        for t = 2:T/dt %INNER time loop (T sec)

            %%%%%%%%%%%%%%%%%%%%%
            %%%Neural dynamics%%%
            %%%%%%%%%%%%%%%%%%%%%

                %I pop
                g_F_I = beta_LS_I*exp(-(x_prefs_I - x(t)).^2/(2*sigma_LS)^2);   %loc-spec inputs
                G_I = A_I.*g_F_I;                                               %total input conductance (no recurrents)

                %EL pop
                g_F_E = beta_LS_E*exp(-(x_prefs_E - x(t)).^2/(2*sigma_LS)^2);   %loc-spec inputs
                v_L = (1 - beta_vel*v(t)); v_L = v_L*(v_L>0);                   %velocity input
                G_EL = v_L*A_E.*g_F_E;                                          %total input conductance (no recurrents)

                %ER pop
                g_F_E = beta_LS_E*exp(-(x_prefs_E - x(t)).^2/(2*sigma_LS)^2);   %loc-spec inputs
                v_R = (1 + beta_vel*v(t)); v_R = v_R*(v_R>0);                   %velocity input
                G_ER = v_R*A_E.*g_F_E;                                          %total input conductance (no recurrents)           

                %pass conductance variables through nonlinearity to generate output rates, F
                    G = [G_I;G_EL;G_ER];                             
                    F = zeros(N_tot,1) + G.*(G>=0);   %linear transfer function

                %subdivide interval m times and generate spikes (spk variable)
                    spk(:,t) = poissrnd(F*dt);
                    
                %update neural activation variables
                    s_new = s_prev + spk(:,t) - s_prev*dt./tau_s; 
                    s_prev = s_new;

            %%%%%%%%%%%%%%%%%%%%%%%
            %%%Synaptic dynamics%%%
            %%%%%%%%%%%%%%%%%%%%%%%         

                %Convolution of spks with STPD kernals
                    if(t>l/dt)     %here, l/dt is the size of the kernel
                        spk_conv_1_pos = spk(:,t-l/dt+1:t)*K1_pos';
                        spk_conv_1_neg = spk(:,t-l/dt+1:t)*K1_neg';
                        spk_conv_2_pos = spk(:,t-l/dt+1:t)*K2_pos';
                        spk_conv_2_neg = spk(:,t-l/dt+1:t)*K2_neg';
                    else                %if t<l/dt, must go back to previous spk matrix
                        spk_temp = [spk_prev(:,t+1:end),spk(:,1:t)];
                        spk_conv_1_pos = spk_temp*K1_pos';
                        spk_conv_1_neg = spk_temp*K1_neg';
                        spk_conv_2_pos = spk_temp*K2_pos';
                        spk_conv_2_neg = spk_temp*K2_neg';
                    end
                    
                %compute weight updates
                    delG_I_EL = spk(n_I,t)*spk_conv_1_pos(n_EL)' + spk_conv_1_neg(n_I)*spk(n_EL,t)';
                    delG_I_ER = spk(n_I,t)*spk_conv_1_pos(n_EL)' + spk_conv_1_neg(n_I)*spk(n_EL,t)';
                    delG_EL_I = spk(n_EL,t)*spk_conv_2_neg(n_I)' + spk_conv_2_pos(n_EL)*spk(n_I,t)';
                    delG_ER_I = spk(n_EL,t)*spk_conv_2_neg(n_I)' + spk_conv_2_pos(n_EL)*spk(n_I,t)';
                    delG_I_I = spk(n_I,t)*spk_conv_2_neg(n_I)' + spk_conv_2_pos(n_I)*spk(n_I,t)';

                %update weight matrices
                    G_I_EL = G_I_EL + eta_IE*(delG_I_EL);
                    G_I_ER = G_I_ER + eta_IE*(delG_I_ER);
                    G_EL_I = G_EL_I + eta_EI*(delG_EL_I);
                    G_ER_I = G_ER_I + eta_EI*(delG_ER_I);
                    G_I_I = G_I_I + eta_II*(delG_I_I);

                %enforce Dale's law
                    G_I_EL = G_I_EL.*(G_I_EL>0);
                    G_I_ER = G_I_ER.*(G_I_ER>0);
                    G_EL_I = G_EL_I.*(G_EL_I>0);
                    G_ER_I = G_ER_I.*(G_ER_I>0);
                    G_I_I = G_I_I.*(G_I_I>0);


            %PLOT results every 200 time steps (100 ms)
            if mod(t,100)==0
                disp(['elapsed time: ', num2str(T*(tt-1) + t*dt),' sec'])
                subplot(8,2,[1 3]), plot(-G_EL_I(:,N_I/2)), hold on, plot(-G_ER_I(:,N_I/2),'r'), hold off, legend('I->EL','I->ER'), title('I->E','fontsize',16), xlim([1 N_E])
                subplot(8,2,[2 4]), plot(G_I_EL(:,N_E/2)), hold on, plot(G_I_ER(:,N_E/2),'r'), hold off, legend('EL->I','ER->I'), title('E->I','fontsize',16), xlim([1 N_I])
                subplot(8,2,[6 8]), plot(-G_I_I(:,N_I/2),'k'), title('I->I','fontsize',16), xlim([1 N_I])
                subplot(8,2,[5 7]), plot(diag(G_I_I,-5),'k'), legend('I->I'), title('weight diagonal from I->I matrix','fontsize',16), xlim([1 N_I])
                subplot(8,2,[9 11]), imagesc(G_EL_I), axis equal, axis([1 N_I 1 N_E]), title('I->EL','fontsize',16)
                subplot(8,2,10), imagesc(G_I_EL), axis equal, axis([1 N_E 1 N_I]), title('EL->I','fontsize',16)
                subplot(8,2,12), imagesc(G_I_I), axis equal, axis([1 N_I 1 N_I]), title('I->I','fontsize',16)
                subplot(8,2,13), plot(F(n_I)), xlim([1 N_I]), legend('I activity'), title('Population activity','fontsize',16)
                subplot(8,2,15), plot(F(n_EL)), hold on, plot(F(n_ER),'r'), hold off, xlim([1 N_E]), legend('EL activity','ER activity')
                subplot(8,2,[14 16]), plot((-l+dt:dt:l)*1000,[K1_neg,K1_pos(end:-1:1)],'b'), hold on, plot((-l+dt:dt:l)*1000,[K2_pos,K2_neg(end:-1:1)],'k'), hold off, legend('E presyn','I presyn'), title('STDP kernels','fontsize',16)
                drawnow
            end

        end %end INNER time loop (T sec)

        spk_prev = spk(:,end - l/dt+1:end); %copy spikes for next iteration of outer time loop

    end %end OUTER time loop (T_long sec)
   

    
    
    
    