function [spk,v,s] = gridcell_dynamics_1D(dynamics,periodic,temp,inh_strength)

% This function returns a spike matrix (spk), membrane potential dynamics 
% (v), and synaptic activations (s) for all neurons and all time in a 
% simulation of a network based on the following setttings:
% - dynamics = 0 for HH; dynamics = 1 for LIF; dynamics = 2 for LNP
% - periodic = 1 for periodic b.c.s; periodic = 0 for aperiodic b.c.s 
% - temp sets the temperature (temp=36 for default)
% - inh_strength sets strength of inhibition (inh_strengh = 1 for default);

%The output matrices have dimensions of neuron-by-time, where along the neuron
%axis the populations are arranged in the order I pop, EL pop, and ER pop

close all 
figure('Position', [900, 100, 800, 900]);

if isempty(temp)==1
    temp = 36;
end
if isempty(inh_strength)==1
    inh_strength = 1;
end

%%%%%%%%%%%%%%%%
%%%Parameters%%%
%%%%%%%%%%%%%%%%

    N_E = 400;                  %[neurons] num excitatory cells per Exc pop
    N_I = 160;                  %[neurons] num inhibitory cells
    N = N_I+2*N_E;              %[neurons] total num. of cells
    ind_I = 1:N_I;                %inhibitory cell indices
    ind_EL = N_I+1:N_I+N_E;       %left exc. cell indices
    ind_ER = N_I+N_E+1:N_I+2*N_E; %right exc. cell indices
    
    T = 1000;                   %[ms] simulation time 
    if dynamics == 0
        dt = 1/40;              %[ms] step size for HH dynamics
    elseif dynamics == 1
        dt = 1/10;              %[ms] step size for LIF 
    else
        dt = 1/2;               %[ms] step size for LNP dynamics
    end

    Q10_tau = 3;                %[au] Q10 factor for changing time constants of conductances
    Q10_amp = 1.3;              %[au] Q10 factor for changing conductance amplitudes

    temp_factor_tau = Q10_tau.^((temp-36)/10);  %[au] temperature-based modulation of conductance time constants
    temp_factor_amp = Q10_amp.^((temp-36)/10);  %[au] temp-based modulation of conductance amplitudes

    %membrane capacitance
    C_m = 1; %[microFarads/cm^2]

    %ionic conductance amplitudes
    g_L = 0.1*temp_factor_amp;  %[mS/cm^2]   
    g_Na = 50*temp_factor_amp;  %[mS/cm^2]   (only for HH dynamics)
    g_K = 5*temp_factor_amp;    %[mS/cm^2]
    g_M = 0.07*temp_factor_amp; %[mS/cm^2]

    %ionic conductance time constant prefactor
    phi = 1/temp_factor_tau;    %[au]

    %synaptic conductance amplitude prefactor
    g_syn = (1.5e-3)*temp_factor_amp; %[au] 

    %synaptic conductance time constant
    tau_g = 10/temp_factor_tau; %[ms]

    %sets width of weight profiles, and thus num. of bumps in population pattern
    rho = 2.2;                  % permissable values = [2,~22], where rho = 22 gives a single bump of activity
    
    %Reversal potentials (ionic and synaptic)
    E_L = -70;                  %[mV]   
    E_Na = 50;                  %[mV]
    E_K = -90;                  %[mV]
    E_E = zeros(2*N_E,1);       %[mV]   Exc. synapse reversal potential 
    E_I = -80*ones(N_I,1);      %[mV]   Inh. synapse reversal potential 

    %spiking threshold (used for both HH dyn. and LIF dynamics)
    V_T = -61.5;                %[mV] 

    %Reset potential to set voltage after spiking 
    Vreset = E_L;               %[mV]   (only for LIF dynamics) 
    
    %uniform excitatory input 
    I_app = 4;                  %[microAmps/cm^2] 
    
    %activity suppressing envelope for aperiodic nets (if periodic=1, envelope is flat)
    A_I = create_envelope(periodic,N_I); 
    A_E = create_envelope(periodic,N_E);
    A_env = [A_I,A_E,A_E]';

    
%%%%%%%%%%%%%%%%%%%%%%
%%%Synaptic weights%%%
%%%%%%%%%%%%%%%%%%%%%%

    [G_I_EL,G_I_ER,G_EL_I,G_ER_I,G_I_I] = compute_hardwired_weights(rho,N_E,N_I,periodic);
        
    G_I_EL = g_syn*G_I_EL;              %[mS/cm^2] synaptic conductance amplitudes: EL pop => I pop 
    G_I_ER = g_syn*G_I_ER;              %[mS/cm^2] '' ER=>I
    G_EL_I = inh_strength*g_syn*G_EL_I; %[mS/cm^2] '' I=>EL
    G_ER_I = inh_strength*g_syn*G_ER_I; %[mS/cm^2] '' I=>ER
    G_I_I = inh_strength*g_syn*G_I_I;   %[mS/cm^2] '' I=>I
    
%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%Initialize variables%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%

    v = V_T*ones(N,T/dt);   %track membrane potential over time
    spk = zeros(N,T/dt);    %track spikes over time
    s = zeros(N,T/dt);      %track synaptic activations over time
    s(:,1:2) = rand(N,2);

    %activation/iactivation variables
    h = rand(N,1);         
    m = rand(N,1);
    n = rand(N,1);
    p = rand(N,1);

%%%%%%%%%%%%%%%
%%%Time loop%%%
%%%%%%%%%%%%%%%

    for t = 3:T/dt

        %set dynamics
        if dynamics==0 %HH cortical neuron

            %ionic currents
                [I_ion,h,m,n,p] = compute_HH_ionic_currents(v(:,t-1),dt,g_Na,g_K,g_M,g_L,phi,E_Na,E_K,E_L,V_T,h,m,n,p);
            %synaptic currents
                [I_syn,s(:,t)] = compute_HH_syn_currents(spk(:,t-1),v(:,t-1),s(:,t-1),dt,ind_EL,ind_ER,ind_I,E_E,E_I,G_I_EL,G_I_ER,G_EL_I,G_ER_I,G_I_I,tau_g);
            %update internal state (v); output spikes
                [v(:,t),spk(:,t)] = compute_HH_output(v(:,t-1),v(:,t-2),dt,I_ion,I_syn,I_app,A_env,C_m);

        elseif dynamics==1 %LIF 

            %ionic currents
                [I_ion] = compute_LIF_ionic_currents(v(:,t-1),g_L,E_L);
            %synaptic currents                       
                [I_syn,s(:,t)] = compute_LIF_syn_currents(spk(:,t-1),v(:,t-1),s(:,t-1),dt,ind_EL,ind_ER,ind_I,E_E,E_I,G_I_EL,G_I_ER,G_EL_I,G_ER_I,G_I_I,tau_g);
            %update internal state (v); output spikes      
                [v(:,t),spk(:,t)] = compute_LIF_output(v(:,t-1),dt,I_ion,I_syn,I_app,A_env,C_m,V_T,Vreset);

        else %LNP

            %ionic currents
                [I_ion] = compute_LNP_ionic_currents(g_L);
            %synaptic currents
                [I_syn,s(:,t)] = compute_LNP_syn_currents(spk(:,t-1),s(:,t-1),dt,ind_EL,ind_ER,ind_I,G_I_EL,G_I_ER,G_EL_I,G_ER_I,G_I_I,tau_g);
            %output spikes
                [spk(:,t)] = compute_LNP_output(dt,I_ion,I_syn,I_app,A_env);

        end     

            %plots
            if mod(t,200)==0
                subplot(511), plot(s(ind_EL,t)), hold on, plot(s(ind_ER,t),'r','linewidth',2), hold off, xlabel('neuron'), ylabel('synaptic activation'), title('instantaneous synaptic activation of ER (red) and EL (blue) pops')
                subplot(512), imagesc(dt:10*dt:t*dt,1:N_E,s(ind_ER,1:10:t)), ylabel('neuron'), title('synaptic activation of ER pop'), xlabel('time (ms)')
                subplot(513), plot(dt:10*dt:t*dt,s(ind_ER(end/2-10:end/2+10),1:10:t)'), ylabel('synaptic activation'), title('synaptic activation of subset of ER pop'), xlabel('time (ms)')
                subplot(514), imagesc(dt:10*dt:t*dt,1:N_E,v(ind_ER,1:10:t)), ylabel('neuron'), title('membrane potential of ER pop'), xlabel('time (ms)')
                subplot(515), plot(dt:10*dt:t*dt,v((end/2-10:end/2+10),1:10:t)'), ylabel('mem. pot. (mV)'), xlabel('time (ms)'), title('membrane potential of subset of ER pop')
                drawnow
            end
    end





    