%%%%%%%%%%%
%%%SETUP%%%
%%%%%%%%%%%

    T = 60;             %length of integration time blocks (s)
    dt = .5/1000;       %step size of numerical integration (s)
    tau_s = 30/1000;    %synaptic time constant

    n = 40;             %population size per dim per pop
    N = n^2;            %population size per pop
    
    n_I = 1:N;
    n_EL = N+1:2*N;
    n_ER = 2*N+1:3*N;
    n_EU = 3*N+1:4*N;
    n_ED = 4*N+1:5*N;

    %External inputs
        beta_vel = 2.5;     %velocity gain
        beta_0_E = 60;      %const. excitatory input into E pop
        beta_0_I = 80;      %const. excitatory input into I pop
        beta_0_E_2 = 15;    %non-velocity-modulated const. excitatory input into E pop

        [X,Y] = meshgrid((1:n)/n,(1:n)/n); %location preferences 
        e = [[-1;0],[1;0],[0;1],[0;-1]];   %preffered directions

    %Suppressive envelope
        R = n-1;
        delr = n-n*.1;  %controls extent of flat inner region of envelope
        a0 = 10;        %controls slope of envelope
        A = zeros(n,n);
        for i = 1:n
            for j = 1:n
                dist = sqrt((n/2-i)^2+(n/2-j)^2);
                if dist<R-delr
                    A(i,j)=1;
                else
                    A(i,j) = exp(-a0*((dist-R+delr)/delr)^2);
                end
            end
        end
        A = reshape(A,N,1);

    %Geights
        load G_2D.mat

        amp_IE = 0.1*0.7; 
        amp_EI = 1.6*0.003;
        amp_II = 0.43*0.03;

        G_I_EL = amp_IE*G_I_EL; G_I_ER = amp_IE*G_I_ER; G_I_EU = amp_IE*G_I_EU; G_I_ED = amp_IE*G_I_ED;
        G_EL_I = amp_EI*G_EL_I; G_ER_I = amp_EI*G_ER_I; G_EU_I = amp_EI*G_EU_I; G_ED_I = amp_EI*G_ED_I;
        G_I_I = amp_II*G_I_I;

    %Plot weights
        singleCell_index = N/2-n/2;
        G_EL_I_reshape = reshape(G_EL_I(:,singleCell_index),n,n); G_ER_I_reshape = reshape(G_ER_I(:,singleCell_index),n,n); G_EU_I_reshape = reshape(G_EU_I(:,singleCell_index),n,n); G_ED_I_reshape = reshape(G_ED_I(:,singleCell_index),n,n);
        G_I_EL_reshape = reshape(G_I_EL(:,singleCell_index),n,n); G_I_ER_reshape = reshape(G_I_ER(:,singleCell_index),n,n); G_I_EU_reshape = reshape(G_I_EU(:,singleCell_index),n,n); G_I_ED_reshape = reshape(G_I_ED(:,singleCell_index),n,n);
        G_I_I_reshape = reshape(G_I_I(:,singleCell_index),n,n);
        
        subplot(3,1,1), imagesc([G_I_EL_reshape, G_I_ER_reshape, G_I_EU_reshape, G_I_ED_reshape]), colorbar, axis equal, axis([1 4*n 1 n]), title('I->EL, I->ER, I->EU, I->ED')
        subplot(3,1,2), imagesc([G_EL_I_reshape, G_ER_I_reshape, G_EU_I_reshape, G_ED_I_reshape, G_I_I_reshape]), colorbar, axis equal, axis([1 5*n 1 n]), title('EL->I, ER->I, EU->I, ED->I, I->I')

    %animal trajectory
        load trajectory_2D.mat
        v = diff(x')/dt; v = [v(1,:);v]; v= v';

    %initialize variables
        s_prev = zeros(5*N,1);
        
%%%%%%%%%%%%%%
%%%DYNAMICS%%%
%%%%%%%%%%%%%%   

    for t = 2:T/dt
           
        %I population
        g_I_I = -G_I_I*s_prev(n_I);
        g_I_EL = G_I_EL*s_prev(n_EL);
        g_I_ER = G_I_ER*s_prev(n_ER);  
        g_I_EU = G_I_EU*s_prev(n_EU);
        g_I_ED = G_I_ED*s_prev(n_ED);
        G_I = A.*(g_I_I + g_I_EL + g_I_ER + g_I_EU + g_I_ED) + beta_0_I;

        %EL population 
        v_L = 1 + beta_vel*dot(v(:,t),e(:,1)); v_L = v_L.*(v_L>0);
        g_EL_I = -G_EL_I*s_prev(n_I);                                  
        G_EL = v_L.*(A.*g_EL_I + beta_0_E) + beta_0_E_2;

        %ER population 
        v_R = 1 + beta_vel*dot(v(:,t),e(:,2)); v_R = v_R.*(v_R>0);
        g_ER_I = -G_ER_I*s_prev(n_I);                                  
        G_ER = v_R.*(A.*g_ER_I + beta_0_E) + beta_0_E_2;

        %EU population 
        v_U = 1 + beta_vel*dot(v(:,t),e(:,3)); v_U = v_U.*(v_U>0);
        g_EU_I = -G_EU_I*s_prev(n_I);                                  
        G_EU = v_U.*(A.*g_EU_I + beta_0_E) + beta_0_E_2;

        %ED population 
        v_D = 1 + beta_vel*dot(v(:,t),e(:,4)); v_D = v_D.*(v_D>0);
        g_ED_I = -G_ED_I*s_prev(n_I);                                  
        G_ED = v_D.*(A.*g_ED_I + beta_0_E) + beta_0_E_2;

        %pass conductance variables through nonlinearity to generate output rates, F
            G = [G_I;G_EL;G_ER;G_EU;G_ED];                             
            F = zeros(5*N,1) + G.*(G>=0);   %linear transfer function

        %spikes generated from inhomogeneous Poisson process
            spk = poissrnd(F*dt);
            
        %update neural activities    
            s_new = s_prev + spk - s_prev*dt/tau_s;
            s_prev = s_new;
            
        
        %Plot
        if (mod(t,50)==0)            
            disp(['elapsed time: ', num2str(t*dt),' sec'])

            F_I = reshape(F(n_I),n,n); 
            F_EL = reshape(F(n_EL),n,n);     
            F_ER = reshape(F(n_ER),n,n);
            F_EU = reshape(F(n_EU),n,n); 
            F_ED = reshape(F(n_ED),n,n); 
            
            subplot(3,2,5), plot(x(1,20:10:t),x(2,20:10:t),'linewidth',2), axis equal, title('animal trajectory')
            subplot(3,2,6), imagesc(F_I); axis square; colorbar, title('I pop activity')       
           
            drawnow
        end
        
    end 

    
    
    
    
    
    
    
    
    
%     bins = linspace(0+.01,2,40);
% edges = cell(2,1);
% edges{1}=bins; edges{2}=bins;

%         spk_singleCell(:,t) = spk(singleCell_index);

%             spk_product = [x(1,:).*spk(singleCell_index,:); x(2,:).*spk(singleCell_index,:)];
%             SN_response = hist3(spk_product','Edges',edges)'./hist3([x(1,:);x(2,:)]','Edges',edges)';
%             subplot(1,2,2), imagesc(SN_response)
