function [] = placecell_dynamics_1D

close all 
figure('Position', [900, 100, 800, 900]);

%%%%%%%%%%%%%%%%
%%%Parameters%%%
%%%%%%%%%%%%%%%%

    periodic = 1;

    N_E = 400;                  %[neurons] num excitatory cells per Exc pop
    N_I = 160;                  %[neurons] num inhibitory cells
    N = N_I+2*N_E;              %[neurons] total num. of cells
    ind_I = 1:N_I;                %inhibitory cell indices
    ind_EL = N_I+1:N_I+N_E;       %left exc. cell indices
    ind_ER = N_I+N_E+1:N_I+2*N_E; %right exc. cell indices
    
    T = 10000;                   %[ms] simulation time 
    dt = 1/2;               %[ms] step size for LNP dynamics

    %sets width of weight profiles, and thus num. of bumps in population pattern
    rho = 2.2;                  % permissable values = [2,~22], where rho = 22 gives a single bump of activity
    g_syn = 1.5e-3; %synaptic conductance amplitude prefactor
    tau_g = 10; %synaptic conductance time constant
    
    %uniform excitatory input 
    I_app = 4;                  %[microAmps/cm^2] 
    
    %activity suppressing envelope for aperiodic nets (if periodic=1, envelope is flat)
    A_I = create_envelope(periodic,N_I); 
    A_E = create_envelope(periodic,N_E);
    A_env = [A_I,A_E,A_E]';

%%%%%%%%%%%%%%%%%%%%%%
%%%Synaptic weights%%%
%%%%%%%%%%%%%%%%%%%%%%

    [G_I_EL,G_I_ER,G_EL_I,G_ER_I,G_I_I] = compute_hardwired_weights(rho,N_E,N_I,periodic);
        
    G_I_EL = g_syn*G_I_EL;              %[mS/cm^2] synaptic conductance amplitudes: EL pop => I pop 
    G_I_ER = g_syn*G_I_ER;              %[mS/cm^2] '' ER=>I
    G_EL_I = g_syn*G_EL_I; %[mS/cm^2] '' I=>EL
    G_ER_I = g_syn*G_ER_I; %[mS/cm^2] '' I=>ER
    G_I_I = g_syn*G_I_I;   %[mS/cm^2] '' I=>I
    
%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%Initialize variables%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%

    spk = zeros(N,T/dt);    %track spikes over time
    s = zeros(N,T/dt);      %track synaptic activations over time
    s(:,1:2) = rand(N,2);

%%%%%%%%%%%%%%%
%%%Time loop%%%
%%%%%%%%%%%%%%%

    for t = 3:T/dt

        %ionic currents
            I_ion = 0;
        %synaptic currents
            [I_syn,s(:,t)] = compute_LNP_syn_currents(spk(:,t-1),s(:,t-1),dt,ind_EL,ind_ER,ind_I,G_I_EL,G_I_ER,G_EL_I,G_ER_I,G_I_I,tau_g);
        %output spikes
            [spk(:,t)] = compute_LNP_output(dt,I_ion,I_syn,I_app,A_env);

            %plots
            if mod(t,200)==0
                subplot(311), plot(s(ind_EL,t)), hold on, plot(s(ind_ER,t),'r','linewidth',2), hold off, xlabel('neuron'), ylabel('synaptic activation'), title('instantaneous synaptic activation of ER (red) and EL (blue) pops')
                subplot(312), imagesc(dt:10*dt:t*dt,1:N_E,s(ind_ER,1:10:t)), ylabel('neuron'), title('synaptic activation of ER pop'), xlabel('time (ms)')
                subplot(313), plot(dt:10*dt:t*dt,s(ind_ER(end/2-10:end/2+10),1:10:t)'), ylabel('synaptic activation'), title('synaptic activation of subset of ER pop'), xlabel('time (ms)')
                drawnow
            end
    end





    