#!/usr/bin/env python
"""
Pattern formation animation
FitzHugh Nagumo model
ENTPD5 patttern of the notochord 
with input from the segmentation clock
as a sink profile
"""

"""
%matplotlib inline
%matplotlib notebook
PLEASE switch to the notebook environment with this command
"""

import numpy as np
import matplotlib.pyplot as plt
import os, glob
import time

csfont = {'fontname':'Comic Sans MS'}
hfont  = {'fontname':'Helvetica'}

start_time = time.time()

''' this one computes the Laplacian '''
def laplacian(Z):
    Zleft   = Z[0:-2]
    Zright  = Z[2:]
    Zcenter = Z[1:-1]
    return (Zleft + Zright - 2 * Zcenter) / dx**2
    
''' this one integrates the equations '''
def fhnsolve():
    global U, V
    for t in range(tu):
        # computer, compute the Laplacian of u and v
        deltaU = laplacian(U)
        deltaV = laplacian(V)
        # take the values of u and v inside the grid
        Uc = U[1:-1]
        Vc = V[1:-1]
        # this one updates the variables
        U[1:-1], V[1:-1] = \
            Uc + dt * (a * deltaU + Uc - Uc**3   - Vc + k), \
            Vc + dt * (b * deltaV + Uc - c[1:-1] * Vc - d * Vc) / tau
        # these are Neumann conditions: derivatives at the edges are null
        for Z in (U, V):
            Z[0]  = Z[1]
            Z[-1] = Z[-2]      

def multicolor_ylabel(ax,list_of_strings,list_of_colors,axis='x',anchorpad=0,**kw):
    """this function creates axes labels with multiple colors
    ax specifies the axes object where the labels should be drawn
    list_of_strings is a list of all of the text items
    list_if_colors is a corresponding list of colors for the strings
    axis='x', 'y', or 'both' and specifies which label(s) should be drawn"""
    from matplotlib.offsetbox import AnchoredOffsetbox, TextArea, HPacker, VPacker

    # x-axis label
    if axis=='x' or axis=='both':
        boxes = [TextArea(text, textprops=dict(color=color, ha='left',va='bottom',**kw)) 
                    for text,color in zip(list_of_strings,list_of_colors) ]
        xbox = HPacker(children=boxes,align="center",pad=0, sep=5)
        anchored_xbox = AnchoredOffsetbox(loc=3, child=xbox, pad=anchorpad,frameon=False,bbox_to_anchor=(0.2, -0.09),
                                          bbox_transform=ax.transAxes, borderpad=0.)
        ax.add_artist(anchored_xbox)

    # y-axis label
    if axis=='y' or axis=='both':
        boxes = [TextArea(text, textprops=dict(color=color, ha='left',va='bottom',rotation=90,**kw)) 
                     for text,color in zip(list_of_strings[::-1],list_of_colors) ]
        ybox = VPacker(children=boxes,align="center", pad=0, sep=5)
        anchored_ybox = AnchoredOffsetbox(loc=3, child=ybox, pad=anchorpad, frameon=False, bbox_to_anchor=(-0.10, 0.2), 
                                          bbox_transform=ax.transAxes, borderpad=0.)
        ax.add_artist(anchored_ybox)


''' numerics parameters '''
answer = 42
numrel =  1                 # number of realizations

WTL= 17.1                   # wildtype total length
WTW= 0.57                   # average wildtype wavelength

L  = WTL         / 1        # system length 
T  = 218.4       / 1        # total integration time 
numframes = 550  / 1        # the number of frames in the animation

dx = 0.010                  # discretization length
discretex=np.arange(0.,L,dx)# discretized length axis
Lx = np.int(np.rint(L/dx))  # number of lattice sites  >> Lx = int(L/dx)          

dt = 0.9 * dx**2 / 2        # time step
nt = int(T/dt)              # total number of timesteps in the simulation
tu = int (T/(numframes*dt)) # given number of frames, compute the dimensionless timeunit for animation

''' this one defines the length axis scaling for plotting '''
xscaled = discretex / WTL   # normalized to wildtype total length


''' theory parameters '''
a  = 1.0e-3
b  = 1.0e-2
tau= 0.1
k  = 0.0
d  = 0.5

#for a in np.linspace(0.0001,0.001,10):
for a in np.linspace(0.001,0.001,1):
#   for b in np.linspace(0.001 ,0.01 ,10):
    for b in np.linspace(0.01 ,0.01 ,1):
        for r in range(numrel):
#       for r in range(numrel,5*numrel):
            plt.figure(44, figsize=(12, 4), dpi=100, facecolor='w')
            plt.clf() 
                        
            ''' smooth sinks with random positions '''
            ''' sequential adding sinks of perturbed wavelength '''
            sinko  = 8.00
            sinkwl = WTW
            relerr = 0.05

            sinkwe = relerr * sinkwl
            sinksteep = 100.
            sinkwidth = 0.05
            numsinks  = int(L/sinkwl)
            sinkpositions = [0.5*sinkwl]
            np.array([sinkpositions.append(sinkpositions[-1] + sinkwl + sinkwe * (2.*np.random.rand() - 1.)) for i in range(numsinks)])
            sinkpositions = np.around(sinkpositions,8)
            singlesinks = lambda sp,xp: -np.tanh(sinksteep*(-sp + xp - sinkwidth)) + np.tanh(sinksteep*(-sp + xp + sinkwidth))
            sinkprofile = sinko*0.5*np.array([np.sum(singlesinks(sinkpositions,xi)) for xi in discretex])
            c = sinkprofile

            """ smooth localized source initial condition """
            inisteep = 5.0;
            iniwidth = 0.5;
            uo = 0.15 + 0.5 * np.random.rand();
            vo = 0.15 + 0.5 * np.random.rand();
            U = 0.5 * uo * (1. - np.tanh( inisteep * (discretex - iniwidth) ) )
            V = 0.5 * vo * (1. - np.tanh( inisteep * (discretex - iniwidth) ) )

            """ plot initial conditions if you may """
            plt.figure(22)
            plt.clf()
            plt.subplot(3,1,1)
            plt.plot(xscaled,U, color='xkcd:green', lw=2)
            plt.plot(xscaled,V, color='xkcd:red',   lw=2)
            plt.xlim(0,np.around(xscaled[-1],2))
            plt.ylim(-1.0,1.0)
            plt.xticks([])
            plt.show()

            """ this creates temporary output folder and goes there for the ride """
            folder = ('_snapshots_a{}_b{}_sinko{}_sinkwl{}_error{}_r{}.dat'.format(a, b, sinko, sinkwl, relerr, r))
            os.mkdir(folder)
            os.chdir(folder)
            
            """ plot sink profile and activator - inhibitor / save figure         """
            """ integrates the equations and saves snapshots at regular intervals """
            for t in range(numframes):
                plt.figure(44)
                plt.subplot(3,1,2)
                plt.title('frame = '+str(t), fontsize=10, **csfont)
                plt.plot(xscaled,c,  '-', color='xkcd:azure', lw=2)
                plt.xlim(0,np.around(xscaled[-1],2))
                plt.ylim(0,12)
                plt.xticks([])
                plt.ylabel('sink',      color='xkcd:azure')
                
                ax3=plt.subplot(3,1,3)
                plt.plot(xscaled,U, color='xkcd:green', lw=2)
                plt.plot(xscaled,V, color='xkcd:red',   lw=2)
                plt.xlim(0,np.around(xscaled[-1],2))
                plt.ylim(-1,1)
                plt.xlabel('position / $L_{WT}$',    color='xkcd:black')
                multicolor_ylabel(ax3,('activator','and','inhibitor'),
                                  ('xkcd:red','xkcd:black','xkcd:green'),axis='y')
                plt.legend(bbox_to_anchor=(1.002, 1), loc=2, borderaxespad=0.)
                plt.show()

                ''' this one saves you some pdf snapshots '''
                if t in [0, 100, 200, 300, 400, 500]:
                    plt.savefig('fhn_1d_elife_animationsnapshot_a{}_b{}_sinko{}_sinkwl{}_error{}_r{}_t{}.pdf'.format(a, b, sinko, sinkwl, relerr, r, t),format='pdf')
                
                filename = str('%04d' % t) + '.png'
                plt.savefig(filename, dpi=100)
                plt.clf()

                fhnsolve()            

            print 'png Done.'
            print("--- %s seconds ---" % (time.time() - start_time))

            plt.figure(22)
            plt.subplot(3,1,2)
            plt.plot(xscaled,c,  '-', color='xkcd:azure', lw=2)
            plt.xlim(0,np.around(xscaled[-1],2))
            plt.ylim(0,30)
            plt.xticks([])
            plt.show()

            plt.subplot(3,1,3)
            plt.plot(xscaled,U, color='xkcd:green', lw=2)
            plt.plot(xscaled,V, color='xkcd:red',   lw=2)
            plt.xlim(0,np.around(xscaled[-1],2))
            plt.ylim(-1,1)
            plt.show()
            plt.savefig('fhn_1d_elife_animationsnapshot_a{}_b{}_sinko{}_sinkwl{}_error{}_r{}.pdf'.format(a, b, sinko, sinkwl, relerr, r),format='pdf')

            ''' this one saved the final snapshot data for later reference or use '''
            np.savetxt('fhn_1d_elife_animationsnapshotdata_a{}_b{}_sinko{}_sinkwl{}_error{}_r{}.dat'.format(a, b, sinko, sinkwl, relerr, r), np.transpose([xscaled,c,U,V]), fmt='% 10.4f')
            ''' read from elsewhere with:
                [xx,cc,uu,vv] = np.loadtxt('test.out', unpack=True) '''
            
            ''' this one saved the animation '''
            aniname =  'fhn_1d_elife_animation_a{}_b{}_sinko{}_sinkwl{}_error{}_r{}.avi'.format(a, b, sinko, sinkwl, relerr, r)
            command = ('mencoder',
                       'mf://*.png',
                       '-mf',
                       'type=png:w=1200:h=400:fps=60',
                       '-ovc',
                       'lavc',
                       '-lavcopts',
                       'vcodec=mpeg4',
                       '-oac',
                       'copy',
                       '-o',
                       aniname)            
            os.spawnvp(os.P_WAIT, 'mencoder', command)
            print 'avi Done.'
                        
            ''' this one saved sanpshots of the animation '''
            snaps = [ 0, 100, 200, 300, 400, 500 ]
            for timestamp in snaps:
                filename    = str('%04d' % timestamp) + '.png'
                filerename  = 'fhn_1d_elife_animationsnapshot_a{}_b{}_sinko{}_sinkwl{}_error{}_time{}.png'.format(a, b, sinko, sinkwl, relerr, timestamp)
                copycommand = ['cp', filename, filerename]
                os.spawnvpe(os.P_WAIT, 'cp', copycommand, os.environ)
            print 'snaps Done.'


            ''' this one cleared temporary files '''
            [os.remove(thesefiles) for thesefiles in glob.glob("0*.png")]

            print("--- %s seconds ---" % (time.time() - start_time))                        
            print("Realization "+str(r)+" is done mate.")
            ''' now take me back to the previous folder '''
            os.chdir('../')


''' this one saved the animation using ffmpeg instead of mencoder '''
#command = ('ffmpeg',
#'-r 60',             # frame rate
#'-f image2',         # force format ???
#'-s 600x400',        # resolution
#'-i %04d.png',       # input files
#'-vcodec libx264',   # codec used to compress video
#'-crf 10',           # quality, lower value better quality, 15-25 is ok
#'-pix_fmt yuv420p',  # pixel format
#'___test.mp4')       # output file
#
#os.spawnvp(os.P_WAIT, 'ffmpeg', command)
#print 'mp4 Done.'
#
#'''
#ffmpeg -r 60 -f image2 -s 600x400 -i %04d.png -vcodec libx264 -crf 10 -pix_fmt yuv420p ___test.mp4
#'''















