% plot detection performance and reaction time
%
%INPUT:
%options structure with fields -  fold:        ('on'/'off')
%                              -  contrast:    (1, 2, 'both', 'combine')
%                              -  session:     (1, 2, 'both')                
%                              -  stats:       ('on'/'off')                
%                              -  scatterplot: ('on'/'off')  
%                              -  lineplot:    ('on'/'off')  

%FIGURES
%figure 1: detection performance vs angle
%figure 2: RT vs angle
%figure 3: detection performance scatterplot
%figure 4: RT scatterplot
%
%STATS
% detection ANOVA

function [y, RT] = detection(new_opt, root, ind_plot, split)


%% set path and option structure
addpath(genpath('..'));
%close all

opt = struct(...
    'fold', 'on',...
    'contrast', 'combine',...
    'session',  'both',...
    'stats', 'off',...
    'scatterplot', 'on',...
    'lineplot', 'on'...
    );

names = fieldnames(opt);
for i = 1:numel(names)
    try
        eval(sprintf('opt.%s = new_opt.%s;', names{i}, names{i}));
    end
end

try 
    y = csvread('Detection.csv');
    RT = csvread('EstimationRT.csv');
catch
    %% collect data
    DATA = collect_data(root); 
    angles = [-64 -48 -32  -16  0  16  32 48 64];
    
    if strcmp(opt.contrast, 'combine')
        for i = 1:size(DATA, 1)
            DATA{i}(round(DATA{i}(:, 2))==3, 2) = 2; opt.contrast = 1;        
        end
    end
    
    %% collect matrix z of 'fraction detected data'
    [n_det_stim, n_stim, rt, rt_std] = data_detect(DATA);
    Y = n_det_stim./n_stim;

    %% average data across contrasts
    if strcmp(opt.contrast, 'both')
        y = squeeze(mean(Y, 3));
        RT = squeeze(mean(rt, 3));
        RT_std = squeeze(mean(rt_std, 3));
        numb_part = size(y,1);      %number of participants
    else
        y = squeeze(Y(:, :, opt.contrast));
        RT = squeeze(rt(:, :, opt.contrast));
        RT_std = squeeze(rt_std(:, :, opt.contrast));
        numb_part = size(y,1);
    end
end

angles = [-64 -48 -32  -16  0  16  32 48 64];

%% if fold, collapse data across central angle
if strcmp(opt.fold, 'on')
    angles = [0 16 32 48 64];
    y = [y(:, 5) (y(:, 6:9)+y(:, 4:-1:1))/2];
    RT = [RT(:, 5) (RT(:, 6:9)+RT(:, 4:-1:1))/2];
    RT_std = [RT_std(:, 5) (RT_std(:, 6:9)+RT_std(:, 4:-1:1))/2];
end

if strcmp(split, 'on')

    %% general population data
    Traits = csvread('Traits.csv');
    AQ = Traits(:,3);
    
    %AQ
    spl = median(AQ);
    hy = y(AQ>spl,:); hRT = RT(AQ>spl,:);
    ly = y(AQ<spl,:); lRT = RT(AQ<spl,:);
    leg1 = 'High AQ'; leg2 = 'Low AQ';

    %RISC
%     spl = median(RISC);
%     hy = y(RISC>spl,:); hRT = RT(RISC>spl,:);
%     ly = y(RISC<=spl,:); lRT = RT(RISC<=spl,:);
%     leg1 = 'High RISC'; leg2 = 'Low RISC';

    %SPQ
%     RT(40:end,:)=[]; MU(40:end,:)=[]; B(40:end,:)=[]; 
%     y(40:end,:)=[]; SIGMA(40:end,:)=[];
%     spl = median(SPQ);
%     hy = y(SPQ>spl,:); hRT = RT(SPQ>spl,:);
%     ly = y(SPQ<=spl,:); lRT = RT(SPQ<=spl,:);
%     leg1 = 'High SPQ'; leg2 = 'Low SPQ';

    %% patients and controls data   
%     num = 17; % number of controls
%     ly = y(1:num,:); lRT = RT(1:num,:);
%     hy = y(num+1:end,:); hRT = RT(num+1:end,:);  
%     leg1 = 'Patients'; leg2 = 'Controls';
end

if strcmp(opt.lineplot, 'on')
    
%% Fraction detected as a function of angles    
    figure(1); hold on; grid on
    
    if strcmp(split, 'on')
        withinsubject_errorbar(angles, 100*y, 'k')
        withinsubject_errorbar(angles, 100*hy, 'b')           %higher half
        withinsubject_errorbar(angles, 100*ly, 'r')           %lower half
    else
        withinsubject_errorbar(angles, 100*y, 'k')
        
        if strcmp(ind_plot, 'on')
            plot(angles, 100*y, 'o-', 'LineWidth', 1)           %Individual perf
        end
    end
   
    if strcmp(opt.fold, 'on')
        set(gca, 'Xlim', [-5 70], 'Fontsize', 16); %, 'Ylim', [55 80], 'Fontsize', 16)
    else
        set(gca, 'Xlim', [-70 70], 'Fontsize', 16, 'Ylim', [55 80], 'Fontsize', 16)
    end
    
    vline(32, 'k--'); vline(-32, 'k--');
    xlabel('Motion direction (deg)', 'Fontsize', 20)
    ylabel('Fraction detected (%)', 'Fontsize', 20)
    
    if strcmp(split, 'on')
        legend('All',leg1, leg2)
    end
    
    set(gca, 'box', 'on', 'linewidth', 1)    
    %set(gca, 'ytick',  [60 65, 70], 'xtick', 0:20:60)

%% Reaction time as a function of angles    
    figure(2); hold on; grid on;
    
    if strcmp(split, 'on')
        withinsubject_errorbar(angles, RT, 'k')
        withinsubject_errorbar(angles, hRT, 'b')           %higher half
        withinsubject_errorbar(angles, lRT, 'r')           %lower half
    else
        withinsubject_errorbar(angles, RT, 'k')
        
        if strcmp(ind_plot, 'on')
            plot(angles, RT, 'o-', 'LineWidth', 1)           %Individual perf
        end
    end
   
    if strcmp(opt.fold, 'on')
        set(gca, 'Xlim', [-5 70], 'Fontsize', 16); %, 'Ylim', [-18 10], 'Fontsize', 16)
    else
        set(gca, 'Xlim', [-70 70], 'Fontsize', 16); %, 'Ylim', [1.95 2.25], 'Fontsize', 16)
    end
    
    vline(32, 'k--'); vline(-32, 'k--');
    xlabel('Motion direction (deg)', 'Fontsize', 20)
    ylabel('Reaction time (s)', 'Fontsize', 20)
    
    if strcmp(split, 'on')
        legend('All',leg1, leg2)
    end
    set(gca, 'box', 'on', 'linewidth', 1)   
end

%% scatterplot: attended performance vs unattended performance
if strcmp(opt.scatterplot, 'on')    
    if strcmp(opt.fold, 'off')
        y = [y(:, 5) (y(:, 6:9)+y(:, 4:-1:1))/2];
        RT= [RT(:, 5) (RT(:, 6:9)+RT(:, 4:-1:1))/2];
    end
    %detection perf
    attended_det =  y(:, 3);
    unattended_det = mean(y(:, [1 2 4 5]), 2);
    all_det = mean(y(:, [1 2 3 4 5]), 2);
    frac_det = attended_det - mean(y(:,[2 4]), 2);
    [max_det, Ind] = max(y,[],2);
    
    for i = 1:length(Ind)
        dum = y(i,:);
        dum(Ind(i)) = [];
        rest_det(i,:) = dum;
    end
    
    frac_det = max_det - mean(rest_det,2);
   
    attended_RT =  RT(:, 3);
    unattended_RT = mean(RT(:, [1 2 4 5]), 2);
    all_RT = mean(RT(:, [1 2 3 4 5]), 2);
    frac_RT = attended_RT - unattended_RT;

    p_det = signrank(attended_det, unattended_det);
    p_RT = signrank(attended_RT, unattended_RT);
    
    figure('Name', 'Detection performance')
    scatter_plot(100*attended_det, 100*unattended_det, 'k', 1)
    xlabel('% detected frequent directions', 'Fontsize', 24)
    ylabel('% detected infrequent directions', 'Fontsize', 24)        
    set(gca, 'ytick',  50:10:80, 'xtick',  50:10:80, 'Fontsize', 16, ...
            'Xlim', [45 85], 'Ylim', [45 85]);        
    text(48, 80, 'p<0.001', 'Fontsize', 24)
    
    figure('Name', 'Detection performance')
    scatter_plot(attended_RT, unattended_RT, 'k', 1)
    xlabel('reaction time frequent directions (ms)', 'Fontsize', 24)
    ylabel('reaction time infrequent directions(ms)', 'Fontsize', 24)    
    set(gca, 'ytick',  1.6:0.2:2.6, 'xtick',  1.6:0.2:2.6, 'Fontsize', 16, ...
        'Xlim', [1.45 2.6], 'Ylim', [1.45 2.6])
    text(1.6, 2.35, 'p<0.001', 'Fontsize', 24)
    

    fprintf('fraction detected at 32 = %f%f\n', 100*mean(attended_det), 100*std(attended_det)/sqrt(numb_part));
    fprintf('average for the remaining directions= %f%f\n', 100*mean(unattended_det), 100*std(unattended_det)/sqrt(numb_part));
    fprintf('average fraction detected= %f%f\n', 100*mean((all_det)), 100*std(all_det)/sqrt(numb_part));
    
    fprintf('difference in average fraction detected= %f%f\n', 100*mean(attended_det - unattended_det), 100*std(attended_det - unattended_det)/sqrt(numb_part));

    fprintf('RT at 32 = %f%f\n', 100*mean(attended_RT), 100*std(attended_RT)/sqrt(numb_part));
    fprintf('average RT= %f%f\n', 100*mean(unattended_RT), 100*std(unattended_RT)/sqrt(numb_part));
end


%% statistics
if strcmp(opt.stats, 'on')
    angles = [-64 -48 -32  -16  0  16  32 48 64];
    opt.display_stats = 'on';
    stats(Y,  angles, opt); 
    stats(rt,  angles, opt);
%     mstats(Y, rt,  angles, opt);    
end