% --------------------------------------------------------------
% MAIN SCRIPT
% AUTHOR: GURDIP S. UPPAL
% DATE: 2/2/2018
% --------------------------------------------------------------
% === CLEAR WORKSPACE ===
close all; clear; clc;
% TIME SIMULATION:
tic; % start stopwatch
%----------------------------------------------------------------
%----------------------------------------------------------------


% ===== USER DEFINED PARAMETERS =====

% === SPACE-TIME PARAMETERS ===
% SPACE:
xmin = -25;
xmax = 25;
ymin = -20;
ymax = 20;

% INITIALIZATION DOMAIN:
xbmin = xmin;
xbmax = xmax;
ybmin = ymin;
ybmax = ymax;

% BOUNDARY CONDITIONS:
% (0 == periodic --- 1 == no flux)
xbnd = 0;
ybnd = 1;

% TIME:
tmax = 2.0;

% === SYSTEM PARAMETERS ===
% diffusion:
db = 0.3906;       % bacteria diffusion 
d1 = 5;         % resource diffusion contant
d2 = 15;        % waste diffusion constant
% fitness:
a1 = 75;        % resource benefit
a2 = 80;        % waste harm
b1 = 0.2;       % resource secretion cost
k1 = 0.01;      % resource saturation constant
k2 = 0.1;       % waste saturation constant
% chemicals:
s1 = 92;       % initial resource secretion (subject to mutations)
s2 = 91;       % waste secretion rate
l1 = 50;        % resource decay rate
l2 = 15;        % waste decay rate


% === FLOW VELOCITY PARAMETERS ===
% veltype options:
% 1 = Couette
% 2 = Hagen - Poisuille
% 3 = Rankine Vortex
% 4 = Constant (useful for sticky bacteria)
% else: no flow 
veltype = 2;
vmax = 80;           % maximum velocity for flow types 1,2, and 4
rotation = 0;       % vortex rotation rate \Gamma for flow type 3
vrad = 0;           % vortex radius R for flow type 3


% === CHEMICAL CONTROL FUNCTIONS ===
% amplitude:
mu1 = 0.0;      % amplitude of externally added chemical signal for public good (set to 0 for no intervention)
mu2 = 0.0;      % amplitude of externally added chemical signal for toxin (set to 0 for no intervention)
% pulses    (1 == pulsed, 0 == constant)
pulses = 0;
% delays:
mudel1 = 0.0;
mudel2 = 0.0;
% on period:
muOn1 = 0.00;
muOn2 = 0.00; 
% off period:
muOff1 = 0.0;
muOff2 = 0.0; 


% === MUTATION PARAMETERS ===
mscale = 10^(-7);
mutProb = 8*mscale;                 % mutation probability; between 0 and 1
mutDiff = s1;                       % how much secretion rate changes on evolution
binaryMutation = 0;                 % this setting means a mutations only have two types: 0 and s1

% manual mutation:
numDetMutants = 0;                      % number of mutants introduced manually
deterministicMutationTime = 0.00;       % when to introduce mutants manually 
% (can set mutation probablity to 0 to only have this mutation)


% === BACTERIA INITIALIZATION ===
initialNumBacteria = 1000;       % initial bacteria population
initialNumGroups = 10;           % number of initial groups (0 for uniform)
aligngroups = 0;                % option to align groups in center (1 == true)
sticky = 0;                     % option to have bacteria uneffected by flow (1 == true)


% === SAVING AND VISUALIZATION ===
savePeriod = 100; 
% saves after this many time steps
saveChemicals = 0;
% to save the chemical fields alongside the bacteria
savefile = sprintf('test'); 
% file name for data and video output
graphics = 1;
% option to display graphics as running (1 == true) (having this on slows
% down simulation)
framePeriod = 100;  % after how many time steps to display data
saveVideo = 0;      % option to save video of simulation (1 == true)






%-----------------------------------------------------------------------
%-----------------------------------------------------------------------
% ===== DERIVED AND RESERVED PARAMETERS =====
% (normally don't change these)

% === DISCRETIZATION ===
Nx = round((xmax-xmin)*5);    % x nodes - 1, dx = 0.2 
Ny = round((ymax-ymin)*5);    % y nodes - 1, dy = 0.2 
dt = 0.0001;                  % time step size

% === MAX BACTERIA SIZE ===
maxBacteria = 100000; 
% simulation stops after number of bacteria exceed maxBacteria

% === DELAYS ===
veldelay = 0.0010;       % delay time for flow 
mutdelay = 0.0010;       % delay for mutation 
repdelay = 0.0005;       % delay for reproduction

% === SAVE PARAMETERS ===
datechar = datestr(now,'mm-dd-yy_HH-MM-SS');
dataFile = sprintf('%s_%s_data.mat',savefile,datechar);
vidOption = 1; %(2 == avi, else mp4)
vidFile = sprintf('%s_%s_vid',savefile,datechar);



%-------------------------------------------------------------------
%-------------------------------------------------------------------
%-------------------------------------------------------------------

% ===== RUN PROGRAM =====
[dataB, dataChem] = simulator(xmin,xmax,xbnd,Nx,ymin,ymax,ybnd,Ny,tmax,dt,...
    xbmin,xbmax,ybmin,ybmax,...
    db,d1,d2,a1,a2,b1,k1,k2,s1,s2,l1,l2, ...
    veltype,vmax,rotation,vrad,...
    mu1,mu2,pulses,mudel1,mudel2,muOn1,muOn2,muOff1,muOff2,...
    mutProb,mutDiff,binaryMutation,deterministicMutationTime,numDetMutants,...
    sticky,initialNumBacteria,initialNumGroups,aligngroups,maxBacteria,...
    veldelay,mutdelay,repdelay,...
    savePeriod,saveChemicals,...
    graphics,framePeriod,saveVideo,vidOption,vidFile);

% === SAVE DATA ===
if saveChemicals == 1
    data = {dataB, dataChem};
else
    data = dataB;
end

parsave(dataFile,data);

% === DISPLAY RUN TIME ===
rntime = toc;
fprintf('\nProgram run time: %f \n',rntime);


%=======================================================================
% END OF FILE
%=======================================================================




