#!/usr/local/bin/Rscript
## functions for vennpie plot
# 2014-05-15, tushikui@gmail

vennpie <- function( A,B,AB, main="venn-pie", labels.diag=rownames(A), show.overlap.size=FALSE, show.overlap.pvalue=FALSE, overlap.test.all=NA, graph.grid="upper-triangle", significance.threshold=1e-3, cex.overlap.size=0.3, vp.cols=NULL )
# Input: A: n*n matrix for num of setA elements
#        B: n*n matrix for num of setB elements
#       AB: n*n matrix for num of intersect(setA, setB) elements
# (optional)
#    graph.grid: "upper-triangle" (default, for pairwise overlap among ONE group)
#                "bipartite" (for pairwise overlap between TWO groups)
{
	tot <- A + B - AB;  # total num of elements (union)

  ## CASE(1)
  if( graph.grid == "upper-triangle" )
  {
	n <- nrow(A);            # num of categories
	if( is.null(vp.cols) ){
		vp.cols <- rainbow(n);   # colors for each category
	}
	
	# Empty figure: rectangle(0,0,n,n)
	plot( c(0,n),c(0,n), type="n",xlab="",ylab="",main=main,xlim=c(0,n),ylim=c(0,n),axes=FALSE );

	# plot a venn-pie for each grid (upper-triangle)
	for( i in 1:n ){
		for( j in 1:n ){
			if( i==j ){  # names on diagonal grids
				text( i-0.5,j-0.5, labels=sprintf("%s\n(%d)",labels.diag[i],A[i,j]), pos=NULL, cex=max(c(0.3,1-0.4*n/10)), col=vp.cols[i] );
			}
			# venn-pie on off-diagonal grids
			if( i!=j ){ 
				if( tot[i,j] > 0 ){
					draw.vennpie( x=i-0.5,y=j-0.5, s=c(A[i,j],B[i,j],AB[i,j]),vpcol=c(vp.cols[i],vp.cols[j],"grey") );

					if( show.overlap.size ){
						if( !is.na(overlap.test.all) ){
							pv <- vennPvalue( overlap.test.all, A[i,j],B[i,j],AB[i,j] );
						}else{
							pv <- 1;
						}
						if( pv < significance.threshold ){
						  text( x=i-0.5,y=j-0.5, labels=sprintf("%d",AB[i,j]),cex=cex.overlap.size+0.05,font=4 );
						}else{
						  text( x=i-0.5,y=j-0.5, labels=sprintf("%d",AB[i,j]),cex=cex.overlap.size,font=1,col="grey50" );
						}
					}
					if( show.overlap.pvalue ){
						text( x=i-0.5,y=j-0.5, labels=sprintf("%.0e",vennPvalue(overlap.test.all, A[i,j],B[i,j],AB[i,j]) ), cex=0.3 );
					}
				}
			}
		}
	}
  }

  ## CASE(2)
  if( graph.grid == "bipartite" )
  {
	nA <- nrow(A);  # nA*nB matrix
	nB <- ncol(B);  # nA*nB matrix
	if( is.null(vp.cols) ){
		vp.cols <- rainbow(nA+nB);
	}
	vp.cols.A <- vp.cols[1:nA];
	vp.cols.B <- vp.cols[(nA+1):(nA+nB)];

	# Empty figure: rectangle(0,0,n,n)
	plot( c(0,nA),c(0,nB), type="n",xlab="",ylab="",main=main,xlim=c(0,max(nA,nB)),ylim=c(0,max(nA,nB)),axes=FALSE );
	text( (1:nA)-0.5,0,labels=rownames(A),srt=-45,adj=0,cex=0.6,xpd=TRUE,col=vp.cols.A );
	text( 0,(1:nB)-0.5,labels=colnames(B),pos=2,cex=0.6,xpd=TRUE,col=vp.cols.B );
	text( (1:nA)-0.5,nB, labels=A[,1], pos=3,cex=0.6,xpd=TRUE );
	text( nA,(1:nB)-0.5, labels=B[1,], pos=4,cex=0.6,xpd=TRUE );
	#axis( 1, 1:nA, labels=rep("",nA), tick=TRUE );

	# plot a venn-pie for each grid (rectangle)
	for( i in 1:nA ){
		for( j in 1:nB ){
			if( tot[i,j] > 0 ){
				draw.vennpie( x=i-0.5,y=j-0.5, s=c(A[i,j],B[i,j],AB[i,j]),vpcol=c(vp.cols.A[i],vp.cols.B[j],"grey") );
				if( show.overlap.size ){
					pv <- vennPvalue( overlap.test.all, A[i,j],B[i,j],AB[i,j] );
					if( pv < significance.threshold ){
					  text( x=i-0.5,y=j-0.5, labels=sprintf("%d",AB[i,j]),cex=cex.overlap.size+0.05,font=4 );
					}else{
					  text( x=i-0.5,y=j-0.5, labels=sprintf("%d",AB[i,j]),cex=cex.overlap.size,font=1,col="grey50" );
					}
				}
				if( show.overlap.pvalue ){
					text( x=i-0.5,y=j-0.5, labels=sprintf("%.0e",vennPvalue(overlap.test.all, A[i,j],B[i,j],AB[i,j]) ), cex=0.3 );
				}
			}
		}
	}
  }
}

##
draw.vennpie <- function( x=0,y=0,s=c(2,2,1),vpcol=c("blue","red","grey") )
{
	library("shape");   # 'filledcircle': plots (part of) outer and inner circle and colors inbetween;
	tot <- s[1] + s[2] - s[3];   # total of union
	p <- s/tot;     # percentages

	# overlap part
	filledcircle( r1=0.40, r2=0.32, mid=c(x,y), from=0,to=2*pi*p[1], col=vpcol[1] );
	filledcircle( r1=0.30, r2=0.22, mid=c(x,y), from=2*pi*(p[3]-p[2]),to=2*pi*p[3], col=vpcol[2] );
	filledcircle( r1=0.20, r2=0.01, mid=c(x,y), from=0,to=2*pi*p[3], col=vpcol[3] );
}


## calculate the significance of overlap between two sets by hypergeometric distribution
# usage:  N (all), m (size of setA), n (size of setB), k (size of overlap between A and B)
vennPvalue <- function(N, m, n, k)
{
	require(gmp);
    i <- k:min(m,n)
    return( as.numeric( sum(chooseZ(m,i)*chooseZ(N-m,n-i))/chooseZ(N,n) ) );
}

## calculate the significance of overlap between two sets by hypergeometric distribution
# usage:
#   All: N, setA_only: A, setB_only: B, overlap: k
#   vennPvalue( N, A, B, k );

# install.packages("gmp");

#vennPvalue <- function(N, A, B, k)
#{
#	require(gmp);
#    m <- A + k
#    n <- B + k
#    i <- k:min(m,n)
#
#    return( as.numeric( sum(chooseZ(m,i)*chooseZ(N-m,n-i))/chooseZ(N,n) ) );
#}
#
#
