function run_for_fig3()

% Christopher R. Fetsch, Naomi N. Odean, Danique Jeurissen,
% Yasmine El-Shamayleh, Gregory D. Horwitz, and Michael N. Shadlen
% Focal optogenetic suppression in macaque area MT biases direction
% discrimination and decision confidence, but only transiently

% CRF, with code from MNS and former lab members, especially Roozbeh Kiani

addpath('./functions')
load(fullfile('data','data_SU.mat'));

% N=26 single units recorded during passive viewing of random dot motion;
% motion coherence was 99% and dots duration was 265 ms.
% laser was delivered on 50% of trials starting 20 ms after dots onset and
% ending at dots offset.

% d is a struct with the following fields
% laser: presence/absence of laser
% dir: direction (data for each neuron has been rotated so that pref dir is always 180)
    prefDir = 180;
% fr: firing rate (sp/s) calculated from 50 ms after dots onset until dots offset
% cellID: index of neuron (1 through 26)

% trialRaster is a matrix of N trials x 451 1-ms bins, where 1 is a spike and 0 is no spike.
% t is the corresponding time axis aligned to dots onset.


%% Fig 3A: histogram of fractional change in firing rate for 26 single units
uCID = unique(d.cellID);
deltaFR = nan(length(uCID),1);
for c = 1:length(uCID)
    I = d.cellID==c & d.dir==prefDir;
    deltaFR(c) = (nanmean(d.fr(I & d.laser==1)) - nanmean(d.fr(I & d.laser==0))) / nanmean(d.fr(I & d.laser==0));
end

figure; set(gcf, 'Color', [1 1 1], 'Position', [100 200 400 310], 'PaperPositionMode', 'auto');
[N,X] = hist(deltaFR,10);
bar(X,N,'FaceColor','k','EdgeColor','k','LineWidth',0.5);
xlim([-1 1]); ylim([0 12]);
set(gca,'ytick',0:4:12)
% set(gca,'xtick',[-1:0.25:1])
box off;
set(gca,'tickdir','out')
xlabel('Fractional change in firing rate');
ylabel('Number of neurons');
title('3A');
changeAxesFontSize(gca,17,17);


%% Fig 3B: PSTH of suppressed single units

% convolution kernel for PSTH: 40 ms boxcar (acausal/centered),
% but time axis has been shifted to make it causal
convKernel = fspecial('average', [1 40]);

I = d.dir==prefDir & ~ismember(d.cellID,[1 2 5 6 10 16]);
                     % ^limit PSTH to only the 20 cells with suppression (or don't)
[psth,psth_se] = calc_mean(trialRaster(I,:), d.laser(I), [0 1]');
psth = psth*1e3; psth_se = psth_se*1e3; % convert to spikes/s
for r = 1 : size(psth,1)
    psth(r,:) = nanconv(psth(r,:),convKernel,'same');
    psth_se(r,:) = nanconv(psth_se(r,:),convKernel,'same');
end

figure; set(gcf, 'Color', 'w', 'Position', [500 250 400 310], 'PaperPositionMode', 'auto');
plot(t, psth(1,1:length(t)),'k-','Linewidth',2); hold on;
plot(t, psth(2,1:length(t)),'r-','Linewidth',2);
% plot arrows at dots on/off
yl=ylim;
plot([20 20],[0.01 0.04]*yl(2),'r-','Linewidth',2);
plot([20 16],[0.01 0.025]*yl(2),'r-','Linewidth',2);
plot([20 24],[0.01 0.025]*yl(2),'r-','Linewidth',2);
plot([265 265],[0.01 0.04]*yl(2),'r-','Linewidth',2)
plot([265 261],[0.01 0.025]*yl(2),'r-','Linewidth',2);
plot([265 269],[0.01 0.025]*yl(2),'r-','Linewidth',2);

set(gca, 'XLim',[-10 330],'Xtick',0:50:300,'XTickLabel',{'0','','100','','200','','300'}, 'TickDir', 'out'); box off;
xlabel('Time from dots on (ms)'); ylabel('Firing rate (spikes/s)');
h=legend('no laser','laser','Location','Northeast'); legend('boxoff')
set(h,'FontSize',14);
title('3B');
changeAxesFontSize(gca, 17, 17);


%% Fig 3A inset: histogram of (session-wise) fractional change in firing rate based on MUA in 23 behavioral sessions

clear
load(fullfile('data','data_MU.mat'));

% d is a struct with the following fields
% coh: signed motion coherence (positive = neurons' preferred direction)
% strg: presence/absence of sure-bet target (opt-out option)
% duration: viewing duration in ms
% trialNum: trial number within session
% laser: presence/absence of laser (photosuppression)
% choice: 1 = null (antipreferred) direction, 2 = preferred direction, 3 = sure bet (opt-out)
% fr: firing rate (sp/s) calculated from 60 ms after dots onset until 60 ms after dots offset
% deltaFR: fractional change in firing rate due to photosuppression (see text)
% sessionID: index of behavioral+photosuppression session (1 through 23)

% trialRaster and t are as above, but range out past the maximum duration
% for these sessions. NaNs indicate time bins later than 40 ms beyond dots offset.

uSID = unique(d.sessionID);
deltaFR_mean = nan(length(uSID),1);
for s = 1:length(uSID)
    I = d.sessionID==s;
    deltaFR_mean(s) = (nanmean(d.deltaFR(I)));
end

figure; set(gcf, 'Color', [1 1 1], 'Position', [100 200 400 310], 'PaperPositionMode', 'auto');
[N,X] = hist(deltaFR_mean,13);
bar(X,N,'FaceColor','k','EdgeColor','k','LineWidth',0.5);
xlim([-1 0]); ylim([0 6]);
set(gca,'ytick',0:2:6)
box off;
set(gca,'xtick',-1:0.25:0)
set(gca,'tickdir','out')
xlabel('Frac. change in MU response');
ylabel('Number of sites');
title('3A-inset');
changeAxesFontSize(gca,19,19);


%% Fig 3C: avg PSTH for behavioral sessions, aligned to dots-on

convKernel = fspecial('average', [1 40]);
[psth,psth_se] = calc_mean(trialRaster, d.laser, [0 1]');
psth = psth*1e3; psth_se = psth_se*1e3; % convert to spikes/s
for r = 1 : size(psth,1)
    psth(r,:) = nanconv(psth(r,:),convKernel,'same');
    psth_se(r,:) = nanconv(psth_se(r,:),convKernel,'same');
end
endHere = min([find(isnan(psth(1,:)),1,'first') find(isnan(psth(2,:)),1,'first')])-1;
t = t(1:endHere);
YLim = [0 max(max(psth(:,1:length(t))))*1.1];
figure; set(gcf, 'Color', 'w', 'Position', [600 250 400 350], 'PaperPositionMode', 'auto');
[hl, ~] = boundedline(t,psth(1,1:length(t)),psth_se(1,1:length(t)), 'k-', ...
                       t,psth(2,1:length(t)),psth_se(2,1:length(t)),'r','alpha','transparency', 0.15);
set(gca, 'XLim', [-20 500], 'YLim', YLim, 'XTick', 0:100:500, 'YTick', 0:20:120, 'TickDir','out'); box off;
xlabel('Time from dots on (ms)');
ylabel('Firing rate (spikes/s)');
plot([20 20],[0 13],'r-','LineWidth',2) % laser power profile
plot([20 500],[13 13],'r-','LineWidth',2); 

legend(hl, 'no laser','laser','location','Northeast'); legend('boxoff')
title('3C');
changeAxesFontSize(gca, 17, 17);    


%% Fig 3D: avg PSTH for behavioral sessions, aligned to dots-off

load(fullfile('data','trialRaster_dotsOff.mat'));

convKernel = fspecial('average', [1 40]);
I = ismember(d.sessionID,12:23); % sessions with power rampdown
[psth,psth_se] = calc_mean(trialRaster_dotsOff_rampdown, d.laser(I), [0 1]');
psth = psth*1e3; psth_se = psth_se*1e3; 
psth = smoothRaster(psth, convKernel);
psth_se = smoothRaster(psth_se, convKernel);
t = t(~isnan(psth(1,:))); 
YLim = [0 130];
XLim = [-110 210];
figure; set(gcf, 'Color', 'w', 'Position', [500 500 400*(320/520) 350], 'PaperPositionMode', 'auto');
[hl, ~] = boundedline(t,psth(1,1:length(t)),psth_se(1,1:length(t)), 'k-', ...
                       t,psth(2,1:length(t)),psth_se(2,1:length(t)), 'r-','alpha','transparency', 0.15);
% laser power profile
plot([-110 20], [13 13], 'r-','LineWidth',2); 
plot([20 160], [13 0], 'r-','LineWidth',2); % rampdown
set(gca, 'XLim', XLim, 'YLim', YLim, 'Xtick', -100:100:200, 'Ytick', 0:20:120, 'TickDir','out'); box off;
xlabel('Time from dots off (ms)'); ylabel('Spikes/s');
% set(gca, 'YtickLabel',[]);
legend(hl, 'no laser','laser','location','Northwest'); legend('boxoff')
title('3D');
changeAxesFontSize(gca, 15, 15);


% inset (no rampdown)
I = ismember(d.sessionID,1:11);
[psth,psth_se] = calc_mean(trialRaster_dotsOff_noRampdown, d.laser(I), [0 1]');
psth = psth*1e3; psth_se = psth_se*1e3; 
psth = smoothRaster(psth, convKernel);
psth_se = smoothRaster(psth_se, convKernel);
t = t(~isnan(psth(1,:))); 
YLim = [0 130];
XLim = [-110 210];
figure; set(gcf, 'Color', 'w', 'Position', [750 500 400*(320/520) 350], 'PaperPositionMode', 'auto');
[hl, ~] = boundedline(t,psth(1,1:length(t)),psth_se(1,1:length(t)), 'k-', ...
                       t,psth(2,1:length(t)),psth_se(2,1:length(t)), 'r-','alpha','transparency', 0.15);
% laser power profile
plot([-110 20], [13 13], 'r-','LineWidth',2); 
plot([20 20], [13 0], 'r-','LineWidth',2); % no rampdown
set(gca, 'XLim', XLim, 'YLim', YLim, 'Xtick', -100:100:200, 'Ytick', 0:20:120, 'TickDir','out'); box off;
xlabel('Time from dots off (ms)'); ylabel('Spikes/s');
% set(gca, 'YtickLabel',[]);
legend(hl, 'no laser','laser','location','Northwest'); legend('boxoff')
title('3D-inset');
changeAxesFontSize(gca, 15, 15);



%% Fig 3 - Figure Supplement 1: Predicted irradiance based on Monte Carlo simulations

% Uses MonteCarloLight.m and LightHeatPlotter.m from Stujenske, J.M., 
% Spellman, T., Gordon, J.A., 2015. Modeling the Spatiotemporal Dynamics
% of Light and Heat Propagation for In Vivo Optogenetics.
% Cell Reports 12, 525-534. 
%*********
% WARNING: takes several minutes to run!
%*********

clear

warning('Fig 3 - Fig Suppl 1 takes several minutes to run');
reply = input('Run it anyway? Y/N : ','s');
if isempty(reply) || strcmp(reply,'y')
  reply = 'Y';
end

if strcmp(reply,'Y')
    n = 1.36; % refractive index of brain
    theta = 30; % half-angle of beam in air (deg), estimate for tapered fiber
    NA = n*sind(theta); % numerical aperture
    r = 0.05; % radius of optical fiber in mm
    w = 633; % wavelength
    modelnumber = 1;

    [~,frac_trans,~,r,depth] = MonteCarloLight(modelnumber,r,NA,w);

    power = [0.3 0.7 1.2];
    figure; set(gcf, 'Color', [1 1 1], 'Position', [50 200 1520 400], 'PaperPositionMode', 'auto');
    for p = 1:length(power)
        subplot(1,3,p);
        LightHeatPlotter(r,depth,log10(power(p)*frac_trans),'jet');
        xlim([-0.8 0.8]); ylim([-0.2 1]);
        if p==2; xlabel('Radial distance from fiber tip (mm)'); else; xlabel([]); end
        if p==1; ylabel('Depth from fiber tip (mm)'); else; ylabel([]);  end
        title(['power = ' num2str(power(p)) ' mW']);
        cbh=colorbar; caxis([-2 2]);
        set(cbh,'YTick',-2:1:2,'YTickLabel',{'0.01' '0.1' '1' '10' '100'});
        changeAxesFontSize(gca,15,15);
    end
end
% title('3-Supp1');


%% Fig 3 - Figure Supplement 2: Example of titration of laser power

clear
load(fullfile('data','dataVaryPower.mat'));

% MUA from representative subset of mapping (fixation) blocks prior to
% initiating behavioral task. Laser power was manually varied to estimate
% saturation point of photosuppression. For the figure, approx irradiance
% was arbitrarily grouped into three levels (d.irradLevel):
% 1 = 0.1-0.5 mW/mm^2
% 2 = 3-6 mW/mm^2
% 3 = 8-9 mW/mm^2

convKernel = fspecial('average', [1 40]);
prefDir = 180;

% PSTHs
figure; set(gcf, 'Color', 'w', 'Position', [650 250 920 380], 'PaperPositionMode', 'auto');
subplot(1,2,1);
I = d.dir==prefDir & d.laser==0;
psth_nolaser = nanmean(trialRaster(I,:))*1e3;
psth_nolaser = smoothRaster(psth_nolaser, convKernel);
plot(t, psth_nolaser(1,1:length(t)),'k-','LineWidth',2); hold on;
ulvl = unique(d.irradLevel);
clr = cool(length(ulvl));

for p = 1:length(ulvl)
    I = d.dir==prefDir & d.laser==1 & d.irradLevel==ulvl(p);
    psth_laser = nanmean(trialRaster(I,:))*1e3;
    psth_laser = smoothRaster(psth_laser, convKernel);
    plot(t, psth_laser(1:length(t)),'Color', clr(p,:),'LineWidth',2);
end

set(gca, 'XLim', [t(1) 350]); box off;
xlabel('Time from dots-on (ms)'); ylabel('Firing rate (spk/s)');
xlim([-30 300]);
ylim([0 120]);
h = legend('laser off', 'irrad < 0.5 mW/mm^2','irrad = 3-6 mW/mm^2','irrad = 8-9 mW/mm^2','location','Northwest');
legend('boxoff');
title('3-Supp2A');
changeAxesFontSize(gca, 15, 15);
set(h,'Position',[0.1370 0.7276 0.2087 0.2645]);

% tuning curves
subplot(1,2,2);
I = d.laser==0;
dir_set = unique(d.dir);
[meanFR, se] = calc_mean(d.fr(I),d.dir(I),dir_set);
meanFR(end+1) = meanFR(1);
se(end+1) = se(1);
errorbar([dir_set;360], meanFR, se, 'ko-','LineWidth',2); hold on;

for p = 1:length(ulvl)
    I = d.laser==1 & d.irradLevel==ulvl(p);
    meanFR = calc_mean(d.fr(I),d.dir(I),dir_set);
%         % temp kluge for 0/180 sessions (interpolate):
%         meanFR(isnan(meanFR))=nanmean(meanFR);
    meanFR(end+1) = meanFR(1);
    plot([dir_set;360], meanFR,'Color',clr(p,:),'LineWidth',2);
end

set(gca, 'XLim', [0 360],'XTick',0:90:360,'TickDir','out'); box off;
yl=ylim; ylim([0 yl(2)]);
xlabel('Direction (deg)'); ylabel('Firing rate (spk/s)');
title('3-Supp2B');
changeAxesFontSize(gca, 15, 15);






