function run_for_fig4()

% Christopher R. Fetsch, Naomi N. Odean, Danique Jeurissen,
% Yasmine El-Shamayleh, Gregory D. Horwitz, and Michael N. Shadlen
% Focal optogenetic suppression in macaque area MT biases direction
% discrimination and decision confidence, but only transiently

% CRF, with code from MNS and former lab members, especially Roozbeh Kiani

addpath('./functions')
load(fullfile('data','data.mat'));

% d is a struct with the following fields
% coh: signed motion coherence (positive = neurons' preferred direction)
% strg: presence/absence of sure-bet target (opt-out option)
% duration: viewing duration in ms
% trialNum: trial number within session
% laser: presence/absence of laser (photosuppression)
% choice: 1 = null (antipreferred) direction, 2 = preferred direction, 3 = sure bet (opt-out)
% fr: firing rate (sp/s) calculated from 60 ms after dots onset until 60 ms after dots offset
% deltaFR: fractional change in firing rate due to photosuppression (see text)
% sessionID: index of behavioral+photosuppression sessions (1 through 23)


%% Fig 4A: histogram of trial-wise fractional change in firing rate (deltaFR) 

figure; set(gcf, 'Color', [1 1 1], 'Position', [100 200 400 250], 'PaperPositionMode', 'auto');
ctrs = -0.95:0.1:1.55;
[N,X] = hist(d.deltaFR,ctrs);
bar(X,N,'FaceColor','w','EdgeColor','k','LineWidth',1);
xlim([-1 1]);
xlabel('Fractional change in firing rate');
ylabel('Trial count');
set(gca,'box','off','Tickdir','out');
title('4A');
changeAxesFontSize(gca,15,15);


%% Fig 4B: behavioral effects as a function of deltaFR (sliding window)

windowSize = 2400; % in number of laser trials
windowStep = 50; % change to 10 for paper figure (slow)
minWindowSize = 2000;
selectTrials = d.sessionID ~= 18; % exclude one 'paradoxical site' (see Methods)
dFR = d.deltaFR;

% label excluded trials and no-laser trials in a sortable way
dFR(~selectTrials) = -1000;
dFR(isnan(dFR)) = -999;
[deltaFR_sorted,deltaFR_sorted_ind] = sort(dFR);
endOfDiscardedTrials = find(deltaFR_sorted==-999,1,'last');
nTrials = sum(dFR>-999);

% generate sliding windows
startOfWindow = 1;
endOfWindow = windowSize;
w=0;
while endOfWindow < nTrials
    w = w + 1;
    windows{w} = startOfWindow:endOfWindow;
    startOfWindow = startOfWindow + windowStep;
    endOfWindow = endOfWindow + windowStep;
end
deltaFR_set = cell(1,length(windows));
deltaFR_ind_set = cell(1,length(windows));
plotW = nan(size(deltaFR_set)); % bin center (avg dFR for a given window) for plotting 

% find trial indices corresponding to deltaFR in each window
for w = 1:length(windows)
    deltaFR_ind_set{w} = deltaFR_sorted_ind(windows{w}+endOfDiscardedTrials);
    deltaFR_set{w} = [min(dFR(deltaFR_ind_set{w})) max(dFR(deltaFR_ind_set{w}))];
    plotW(w) = nanmean(dFR(deltaFR_ind_set{w}));
end
lastWindowWouldBe = windows{end}(1)+windowStep:nTrials;
if length(lastWindowWouldBe) > minWindowSize
    w=w+1;
    windows{w} = lastWindowWouldBe;
    deltaFR_ind_set{w} = deltaFR_sorted_ind(windows{w}+endOfDiscardedTrials);
    deltaFR_set{w} = [min(dFR(deltaFR_ind_set{w})) max(dFR(deltaFR_ind_set{w}))];
    plotW(w) = nanmean(dFR(deltaFR_ind_set{w}));
end

% kluge to get symmetric bins at the edges
deltaBinSize = floor((length(windows{1})-length(windows{end}))/2);
windows{1} = windows{1}(1:end-deltaBinSize);
windows{end} = windows{end}(1)-(deltaBinSize+1):windows{end}(end);
for W = [1 length(windows)]
    deltaFR_ind_set{W} = deltaFR_sorted_ind(windows{W}+endOfDiscardedTrials);
    deltaFR_set{W} = [min(dFR(deltaFR_ind_set{W})) max(dFR(deltaFR_ind_set{W}))];
    plotW(W) = nanmean(dFR(deltaFR_ind_set{W}));
end

%
% fit Eqs 1 and 3 for each sliding window
%
var = nan(2,length(deltaFR_set));
err = nan(2,length(deltaFR_set));

for p = 1:length(deltaFR_set)
      
    if mod(p,20)==0
        sprintf('Progress: %d out of %d',p,length(deltaFR_set))
    end

    thisI = selectTrials; % reinitialize this each time
    tempind = (1:length(d.deltaFR))';
    deltaFR_set_found = ismember(tempind, deltaFR_ind_set{p}) | dFR == -999;
        % note each window contains all no-laser trials, originally NaN but relabeled -999 for sorting
    thisI = thisI & deltaFR_set_found;

    coh_ = d.coh(thisI);
    strg_ = d.strg(thisI);
    laser_ = d.laser(thisI);
    choice_ = d.choice(thisI);

    prefTarg = 2; % by convention, choice=2 corresponds to the pref dir

    % choice (logistic, Eq. 1 in text)
    I = ismember(choice_,[1 2]);
    X = [coh_(I) laser_(I) coh_(I).*laser_(I)];
    y = choice_(I)==prefTarg;
    [beta_glm, ~, stats] = glmfit(X, y, 'binomial');
    choice_shift = beta_glm(3)/beta_glm(2) * 100;
    choice_shift_se = stats.se(3)/beta_glm(2) * 100;

    % confidence (Gaussian, Eq. 3 in text)
    psure_fcn_b4 = @(b,coh,IL) min(max(b(1),0),1).*exp(-(coh+b(3)+b(4)*IL).^2 ./ (2*b(2).^2));
        % error to minimize is negative log likelihood of observing the data, which is: -[ log(Psurebet) + log(1-(~Psurebet)) ]
    err_fcn_b4 = @(param,choice,coh,IL) -(sum(log(psure_fcn_b4(param,coh(choice),IL(choice))))+sum(log(1-psure_fcn_b4(param,coh(~choice),IL(~choice)))));

    J = strg_==1; % trials with sure-bet target available
    x0 = [0.6 0.25 -.003 .02];
    opts = optimset('MaxFunEvals', 1e6, 'Algorithm','interior-point','LargeScale','off','Display','off');
    [Psure_beta,~,~,~,~,hessian] = fminunc(@(x) err_fcn_b4(x,choice_(J)==3,coh_(J),laser_(J)==1), x0, opts);
    se = sqrt(diag(inv(hessian)));
    Psure_shift = Psure_beta(4)*100;
    Psure_shift_se = se(4)*100;

    var(1,p) = choice_shift;
    var(2,p) = Psure_shift;

    err(1,p) = choice_shift_se;
    err(2,p) = Psure_shift_se;

end

%
% plot it
%
figure; set(gcf, 'Color', [1 1 1], 'Position', [100 100 600 400], 'PaperPositionMode', 'auto');
[~, hp] = boundedline(plotW,var(1,:),err(1,:), 'r-', ...
           plotW,var(2,:),err(2,:),'b-','alpha','transparency', 0.15);
xlabel('Fractional change in firing rate');
ylabel('Behavioral effect (%coh)');
xlim([round((plotW(1)-0.05)*10)/10 round((plotW(end)+0.05)*10)/10])
set(gca,'Xtick',-0.7:0.1:0.1, 'Ytick',-8:4:8);
xlim([-0.71 0.11]); ylim([-8 8]);
xl = xlim;
plot([xl(1) xl(2)],[0 0],'k--');
H = legend(hp,'choice','confidence','Location','Northwest');
set(H,'FontSize',20)
legend('boxoff');     
set(gca,'TickDir','out','box','off');
title('4B');
changeAxesFontSize(gca,17,17);


%% Fig 4C: choice effect for deltaFR < -0.45

selectTrials = selectTrials & (d.deltaFR < -0.45 | isnan(d.deltaFR)); % include NaNs, i.e. all laser-off trials

% calculate frequencies of pref-dir and sure-bet choices
coh_set = unique(d.coh);
nPref_all = nan(length(coh_set), 2); % coh x nolaser/laser
pPref_all = nan(length(coh_set), 2);
pSureTarg = nan(length(coh_set), 2);
nSureTarg = nan(length(coh_set), 2); 

J = d.strg; % trials with sure-bet target available
prefTarg = 2; % by convention, choice=2 corresponds to the pref dir

for c = 1 : length(coh_set)              
    I = selectTrials & d.coh==coh_set(c);
    for l = 0 : 1
        L = selectTrials & d.laser==l;

        nPref_all(c,l+1) = sum(I & L & ismember(d.choice,[1 2]));
        pPref_all(c,l+1) = sum(I & L & d.choice==prefTarg) / nPref_all(c,l+1);

        nSureTarg(c,l+1) = sum(I & J & L);
        pSureTarg(c,l+1) = sum(I & J & L & d.choice==3) / nSureTarg(c,l+1);
        
    end
end
% standard errors
pPref_all_se = sqrt(pPref_all.*(1-pPref_all)./nPref_all);
pSureTarg_se = sqrt(pSureTarg.*(1-pSureTarg)./nSureTarg);

g_coh = (-0.52:0.01:0.51)'; g_coh(1) = coh_set(1); g_coh(end) = coh_set(end);

% fit separately for laser and no-laser
I = selectTrials & ismember(d.choice,[1 2]) & d.laser==0; % includes trials with and without strg
X = d.coh(I);
y = d.choice(I)==prefTarg;
[beta_nolaser, ~, ~] = glmfit(X, y, 'binomial');
g_pPref_nolaser = logistic(beta_nolaser(1) + beta_nolaser(2)*g_coh);

I = selectTrials & ismember(d.choice,[1 2]) & d.laser==1; % includes trials with and without strg 
X = d.coh(I);
y = d.choice(I)==prefTarg;
[beta_wlaser, ~, ~] = glmfit(X, y, 'binomial');
g_pPref_wlaser = logistic(beta_wlaser(1) + beta_wlaser(2)*g_coh);


% plot it
figure; set(gcf, 'Color', [1 1 1], 'Position', [100 500 400 350], 'PaperPositionMode', 'auto'); hold on;
plot(g_coh*100, g_pPref_nolaser(:,1), 'k-', 'LineWidth', 2);
plot(g_coh*100, g_pPref_wlaser(:,1), 'r-', 'LineWidth', 2);
[hsym1,~,hye] = errorbar2(coh_set*100, pPref_all(:,1), 0, pPref_all_se(:,1), 'o');
set(hsym1,'Color','k','MarkerSize',11,'MarkerFaceColor','k','MarkerEdgeColor','k'); set(hye,'LineWidth',1,'Color','k');
[hsym2,~,hye] = errorbar2(coh_set*100, pPref_all(:,2), 0, pPref_all_se(:,2), 'o');
set(hsym2,'Color','r','MarkerSize',11,'MarkerFaceColor','r','MarkerEdgeColor','r'); set(hye,'LineWidth',1,'Color','r');
set(gca, 'XLim', [-53 53], 'XTick', -50:25:50, 'XTickLabel', makeTickLabel(-50:25:50,25), ...
         'YLim',[-0.02 1.02], 'YTick', 0:0.1:1, 'YTickLabel', makeTickLabel(0:0.1:1,0.2), ...
         'TickDir', 'out');
ylabel('Proportion preferred choices'); xlabel('Motion strength (% coh)');
h=legend([hsym1 hsym2],'no laser','laser','Location', 'Northwest');
set(h,'FontSize',14); legend('boxoff');
title('4C');
changeAxesFontSize(gca, 16, 16);


%% Fig 4D: confidence effect for deltaFR < -0.45

psure_fcn_b3 = @(b,coh) min(max(b(1),0),1).*exp(-(coh+b(3)).^2 ./ (2*b(2).^2));
    % error to minimize is negative log likelihood of observing the data, which is: -[ log(Psurebet) + log(1-(~Psurebet)) ]
err_fcn_b3 = @(param,sbchoice,coh) -(sum(log(psure_fcn_b3(param,coh(sbchoice))))+sum(log(1-psure_fcn_b3(param,coh(~sbchoice)))));
opts = optimset('MaxFunEvals', 1e6, 'Algorithm','interior-point','LargeScale','off','Display','off');

% fit separately for laser and no-laser
I = selectTrials & d.strg==1 & d.laser==0;
[Psure_beta_nolaser,~,~,~,~,~] = fminunc(@(x) err_fcn_b3(x,d.choice(I)==3,d.coh(I)), [0.6 0.25 0], opts);
I = selectTrials & d.strg==1 & d.laser==1;
[Psure_beta_wlaser,~,~,~,~,~] = fminunc(@(x) err_fcn_b3(x,d.choice(I)==3,d.coh(I)), [0.6 0.25 0], opts);

% plot it
figure; set(gcf, 'Color', [1 1 1], 'Position', [100 100 400 350], 'PaperPositionMode', 'auto'); hold on;
plot(g_coh*100, psure_fcn_b3(Psure_beta_nolaser, g_coh), 'k-', 'LineWidth', 2);
plot(g_coh*100, psure_fcn_b3(Psure_beta_wlaser, g_coh), 'r-', 'LineWidth', 2);
[hsym1,~,hye] = errorbar2(coh_set*100, pSureTarg(:,1), 0, pSureTarg_se(:,1), 'o');
set(hsym1,'MarkerSize',11,'MarkerFaceColor','k','MarkerEdgeColor','k'); set(hye,'LineWidth',1,'Color','k');
[hsym2,~,hye] = errorbar2(coh_set*100, pSureTarg(:,2), 0, pSureTarg_se(:,2), 'o');
set(hsym2,'MarkerSize',11,'MarkerFaceColor','r','MarkerEdgeColor','r'); set(hye,'LineWidth',1,'Color','r');

set(gca, 'XLim', [-53 53], 'XTick', -50:25:50, 'XTickLabel', makeTickLabel(-50:25:50,25), ...
         'YLim',[0 1], 'YTick', 0:0.1:1, 'YTickLabel', makeTickLabel(0:0.1:1,0.2), ...
         'TickDir', 'out');
xlabel('Motion strength (% coh)');
ylabel('Proportion sure-bet choices');
title('4D');
changeAxesFontSize(gca, 16, 16);


%% Fig 4 - Figure Supplement 1: Effects of electrical Stim on choice and confidence (monkey D, Fetsch et al. 2014)

% A

clear
load(fullfile('data','data_ustim_monkeyD.mat'));

% calculate frequencies of pref-dir and sure-bet choices
coh_set = unique(d.coh);
nPref_all = nan(length(coh_set), 2); % coh x nolaser/laser
pPref_all = nan(length(coh_set), 2);
pSureTarg = nan(length(coh_set), 2);
nSureTarg = nan(length(coh_set), 2); 

J = d.strg; % trials with sure-bet target available
prefTarg = 2; % by convention, choice=2 corresponds to the pref dir

for c = 1 : length(coh_set)              
    I = d.coh==coh_set(c);
    for l = 0 : 1
        L = d.ustim==l;

        nPref_all(c,l+1) = sum(I & L & ismember(d.choice,[1 2]));
        pPref_all(c,l+1) = sum(I & L & d.choice==prefTarg) / nPref_all(c,l+1);

        nSureTarg(c,l+1) = sum(I & J & L);
        pSureTarg(c,l+1) = sum(I & J & L & d.choice==3) / nSureTarg(c,l+1);
        
    end
end
% standard errors
pPref_all_se = sqrt(pPref_all.*(1-pPref_all)./nPref_all);
pSureTarg_se = sqrt(pSureTarg.*(1-pSureTarg)./nSureTarg);

g_coh = (-0.52:0.01:0.51)'; g_coh(1) = coh_set(1); g_coh(end) = coh_set(end);

% fit separately for stim and no-stim
I = ismember(d.choice,[1 2]) & d.ustim==0; % includes trials with and without strg
X = d.coh(I);
y = d.choice(I)==prefTarg;
[beta_nostim, ~, ~] = glmfit(X, y, 'binomial');
g_pPref_nostim = logistic(beta_nostim(1) + beta_nostim(2)*g_coh);

I = ismember(d.choice,[1 2]) & d.ustim==1; % includes trials with and without strg 
X = d.coh(I);
y = d.choice(I)==prefTarg;
[beta_wstim, ~, ~] = glmfit(X, y, 'binomial');
g_pPref_wstim = logistic(beta_wstim(1) + beta_wstim(2)*g_coh);


% plot it
figure; set(gcf, 'Color', [1 1 1], 'Position', [100 500 400 350], 'PaperPositionMode', 'auto'); hold on;
plot(g_coh*100, g_pPref_nostim(:,1), 'k-', 'LineWidth', 2);
plot(g_coh*100, g_pPref_wstim(:,1), 'g-', 'LineWidth', 2);
[hsym1,~,hye] = errorbar2(coh_set*100, pPref_all(:,1), 0, pPref_all_se(:,1), 'o');
set(hsym1,'Color','k','MarkerSize',11,'MarkerFaceColor','k','MarkerEdgeColor','k'); set(hye,'LineWidth',1,'Color','k');
[hsym2,~,hye] = errorbar2(coh_set*100, pPref_all(:,2), 0, pPref_all_se(:,2), 'o');
set(hsym2,'Color','g','MarkerSize',11,'MarkerFaceColor','g','MarkerEdgeColor','g'); set(hye,'LineWidth',1,'Color','g');
set(gca, 'XLim', [-53 53], 'XTick', -50:25:50, 'XTickLabel', makeTickLabel(-50:25:50,25), ...
         'YLim',[-0.02 1.02], 'YTick', 0:0.1:1, 'YTickLabel', makeTickLabel(0:0.1:1,0.2), ...
         'TickDir', 'out');
ylabel('Proportion preferred choices'); xlabel('Motion strength (% coh)');
h=legend([hsym1 hsym2],'no stim','stim','Location', 'Northwest');
set(h,'FontSize',14); legend('boxoff');
title('4-Supp1A');
changeAxesFontSize(gca, 16, 16);


% B

psure_fcn_b3 = @(b,coh) min(max(b(1),0),1).*exp(-(coh+b(3)).^2 ./ (2*b(2).^2));
    % error to minimize is negative log likelihood of observing the data, which is: -[ log(Psurebet) + log(1-(~Psurebet)) ]
err_fcn_b3 = @(param,sbchoice,coh) -(sum(log(psure_fcn_b3(param,coh(sbchoice))))+sum(log(1-psure_fcn_b3(param,coh(~sbchoice)))));
opts = optimset('MaxFunEvals', 1e6, 'Algorithm','interior-point','LargeScale','off','Display','off');

% fit separately for stim and no-stim
I = d.strg==1 & d.ustim==0;
[Psure_beta_nostim,~,~,~,~,~] = fminunc(@(x) err_fcn_b3(x,d.choice(I)==3,d.coh(I)), [0.6 0.25 0], opts);
I = d.strg==1 & d.ustim==1;
[Psure_beta_wstim,~,~,~,~,~] = fminunc(@(x) err_fcn_b3(x,d.choice(I)==3,d.coh(I)), [0.6 0.25 0], opts);

% plot it
figure; set(gcf, 'Color', [1 1 1], 'Position', [100 100 400 350], 'PaperPositionMode', 'auto'); hold on;
plot(g_coh*100, psure_fcn_b3(Psure_beta_nostim, g_coh), 'k-', 'LineWidth', 2);
plot(g_coh*100, psure_fcn_b3(Psure_beta_wstim, g_coh), 'g-', 'LineWidth', 2);
[hsym1,~,hye] = errorbar2(coh_set*100, pSureTarg(:,1), 0, pSureTarg_se(:,1), 'o');
set(hsym1,'MarkerSize',11,'MarkerFaceColor','k','MarkerEdgeColor','k'); set(hye,'LineWidth',1,'Color','k');
[hsym2,~,hye] = errorbar2(coh_set*100, pSureTarg(:,2), 0, pSureTarg_se(:,2), 'o');
set(hsym2,'MarkerSize',11,'MarkerFaceColor','g','MarkerEdgeColor','g'); set(hye,'LineWidth',1,'Color','g');

set(gca, 'XLim', [-53 53], 'XTick', -50:25:50, 'XTickLabel', makeTickLabel(-50:25:50,25), ...
         'YLim',[0 1], 'YTick', 0:0.1:1, 'YTickLabel', makeTickLabel(0:0.1:1,0.2), ...
         'TickDir', 'out');
xlabel('Motion strength (% coh)');
ylabel('Proportion sure-bet choices');
title('4-Supp1B');
changeAxesFontSize(gca, 16, 16);


%% Fig 4 - Figure Supplement 2: Sham-laser control analysis

clear
load(fullfile('data','data_shamLaser.mat'));

% 50% of no-laser trials were randomly labeled "sham" laser trials,
% sorted by their difference from the corresponding mean (as in Fig 4B),
% and behavioral "effects" were calculated for each of 72 sliding windows.
% This was repeated 100 times with different random sham assignments.

% plotW: vector of window centers (avg fractional "change" in firing rate)
% choice: each row is one of the 100 repeats, each column is the choice
% 'shift' at the corresponding window in plotW
% choiceErr: standard errors of the above, for each window and repeat
% conf, confErr: same as above but for confidence shifts

figure; set(gcf, 'Color', [1 1 1], 'Position', [100 100 600 400], 'PaperPositionMode', 'auto');
[~, hp] = boundedline(plotW, mean(choice), mean(choiceErr), 'r-', ...
           plotW, mean(conf), mean(confErr),'b-','alpha','transparency', 0.15);
xlabel('Fractional change in firing rate');
xlim([-0.35 0.35])
ylim([-8 8]);
xl = xlim;
plot([xl(1) xl(2)],[0 0],'k--');
set(gca,'Xtick',-0.4:0.1:0.4, 'Ytick',-8:4:8);
H = legend(hp,'choice','confidence','Location','Northwest');
% set(H,'FontSize',20)
legend('boxoff');     
ylabel('Behavioral effect (%coh)');
%     title(['Viewing duration: ' num2str(max([dur_setQ{P}(1) min(dur)])) ' to ' num2str(min([max(dur) dur_setQ{P}(2)])) ' ms']);
set(gca,'TickDir','out','box','off');
title('4-Supp2B');
changeAxesFontSize(gca,17,17); 


%% Fig 4 - Figure Supplement 3: Behavioral effects on individual sessions

clear
load(fullfile('data','data_indiv.mat'));

% choice_shift, Psure_shift, and their SEs calculated as above, separately
% for the 23 individual sessions
    
figure; set(gcf, 'Color', [1 1 1], 'Position', [400 400 500 500], 'PaperPositionMode', 'auto');
[hsym,~,~] = errorbar2(choice_shift, Psure_shift, choice_shift_se, Psure_shift_se,'o',1); hold on;
set(hsym,'MarkerFaceColor','w','MarkerSize',11,'Color','k');
xl = max(abs(xlim)); yl = max(abs(ylim));
plot([-xl xl],[0 0],'k--'); plot([0 0],[-yl yl],'k--'); 
xlim([-10 10]); ylim([-20 20]);
xlabel('Choice effect (%coh)');
ylabel('Confidence effect (%coh)');
title('4-Supp3');
changeAxesFontSize(gca,17,17);





