function run_for_fig7()

% Christopher R. Fetsch, Naomi N. Odean, Danique Jeurissen,
% Yasmine El-Shamayleh, Gregory D. Horwitz, and Michael N. Shadlen
% Focal optogenetic suppression in macaque area MT biases direction
% discrimination and decision confidence, but only transiently

% CRF, with code from MNS and former lab members, especially Roozbeh Kiani

addpath('./functions')
load(fullfile('data','data.mat'));

% d is a struct with the following fields
% coh: signed motion coherence (positive = neurons' preferred direction)
% strg: presence/absence of sure-bet target (opt-out option)
% duration: viewing duration in ms
% trialNum*: trial number within session
% laser: presence/absence of laser (photosuppression)
% choice: 1 = null (antipreferred) direction, 2 = preferred direction, 3 = sure bet (opt-out)
% deltaFR: fractional change in firing rate due to photosuppression (see text)
% sessionID: index of behavioral+photosuppression sessions (1 through 23)

% *(In our setup, the trial number variable incremented even after an
% aborted trial (e.g. fixation break), and thus is slightly greater than
% the number of completed/valid trials. We made no assumptions about whether
% the most relevant variable is completed trials, all trials, or elapsed
% time, but typically treated this variable as a proxy for all three.)

% trialRaster is a matrix of N trials x 1101 1-ms bins, where 1 is a spike and 0 is no spike.
% t is the corresponding time axis aligned to dots onset.
% NaNs in trialRaster indicate time bins later than 40 ms beyond dots offset.


    
%% Fig 7A-D: behavioral effects on short- vs. long-duration trials

selectTrials = d.trialNum <= 540 & (d.deltaFR < -0.25 | isnan(d.deltaFR));
                          %^ 540 approximates 500 completed trials (see above*)

title1 = {'A','C','E','G','B','D','F','H'};
T=1;
for F = 1:2 % short/long

    if F==1
        sel = selectTrials & d.duration <= 350; % 540 approximates 500 completed trials (see above*)
    else
        sel = selectTrials & d.duration > 350;
    end
    
    coh_ = d.coh(sel);
    strg_ = d.strg(sel);
    laser_ = d.laser(sel);
    choice_ = d.choice(sel);

    % calculate frequencies of pref-dir and sure-bet choices
    coh_set = unique(coh_);
    nPref_all = nan(length(coh_set), 2); % coh x nolaser/laser
    pPref_all = nan(length(coh_set), 2);
    pSureTarg = nan(length(coh_set), 2);
    nSureTarg = nan(length(coh_set), 2); 

    J = strg_; % trials with sure-bet target available
    prefTarg = 2; % by convention, choice=2 corresponds to the pref dir

    for c = 1 : length(coh_set)              
        I = coh_==coh_set(c);
        for l = 0 : 1
            L = laser_==l;

            nPref_all(c,l+1) = sum(I & L & ismember(choice_,[1 2]));
            pPref_all(c,l+1) = sum(I & L & choice_==prefTarg) / nPref_all(c,l+1);

            nSureTarg(c,l+1) = sum(I & J & L);
            pSureTarg(c,l+1) = sum(I & J & L & choice_==3) / nSureTarg(c,l+1);

        end
    end
    % standard errors
    pPref_all_se = sqrt(pPref_all.*(1-pPref_all)./nPref_all);
    pSureTarg_se = sqrt(pSureTarg.*(1-pSureTarg)./nSureTarg);

    g_coh = (-0.52:0.01:0.51)'; g_coh(1) = coh_set(1); g_coh(end) = coh_set(end);

    %
    % A/B
    %
    
    % fit separately for laser and no-laser
    I = ismember(choice_,[1 2]) & laser_==0; % includes trials with and without strg
    X = coh_(I);
    y = choice_(I)==prefTarg;
    [beta_nolaser, ~, ~] = glmfit(X, y, 'binomial');
    g_pPref_nolaser = logistic(beta_nolaser(1) + beta_nolaser(2)*g_coh);

    I = ismember(choice_,[1 2]) & laser_==1; % includes trials with and without strg 
    X = coh_(I);
    y = choice_(I)==prefTarg;
    [beta_wlaser, ~, ~] = glmfit(X, y, 'binomial');
    g_pPref_wlaser = logistic(beta_wlaser(1) + beta_wlaser(2)*g_coh);


    % plot it
    figure; set(gcf, 'Color', [1 1 1], 'Position', [100+(F-1)*300 500 400 350], 'PaperPositionMode', 'auto'); hold on;
    plot(g_coh*100, g_pPref_nolaser(:,1), 'k-', 'LineWidth', 2);
    plot(g_coh*100, g_pPref_wlaser(:,1), 'r-', 'LineWidth', 2);
    [hsym1,~,hye] = errorbar2(coh_set*100, pPref_all(:,1), 0, pPref_all_se(:,1), 'o');
    set(hsym1,'Color','k','MarkerSize',11,'MarkerFaceColor','k','MarkerEdgeColor','k'); set(hye,'LineWidth',1,'Color','k');
    [hsym2,~,hye] = errorbar2(coh_set*100, pPref_all(:,2), 0, pPref_all_se(:,2), 'o');
    set(hsym2,'Color','r','MarkerSize',11,'MarkerFaceColor','r','MarkerEdgeColor','r'); set(hye,'LineWidth',1,'Color','r');
    set(gca, 'XLim', [-53 53], 'XTick', -50:25:50, 'XTickLabel', makeTickLabel(-50:25:50,25), ...
             'YLim',[-0.02 1.02], 'YTick', 0:0.1:1, 'YTickLabel', makeTickLabel(0:0.1:1,0.2), ...
             'TickDir', 'out');
    ylabel('Proportion preferred choices'); xlabel('Motion strength (% coh)');
    h=legend([hsym1 hsym2],'no laser','laser','Location', 'Northwest');
    set(h,'FontSize',14); legend('boxoff');
    title(['7' title1{T}]); T=T+1;
    changeAxesFontSize(gca, 16, 16);

    %
    % C/D
    %

    psure_fcn_b3 = @(b,coh) min(max(b(1),0),1).*exp(-(coh+b(3)).^2 ./ (2*b(2).^2));
        % error to minimize is negative log likelihood of observing the data, which is: -[ log(Psurebet) + log(1-(~Psurebet)) ]
    err_fcn_b3 = @(param,sbchoice,coh) -(sum(log(psure_fcn_b3(param,coh(sbchoice))))+sum(log(1-psure_fcn_b3(param,coh(~sbchoice)))));
    opts = optimset('MaxFunEvals', 1e6, 'Algorithm','interior-point','LargeScale','off','Display','off');

    % fit separately for laser and no-laser
    I = strg_==1 & laser_==0;
    [Psure_beta_nolaser,~,~,~,~,~] = fminunc(@(x) err_fcn_b3(x,choice_(I)==3,coh_(I)), [0.6 0.25 0], opts);
    I = strg_==1 & laser_==1;
    [Psure_beta_wlaser,~,~,~,~,~] = fminunc(@(x) err_fcn_b3(x,choice_(I)==3,coh_(I)), [0.6 0.25 0], opts);

    % plot it
    figure; set(gcf, 'Color', [1 1 1], 'Position', [100+(F-1)*300 100 400 350], 'PaperPositionMode', 'auto'); hold on;
    plot(g_coh*100, psure_fcn_b3(Psure_beta_nolaser, g_coh), 'k-', 'LineWidth', 2);
    plot(g_coh*100, psure_fcn_b3(Psure_beta_wlaser, g_coh), 'r-', 'LineWidth', 2);
    [hsym1,~,hye] = errorbar2(coh_set*100, pSureTarg(:,1), 0, pSureTarg_se(:,1), 'o');
    set(hsym1,'MarkerSize',11,'MarkerFaceColor','k','MarkerEdgeColor','k'); set(hye,'LineWidth',1,'Color','k');
    [hsym2,~,hye] = errorbar2(coh_set*100, pSureTarg(:,2), 0, pSureTarg_se(:,2), 'o');
    set(hsym2,'MarkerSize',11,'MarkerFaceColor','r','MarkerEdgeColor','r'); set(hye,'LineWidth',1,'Color','r');

    set(gca, 'XLim', [-53 53], 'XTick', -50:25:50, 'XTickLabel', makeTickLabel(-50:25:50,25), ...
             'YLim',[0 1], 'YTick', 0:0.1:1, 'YTickLabel', makeTickLabel(0:0.1:1,0.2), ...
             'TickDir', 'out');
    xlabel('Motion strength (% coh)');
    ylabel('Proportion sure-bet choices');
    title(['7' title1{T}]); T=T+1;
    changeAxesFontSize(gca, 16, 16);
    
    
% Fig 5E-H: neural suppression and selectivity on short- vs. long-duration trials

    %
    % E/F
    %

    % the neural data does not exclude dFR>-0.25 or later trials
    if F==1
        sel = d.duration <= 350; % 540 approximates 500 completed trials (see above*)
        xmax = 320;
    else
        sel = d.duration > 350;
        xmax = 600;
    end
    
    convKernel = fspecial('average', [1 40]);
    [psth,psth_se] = calc_mean(trialRaster(sel,:), d.laser(sel), [0 1]');
    psth = psth*1e3; psth_se = psth_se*1e3; % convert to spikes/s
    for r = 1 : size(psth,1)
        psth(r,:) = nanconv(psth(r,:),convKernel,'same');
        psth_se(r,:) = nanconv(psth_se(r,:),convKernel,'same');
    end
    endHere = min([find(isnan(psth(1,:)),1,'first') find(isnan(psth(2,:)),1,'first')])-1;
    tt = t(1:endHere);
    figure; set(gcf, 'Color', [1 1 1], 'Position', [800+(F-1)*300 500 400 350], 'PaperPositionMode', 'auto'); hold on;
    [hl, ~] = boundedline(tt,psth(1,1:length(tt)),psth_se(1,1:length(tt)), 'k-', ...
                          tt,psth(2,1:length(tt)),psth_se(2,1:length(tt)),'r','alpha','transparency', 0.15);
    set(gca, 'XLim', [-20 xmax], 'YLim', [0 150], 'XTick', 0:100:xmax, 'YTick', 0:50:150, 'TickDir','out'); box off;
    xlabel('Time from dots on (ms)');
    ylabel('Firing rate (spikes/s)');
    legend(hl, 'no laser','laser','location','Northeast'); legend('boxoff')
    title(['7' title1{T}]); T=T+1;
    changeAxesFontSize(gca, 17, 17);    
    
 
    %
    % G/H
    %
    
    sel = sel & d.laser==0;
    psth = calc_mean(trialRaster(sel,:), d.coh(sel), coh_set)*1e3;
    psth = smoothRaster(psth, convKernel);
    clr = cool(length(coh_set));
    figure; set(gcf, 'Color', [1 1 1], 'Position', [800+(F-1)*300 100 400 350], 'PaperPositionMode', 'auto'); hold on;
    hold on;
    for c = 1:length(coh_set)
        h(c) = plot(tt, psth(c,1:length(tt)),'Color', clr(c,:));
    end
    xlim([0 xmax]); ylim([0 200]);
    set(gca, 'TickDir','out','Xtick',0:100:xmax,'Ytick',0:50:200); box off;
    xlabel('Time from dots on (ms)');
    ylabel('Firing rate (spikes/s)');
    
    if F==2
        [H,hObj] = legend([h(11) h(9) h(6) h(3) h(1)],' 51.2', ' 12.8','   0','-12.8','-51.2','Location','Northeast'); legend('boxoff');
        set(H,'FontSize',16)
        hL=findobj(hObj,'type','Text');
        set(hL,'FontSize',16);
        hL=findobj(hObj,'type','Line');
        set(hL,'linewidth',2)
        text(425,195,'%coh','FontSize',16);
    end
    title(['7' title1{T}]); T=T+1;
    changeAxesFontSize(gca, 17, 17);
    
end