function run_for_fig1()

% Christopher R. Fetsch, Naomi N. Odean, Danique Jeurissen,
% Yasmine El-Shamayleh, Gregory D. Horwitz, and Michael N. Shadlen
% Focal optogenetic suppression in macaque area MT biases direction
% discrimination and decision confidence, but only transiently

% CRF, with code from MNS and former lab members, especially Roozbeh Kiani

addpath('./functions')
load(fullfile('data','data.mat'));

% d is a struct with the following fields
% coh: signed motion coherence (positive = neurons' preferred direction)
% strg: presence/absence of sure-bet target (opt-out option)
% duration: viewing duration in ms
% trialNum*: trial number within session
% laser: presence/absence of laser (photosuppression)
% choice: 1 = null (antipreferred) direction, 2 = preferred direction, 3 = sure bet (opt-out)
% 
% *(In our setup, the trial number variable incremented even after an
% aborted trial (e.g. fixation break), and thus is slightly greater than
% the number of completed/valid trials. We made no assumptions about whether
% the most relevant variable is completed trials, all trials, or elapsed
% time, but typically treated this variable as a proxy for all three.)

coh_set = unique(d.coh);

% calculate frequencies of pref-dir and sure-bet choices
pSureTarg = nan(length(coh_set), 2); % coh x nolaser/laser
pPref_wstrg = nan(length(coh_set), 2);
pPref_nostrg = nan(length(coh_set), 2);

J = d.strg; % trials with sure-bet target available
prefTarg = 2; % by convention, choice=2 corresponds to the pref dir

for c = 1 : length(coh_set)              
    I = d.coh==coh_set(c);
    for l = 0 : 1
        L = d.laser==l;

        nPref_wstrg = sum(I & J & L & ismember(d.choice,[1 2]));
        pPref_wstrg(c,l+1) = sum(I & J & L & d.choice==prefTarg) / nPref_wstrg;

        nPref_nostrg = sum(I & ~J & L & ismember(d.choice,[1 2]));
        pPref_nostrg(c,l+1) = sum(I & ~J & L & d.choice==prefTarg) / nPref_nostrg;

        nSureTarg = sum(I & J & L);
        pSureTarg(c,l+1) = sum(I & J & L & d.choice==3) / nSureTarg;
        
    end
end

% standard errors
pPref_wstrg_se = sqrt(pPref_wstrg.*(1-pPref_wstrg)./nPref_wstrg);
pPref_nostrg_se = sqrt(pPref_nostrg.*(1-pPref_nostrg)./nPref_nostrg);
pSureTarg_se = sqrt(pSureTarg.*(1-pSureTarg)./nSureTarg);


%% Fig 1B: Gaussian function for fitting P(sure-bet) vs. coh

psure_fcn_b3 = @(b,coh) min(max(b(1),0),1).*exp(-(coh+b(3)).^2 ./ (2*b(2).^2));
% error to minimize is negative log likelihood of observing the data, which is: -[ log(Psurebet) + log(1-(~Psurebet)) ]
err_fcn_b3 = @(param,sbchoice,coh) -(sum(log(psure_fcn_b3(param,coh(sbchoice))))+sum(log(1-psure_fcn_b3(param,coh(~sbchoice)))));
% select and fit no-laser trials
I = d.strg==1 & d.laser==0;
opts = optimset('MaxFunEvals', 1e6, 'Algorithm','interior-point','LargeScale','off','Display','off');
[Psure_beta_nolaser,~,~,~,~,~] = fminunc(@(x) err_fcn_b3(x,d.choice(I)==3,d.coh(I)), [0.6 0.25 0], opts);

% plot it
g_coh = (-0.52:0.01:0.51)'; g_coh(1) = coh_set(1); g_coh(end) = coh_set(end);
figure; set(gcf, 'Color', [1 1 1], 'Position', [100 100 400 350], 'PaperPositionMode', 'auto'); hold on;
plot(g_coh*100, psure_fcn_b3(Psure_beta_nolaser, g_coh), 'k-', 'LineWidth', 2);
[hsym1,~,hye] = errorbar2(coh_set*100, pSureTarg(:,1), 0, pSureTarg_se(:,1), 'o');
set(hsym1,'MarkerSize',11,'MarkerFaceColor','k','MarkerEdgeColor','k');
set(hye,'LineWidth',1,'Color','k');
set(gca, 'XLim', [-53 53], 'XTick', -50:25:50, 'XTickLabel', makeTickLabel(-50:25:50,25), ...
         'YLim',[0 1], 'YTick', 0:0.1:1, 'YTickLabel', makeTickLabel(0:0.1:1,0.2), ...
         'TickDir', 'out');
xlabel('Motion strength (% coh)');
ylabel('Proportion sure-bet choices');
title('1B');
changeAxesFontSize(gca, 16, 16);


%% Fig 1C: logistic regression for p(pref) vs. coh

% with strg
I = d.strg==1 & d.laser==0 & ismember(d.choice,[1 2]);
X = d.coh(I);
y = d.choice(I)==prefTarg;
[beta_wstrg, ~, ~] = glmfit(X, y, 'binomial');
g_pPref_wstrg = logistic(beta_wstrg(1) + beta_wstrg(2)*g_coh);

% no strg
I = d.strg==0 & d.laser==0 & ismember(d.choice,[1 2]);
X = d.coh(I);
y = d.choice(I)==prefTarg;
[beta_nostrg, ~, ~] = glmfit(X, y, 'binomial');
g_pPref_nostrg = logistic(beta_nostrg(1) + beta_nostrg(2)*g_coh);

% plot it
figure; set(gcf, 'Color', [1 1 1], 'Position', [100 100 400 350], 'PaperPositionMode', 'auto'); hold on;
plot(g_coh*100, g_pPref_nostrg(:,1), 'k--', 'LineWidth', 2);
plot(g_coh*100, g_pPref_wstrg(:,1), 'k-', 'LineWidth', 2);
[hsym1,~,hye] = errorbar2(coh_set*100, pPref_nostrg(:,1), 0, pPref_nostrg_se(:,1), 'o');
set(hsym1,'Color','k','MarkerSize',11,'MarkerFaceColor','w','MarkerEdgeColor','k'); set(hye,'LineWidth',1,'Color','k');
[hsym2,~,hye] = errorbar2(coh_set*100, pPref_wstrg(:,1), 0, pPref_wstrg_se(:,1), 'o');
set(hsym2,'Color','k','MarkerSize',11,'MarkerFaceColor','k','MarkerEdgeColor','k'); set(hye,'LineWidth',1,'Color','k');
set(gca, 'XLim', [-53 53], 'XTick', -50:25:50, 'XTickLabel', makeTickLabel(-50:25:50,25), ...
         'YLim',[-0.02 1.02], 'YTick', 0:0.1:1, 'YTickLabel', makeTickLabel(0:0.1:1,0.2), ...
         'TickDir', 'out');
ylabel('Proportion preferred choices'); xlabel('Motion strength (% coh)');
h=legend([hsym1 hsym2],'T_s unavailable','T_s waived','Location', 'Northwest');
set(h,'FontSize',14); legend('boxoff');
title('1C');
changeAxesFontSize(gca, 16, 16);



%% Fig 1 - Figure Supplement 1A: histogram of viewing durations
figure; set(gcf, 'Color', [1 1 1], 'Position', [100 200 400 300], 'PaperPositionMode', 'auto');
[N,X] = hist(d.duration,20);
bar(X,N,'FaceColor','k','EdgeColor','k','LineWidth',1);
xlabel('Stimulus duration (ms)');
ylabel('Trial Count');
xlim([0 1000]); ylim([0 2500])
set(gca,'box','off','Tickdir','out')
set(gca,'xtick',0:500:1000);
set(gca,'ytick',0:800:2400);
title('1-Supp1A');
changeAxesFontSize(gca,17,17);


%% Fig 1 - Figure Supplement 1B: effect of viewing duration on P(sure-bet)
% group pos + neg coherences
coh_set = {[-0.512 0.512], [-0.256 0.256], [-0.128 0.128], [-0.064 0.064], [-0.032 0.032]};  
Q = 5; % reciprocal of fraction of the data to use in the running mean
colorscheme = {'m-','b-','g-','k-','r-','c-'};
figure; set(gcf, 'Color', [1 1 1], 'Position', [100 200 400 420], 'PaperPositionMode', 'auto');
for c = 1:length(coh_set)
    I = d.strg==1 & ismember(d.coh,coh_set{c});
    [xm,ym,~] = running_mean(d.duration(I), d.choice(I)==3, sum(I)/Q);
    startX = 100; 
    h(c) = semilogx(xm(find(xm>=startX,1,'first'):end), ym(find(xm>=startX,1,'first'):end), colorscheme{c}, 'LineWidth',2); hold on;
end
xlabel('Stimulus duration (ms)');
ylabel('Probability sure-bet choice');
set(gca, 'XLim', [96 960], 'XTick', [100 200 400 800], 'XTickLabel', {'100','200','400','800'} ,...
     'YLim',[0 0.7], 'YTick', 0:0.1:0.7, 'YTickLabel', makeTickLabel(0:0.1:0.7,0.1), 'TickDir', 'out');
set(gca, 'box', 'off');
l = legend(fliplr(h),'3.2','6.4','12.8','25.6','51.2');
set(l,'orientation','horizontal'); legend('boxoff')
set(l,'position',[0.1133 0.9432 0.8822 0.0674]);
title('1-Supp1B');
changeAxesFontSize(gca, 18, 18);


