# Construction of a genetic map for Streblospio benedicti
# Step 3: Construction and ordering of an X-linked linkage group, and then addition of this LG to the Sb676 map built in step 2. 

# rqtl is required
library(qtl)

# Several files are required. Load them from the working directory:
wormIDs <- read.csv(file = "ColWormIDsPedChecked.csv") 
depths <- read.table(file = "FilteredSNPs.read.depths.txt") 
genos <- read.table(file = "FilteredAsGenotypes.txt")

#Generate a data.frame that consolidates the information about each worm
wormIDs[9:384,] -> IDmat
IDmat[,12] <- apply(depths[,10:385], 2, mean, na.rm = T)
IDmat[,13] <- apply(genos[,10:385], 2, function(x){sum(!is.na(x))})
names(IDmat)[2] <- "Worm"
names(IDmat)[12] <- "MeanReadDepth"
names(IDmat)[13] <- "SNPsCalled"

#Establish Strain Inclusion Criteria
min.num.called.SNPs <- 1000
min.mean.read.depth <- 15

#Identify strains with cleanish data
which(apply(genos[,10:385], 2, function(x){sum(!is.na(x))}) > min.num.called.SNPs & apply(depths[,10:385], 2, mean, na.rm = T) > min.mean.read.depth & IDmat$Generation > 0) + 9 -> GoodStrainColumns 
# The +9 is to account for the 9 columns of snp descriptions at the start of the objects.

#Now get rid of duplicates: should remove worms #166, 99, 101.
GoodStrainColumns  <- GoodStrainColumns[-c(which(GoodStrainColumns %in% c(166, 99, 101)))] #291 #100

#############
#Now remove strains that after much additional processing have problems.
#These have crappy data based on diverse statistics.
GoodStrainColumns  <- GoodStrainColumns[-c(which(GoodStrainColumns %in% c(168, 215, 259, 268, 373)))] 

#Remove the worms with pedigree errors
GoodStrainColumns  <- GoodStrainColumns[-c(which(GoodStrainColumns %in% c(113,121,149,171,201,206,219,240,265,267,285,288,306,364,374,378)))] 

#Remove the worms with high genotyping errors based on Rqtl analysis (Step 1)
GoodStrainColumns  <- GoodStrainColumns[-c(which(GoodStrainColumns %in% c(139,191,203,233)))] 

#############

## Here are the cleaned datasets
clean.SNPs <- genos[,1:9]
clean.genos <- genos[,GoodStrainColumns]
clean.depths <- depths[,GoodStrainColumns]
clean.IDs <- IDmat[GoodStrainColumns -9,]

CountGenos <- function(x){
	AA <- sum(x == 0, na.rm = T)
	Aa <- sum(x == 1, na.rm = T)
	aa <- sum(x == 2, na.rm = T)
	return(c(AA, Aa, aa))
}

F1genos <- apply(clean.genos[, which(clean.IDs$Family == "F1")], 1, CountGenos)
F1malegenos <- apply(clean.genos[, which(clean.IDs$Family == "F1" & clean.IDs$Sex == "M")], 1, CountGenos)
F1femalegenos <- apply(clean.genos[, which(clean.IDs$Family == "F1" & clean.IDs$Sex == "F")], 1, CountGenos)
SixFgenos <- apply(clean.genos[, which(clean.IDs$Family == "F")], 1, CountGenos)
SixFmales <- apply(clean.genos[, which(clean.IDs$Family == "F" & clean.IDs$Sex == "M")], 1, CountGenos)
SixFfemales <- apply(clean.genos[, which(clean.IDs$Family == "F" & clean.IDs$Sex == "F")], 1, CountGenos)
OneAgenos <- apply(clean.genos[, which(clean.IDs$Family == "A")], 1, CountGenos)
OneAmales <- apply(clean.genos[, which(clean.IDs$Family == "A" & clean.IDs$Sex == "M")], 1, CountGenos)
OneAfemales <- apply(clean.genos[, which(clean.IDs$Family == "A" & clean.IDs$Sex == "F")], 1, CountGenos)
ThreeCgenos <- apply(clean.genos[, which(clean.IDs$Family == "C")], 1, CountGenos)
ThreeCmales <- apply(clean.genos[, which(clean.IDs$Family == "C" & clean.IDs$Sex == "M")], 1, CountGenos)
ThreeCfemales <- apply(clean.genos[, which(clean.IDs$Family == "C" & clean.IDs$Sex == "F")], 1, CountGenos)
EightHgenos <- apply(clean.genos[, which(clean.IDs$Family == "H" )], 1, CountGenos)
EightHmales <- apply(clean.genos[, which(clean.IDs$Family == "H" & clean.IDs$Sex == "M")], 1, CountGenos)
EightHfemales <- apply(clean.genos[, which(clean.IDs$Family == "H" & clean.IDs$Sex == "F")], 1, CountGenos)

#Calculate a fisher exact test for heterogeneity between sexes in each family
F1sexheterogeneity <- NULL;
for(i in 1: length(clean.genos[,1])){
F1sexheterogeneity[i] <- fisher.test(matrix(data = cbind(F1malegenos[,i], F1femalegenos[,i]), nrow = 3))$p.value
}
SixFsexhet <- NULL;
for(i in 1: length(clean.genos[,1])){
SixFsexhet[i] <- fisher.test(matrix(data = cbind(SixFmales[,i], SixFfemales[,i]), nrow = 3))$p.value
}
ThreeCsexhet <- NULL;
for(i in 1: length(clean.genos[,1])){
ThreeCsexhet[i] <- fisher.test(matrix(data = cbind(ThreeCmales[,i], ThreeCfemales[,i]), nrow = 3))$p.value
}
EightHsexhet <- NULL;
for(i in 1: length(clean.genos[,1])){
EightHsexhet[i] <- fisher.test(matrix(data = cbind(EightHmales[,i],EightHfemales[,i]), nrow = 3))$p.value
}
OneAsexhet <- NULL;
for(i in 1: length(clean.genos[,1])){
OneAsexhet[i] <- fisher.test(matrix(data = cbind(OneAmales[,i],OneAfemales[,i]), nrow = 3))$p.value
}

#Combine these all in a Fisher meta-analysis.
sex.linked.fisher.log10p <-  -log(apply(cbind(F1sexheterogeneity, SixFsexhet, ThreeCsexhet, EightHsexhet, OneAsexhet), 1, function(x){pchisq(2*sum(-log(x)),10, lower = F)}), 10)

plot(sort(sex.linked.fisher.log10p), ylab = "-log10(p)", xlab = "SNP")

data.frame(t(F1genos), t(OneAgenos), t(ThreeCgenos), t(SixFgenos), t(EightHgenos), t(F1malegenos), t(F1femalegenos), t(OneAmales), t(OneAfemales), t(ThreeCmales), t(ThreeCfemales), t(SixFmales), t(SixFfemales), t(EightHmales), t(EightHfemales)) -> GenotypeCounts

names(GenotypeCounts) <- c("AA.F1", "Aa.F1", "aa.F1", "AA.1A","Aa.1A","aa.1A","AA.3C","Aa.3C","aa.3C","AA.6F","Aa.6F","aa.6F","AA.8H","Aa.8H","aa.8H" ,
"mAA.F1", "mAa.F1", "maa.F1", "fAA.F1", "fAa.F1", "faa.F1", "mAA.1A","mAa.1A","maa.1A","fAA.1A","fAa.1A","faa.1A", "mAA.3C","mAa.3C","maa.3C","fAA.3C","fAa.3C","faa.3C","mAA.6F","mAa.6F","maa.6F","fAA.6F","fAa.6F","faa.6F","mAA.8H","mAa.8H","maa.8H","fAA.8H","fAa.8H","faa.8H" )

rm(list = ls()[which(ls() %in% c("F1genos", "OneAgenos", "ThreeCgenos", "SixFgenos", "EightHgenos", "F1malegenos", "F1femalegenos", "OneAmales", "OneAfemales", "ThreeCmales", "ThreeCfemales", "SixFmales", "SixFfemales", "EightHmales", "EightHfemales", "F1sexheterogeneity", "SixFsexhet", "ThreeCsexhet", "EightHsexhet", "OneAsexhet"))])


#Next we will calculate the likelihood of each of the 53 possible segregation patterns (neglecting null alleles) for data from F1 and each of the four F2 families. 

SexLikelihoods <- function(x, output = c("logliks", "best")){
	
	getlnL <- function(err.ab, x) {
		
	dmultinom(x[1:3], 
	prob=c(
	pr.Maa*(1-2*err.ab - err.ab^2) + pr.Mab * err.ab + pr.Mbb*err.ab^2, 
	pr.Maa*2*err.ab + pr.Mab* (1- err.ab - err.ab) + pr.Mbb*2*err.ab, 
	pr.Maa*err.ab^2 + pr.Mab*err.ab + pr.Mbb*(1-2*err.ab - err.ab^2)), log = T) +
	dmultinom(x[4:6], prob=c(
	pr.Faa*(1-2*err.ab - err.ab^2) + pr.Fab * err.ab + pr.Fbb*err.ab^2, 
	pr.Faa*2*err.ab + pr.Fab* (1- err.ab - err.ab) + pr.Fbb*2*err.ab, 
	pr.Faa*err.ab^2 + pr.Fab*err.ab + pr.Fbb*(1-2*err.ab - err.ab^2)), log = T) +
	dmultinom(x[7:9], prob=c(
	pr.f2Maa*(1-2*err.ab - err.ab^2) + pr.f2Mab * err.ab + pr.f2Mbb*err.ab^2, 
	pr.f2Maa*2*err.ab + pr.f2Mab* (1- err.ab - err.ab) + pr.f2Mbb*2*err.ab, 
	pr.f2Maa*err.ab^2 + pr.f2Mab*err.ab + pr.f2Mbb*(1-2*err.ab - err.ab^2)), log = T) +
	dmultinom(x[10:12], prob=c(
	pr.f2Faa*(1-2*err.ab - err.ab^2) + pr.f2Fab * err.ab + pr.f2Fbb*err.ab^2, 
	pr.f2Faa*2*err.ab + pr.f2Fab* (1- err.ab - err.ab) + pr.f2Fbb*2*err.ab, 
	pr.f2Faa*err.ab^2 + pr.f2Fab*err.ab + pr.f2Fbb*(1-2*err.ab - err.ab^2)), log = T)
	}
	
######################	
# X-linked segregation patterns, where males are hemizygous	
	
#	AABY 
	pr.Faa <- 0
	pr.Fab <- 1
	pr.Fbb <- 0
	pr.Maa <- 1
	pr.Mab <- 0
	pr.Mbb <- 0
	pr.f2Faa <- .5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0
	pr.f2Mbb <- 0.5	
	AABY <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })
			
#	ABAY1 	
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0.5
	pr.Mab <- 0
	pr.Mbb <- 0.5
	pr.f2Faa <- 1
	pr.f2Fab <- 0
	pr.f2Fbb <- 0
	pr.f2Maa <- 1
	pr.f2Mab <- 0
	pr.f2Mbb <- 0
	ABAY1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	ABAY2 	
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0.5
	pr.Mab <- 0
	pr.Mbb <- 0.5
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0
	pr.f2Mbb <- 0.5
	ABAY2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	ABAY3 	
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0.5
	pr.Mab <- 0
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 1
	pr.f2Fbb <- 0
	pr.f2Maa <- 1
	pr.f2Mab <- 0
	pr.f2Mbb <- 0
	ABAY3 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	ABAY4 	
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0.5
	pr.Mab <- 0
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0
	pr.f2Mbb <- 0.5
	ABAY4 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	ABBY1 	
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0
	pr.Mbb <- 0.5
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0
	pr.f2Mbb <- 0.5
	ABBY1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	ABBY2	
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 1
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 0
	pr.f2Mbb <- 1
	ABBY2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	ABBY3 	
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0
	pr.f2Mbb <- 0.5
	ABBY3 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	ABBY4 	
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 0
	pr.f2Fbb <- 1
	pr.f2Maa <- 0
	pr.f2Mab <- 0
	pr.f2Mbb <- 1
	ABBY4 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	BBAY 	
	pr.Faa <- 0
	pr.Fab <- 1
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 0
	pr.Mbb <- 1
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0
	pr.f2Mbb <- 0.5
	BBAY <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })


######################
# Now for loci that are sex linked but present on both X and Y

#AA x XA,YB
	pr.Faa <- 1
	pr.Fab <- 0
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 1
	pr.Mbb <- 0
	pr.f2Faa <- 1
	pr.f2Fab <- 0
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 1
	pr.f2Mbb <- 0
	AAXAYB <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AA x XB,YA
	pr.Faa <- 0
	pr.Fab <- 1
	pr.Fbb <- 0
	pr.Maa <- 1
	pr.Mab <- 0
	pr.Mbb <- 0
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0
	AAXBYA <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AA x XB,YB
	pr.Faa <- 0
	pr.Fab <- 1
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 1
	pr.Mbb <- 0
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	AAXBYB <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XA,YA 1
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0
	pr.f2Fab <- 1
	pr.f2Fbb <- 0
	pr.f2Maa <- 1
	pr.f2Mab <- 0
	pr.f2Mbb <- 0
	ABXAYA1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XA,YA 2
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0
	ABXAYA2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XA,YB 1
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 1
	pr.f2Fab <- 0
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 1
	pr.f2Mbb <- 0
	ABXAYB1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XA,YB 2
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 1
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 1
	pr.f2Mbb <- 0
	ABXAYB2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XA,YB 3
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	ABXAYB3 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XA,YB 4
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	ABXAYB4 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })


#AB x XB,YA 1
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0
	ABXBYA1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XB,YA 2
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0
	pr.f2Fab <- 1
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 1
	pr.f2Mbb <- 0
	ABXBYA2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XB,YA 3
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0
	ABXBYA3 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XB,YA 4
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0
	pr.f2Fab <- 0
	pr.f2Fbb <- 1
	pr.f2Maa <- 0
	pr.f2Mab <- 1
	pr.f2Mbb <- 0
	ABXBYA4 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XB,YB 1
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	ABXBYB1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AB x XB,YB 2
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 1
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 0
	pr.f2Mbb <- 1
	ABXBYB2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#BB x XA,YA 
	pr.Faa <- 0
	pr.Fab <- 1
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 1
	pr.Mbb <- 0
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0
	BBXAYA <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#BB x XA,YB 
	pr.Faa <- 0
	pr.Fab <- 1
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 0
	pr.Mbb <- 1
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	BBXAYB <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#BB x XB,YA 
	pr.Faa <- 0
	pr.Fab <- 0
	pr.Fbb <- 1
	pr.Maa <- 0
	pr.Mab <- 1
	pr.Mbb <- 0
	pr.f2Faa <- 0
	pr.f2Fab <- 0
	pr.f2Fbb <- 1
	pr.f2Maa <- 0
	pr.f2Mab <- 1
	pr.f2Mbb <- 0
	BBXBYA <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })


######################
# Now for loci that are not sex linked

#AAxAA, nsl 
	pr.Faa <- 1
	pr.Fab <- 0
	pr.Fbb <- 0
	pr.Maa <- 1
	pr.Mab <- 0
	pr.Mbb <- 0
	pr.f2Faa <- 1
	pr.f2Fab <- 0
	pr.f2Fbb <- 0
	pr.f2Maa <- 1
	pr.f2Mab <- 0
	pr.f2Mbb <- 0
	AAxAA <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#BBxBB, nsl 
	pr.Faa <- 0
	pr.Fab <- 0
	pr.Fbb <- 1
	pr.Maa <- 0
	pr.Mab <- 0
	pr.Mbb <- 1
	pr.f2Faa <- 0
	pr.f2Fab <- 0
	pr.f2Fbb <- 1
	pr.f2Maa <- 0
	pr.f2Mab <- 0
	pr.f2Mbb <- 1
	BBxBB <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AAxBB, nsl 
	pr.Faa <- 0
	pr.Fab <- 1
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 1
	pr.Mbb <- 0
	pr.f2Faa <- 0.25
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.25
	pr.f2Maa <- 0.25
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.25
	AAxBB <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AAxAB, nsl 1
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 1
	pr.f2Fab <- 0
	pr.f2Fbb <- 0
	pr.f2Maa <- 1
	pr.f2Mab <- 0
	pr.f2Mbb <- 0
	AAxAB1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AAxAB, nsl 2
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0
	AAxAB2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#AAxAB, nsl 3
	pr.Faa <- 0.5
	pr.Fab <- 0.5
	pr.Fbb <- 0
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0.25
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.25
	pr.f2Maa <- 0.25
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.25
	AAxAB3 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective }) 


#ABxBB, nsl 1
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 0
	pr.f2Fbb <- 1
	pr.f2Maa <- 0
	pr.f2Mab <- 0
	pr.f2Mbb <- 1
	ABxBB1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#ABxBB, nsl 2
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	ABxBB2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#ABxBB, nsl 3
	pr.Faa <- 0
	pr.Fab <- 0.5
	pr.Fbb <- 0.5
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0.25
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.25
	pr.f2Maa <- 0.25
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.25
	ABxBB3 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective }) 

#ABxAB, nsl 1
	pr.Faa <- 0.25
	pr.Fab <- 0.5
	pr.Fbb <- 0.25
	pr.Maa <- 0.25
	pr.Mab <- 0.5
	pr.Mbb <- 0.25
	pr.f2Faa <- 1
	pr.f2Fab <- 0
	pr.f2Fbb <- 0
	pr.f2Maa <- 1
	pr.f2Mab <- 0
	pr.f2Mbb <- 0
	ABxAB1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#ABxAB, nsl 2
	pr.Faa <- 0.25
	pr.Fab <- 0.5
	pr.Fbb <- 0.25
	pr.Maa <- 0.25
	pr.Mab <- 0.5
	pr.Mbb <- 0.25
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0
	ABxAB2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#ABxAB, nsl 3
	pr.Faa <- 0.25
	pr.Fab <- 0.5
	pr.Fbb <- 0.25
	pr.Maa <- 0.25
	pr.Mab <- 0.5
	pr.Mbb <- 0.25
	pr.f2Faa <- 0
	pr.f2Fab <- 1
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 1
	pr.f2Mbb <- 0
	ABxAB3 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#ABxAB, nsl 4
	pr.Faa <- 0.25
	pr.Fab <- 0.5
	pr.Fbb <- 0.25
	pr.Maa <- 0.25
	pr.Mab <- 0.5
	pr.Mbb <- 0.25
	pr.f2Faa <- 0.25
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.25
	pr.f2Maa <- 0.25
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.25
	ABxAB4 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#ABxAB, nsl 5
	pr.Faa <- 0.25
	pr.Fab <- 0.5
	pr.Fbb <- 0.25
	pr.Maa <- 0.25
	pr.Mab <- 0.5
	pr.Mbb <- 0.25
	pr.f2Faa <- 0
	pr.f2Fab <- 0.5
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	ABxAB5 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#ABxAB, nsl 6
	pr.Faa <- 0.25
	pr.Fab <- 0.5
	pr.Fbb <- 0.25
	pr.Maa <- 0.25
	pr.Mab <- 0.5
	pr.Mbb <- 0.25
	pr.f2Faa <- 0
	pr.f2Fab <- 0
	pr.f2Fbb <- 1
	pr.f2Maa <- 0
	pr.f2Mab <- 0
	pr.f2Mbb <- 1
	ABxAB6 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })
	
	
	# Z-linked segregation patterns, where females are hemizygous	
	
#	BWAA 
	pr.Faa <- 1
	pr.Fab <- 0
	pr.Fbb <- 0
	pr.Maa <- 0
	pr.Mab <- 1
	pr.Mbb <- 0
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0	
	BWAA <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })
			
#	AWAB1 	
	pr.Faa <- 0.5
	pr.Fab <- 0
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 1
	pr.f2Fab <- 0
	pr.f2Fbb <- 0
	pr.f2Maa <- 1
	pr.f2Mab <- 0
	pr.f2Mbb <- 0
	AWAB1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	AWAB2 	
	pr.Faa <- 0.5
	pr.Fab <- 0
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0
	AWAB2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	AWAB3 	
	pr.Faa <- 0.5
	pr.Fab <- 0
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 1
	pr.f2Fab <- 0
	pr.f2Fbb <- 0
	pr.f2Maa <- 0
	pr.f2Mab <- 1
	pr.f2Mbb <- 0
	AWAB3 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	AWAB4 	
	pr.Faa <- 0.5
	pr.Fab <- 0
	pr.Fbb <- 0.5
	pr.Maa <- 0.5
	pr.Mab <- 0.5
	pr.Mbb <- 0
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	AWAB4 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	BWAB1 	
	pr.Faa <- 0.5
	pr.Fab <- 0
	pr.Fbb <- 0.5
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0.5
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0
	BWAB1 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	BWAB2	
	pr.Faa <- 0.5
	pr.Fab <- 0
	pr.Fbb <- 0.5
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 0
	pr.f2Fbb <- 1
	pr.f2Maa <- 0
	pr.f2Mab <- 1
	pr.f2Mbb <- 0
	BWAB2 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	BWAB3 	
	pr.Faa <- 0.5
	pr.Fab <- 0
	pr.Fbb <- 0.5
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	BWAB3 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	BWAB4 	
	pr.Faa <- 0.5
	pr.Fab <- 0
	pr.Fbb <- 0.5
	pr.Maa <- 0
	pr.Mab <- 0.5
	pr.Mbb <- 0.5
	pr.f2Faa <- 0
	pr.f2Fab <- 0
	pr.f2Fbb <- 1
	pr.f2Maa <- 0
	pr.f2Mab <- 0
	pr.f2Mbb <- 1
	BWAB4 <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

#	AWBB 	
	pr.Faa <- 0
	pr.Fab <- 0
	pr.Fbb <- 1
	pr.Maa <- 0
	pr.Mab <- 1
	pr.Mbb <- 0
	pr.f2Faa <- 0.5
	pr.f2Fab <- 0
	pr.f2Fbb <- 0.5
	pr.f2Maa <- 0
	pr.f2Mab <- 0.5
	pr.f2Mbb <- 0.5
	AWBB <- apply(x, 1, function(x){optimize(getlnL, x, interval = c(0,.1), maximum = T)$objective })

######################

	LikelihoodSequence <- c(AABY, ABAY1, ABAY2, ABAY3, ABAY4, ABBY1, ABBY2, ABBY3, ABBY4, BBAY, AAXAYB, AAXBYA, AAXBYB, ABXAYA1, ABXAYA2, ABXAYB1, ABXAYB2, ABXAYB3, ABXAYB4, ABXBYA1, ABXBYA2, ABXBYA3, ABXBYA4, ABXBYB1, ABXBYB2, BBXAYA, BBXAYB, BBXBYA, AAxAA, BBxBB, AAxBB, AAxAB1, AAxAB2, AAxAB3,  ABxBB1, ABxBB2, ABxBB3,  ABxAB1, ABxAB2, ABxAB3, ABxAB4, ABxAB5, ABxAB6  ,BWAA, AWBB,AWAB1, AWAB2,AWAB3,AWAB4,BWAB1,BWAB2,BWAB3,BWAB4)
	names(LikelihoodSequence) <- c("AABY", "ABAY1", "ABAY2", "ABAY3","ABAY4", "ABBY1", "ABBY2", "ABBY3", "ABBY4", "BBAY", "AAXAYB", "AAXBYA", "AAXBYB", "ABXAYA1", "ABXAYA2", "ABXAYB1", "ABXAYB2", "ABXAYB3", "ABXAYB4", "ABXBYA1", "ABXBYA2", "ABXBYA3", "ABXBYA4", "ABXBYB1", "ABXBYB2", "BBXAYA", "BBXAYB", "BBXBYA", "AAxAA", "BBxBB", "AAxBB", "AAxAB1", "AAxAB2", "AAxAB3",  "ABxBB1", "ABxBB2", "ABxBB3",  "ABxAB1", "ABxAB2", "ABxAB3", "ABxAB4", "ABxAB5", "ABxAB6", "BWAA", "AWBB","AWAB1", "AWAB2","AWAB3","AWAB4","BWAB1","BWAB2","BWAB3","BWAB4" )

if(output == "logliks"){
	return(LikelihoodSequence)
	}
	else if (output == "best"){
	return(names(LikelihoodSequence) [which(LikelihoodSequence == max(LikelihoodSequence))])
	}
	else{stop("What is the desired output?")}
}

#Now make datasets that isolate the genotypes for each of the four F2 family pedigrees

# x is c(mAA.F1 mAa.F1 maa.F1 fAA.F1 fAa.F1 faa.F1 mAA.3C mAa.3C maa.3C fAA.3C fAa.3C faa.3C)
JustF1sAnd1As <- GenotypeCounts[,c(16:21, 22:27)]
JustF1sAnd3Cs <- GenotypeCounts[,c(16:21, 28:33)]
JustF1sAnd6Fs <- GenotypeCounts[,c(16:21, 34:39)]
JustF1sAnd8Hs <- GenotypeCounts[,c(16:21, 40:45)]

#For each of the four F2 families, find the maximum likelihood segregation type:
Bestsex1A <- NULL;
for(i in 1: dim(GenotypeCounts)[1]){ Bestsex1A[i] <- SexLikelihoods(JustF1sAnd1As[i,], "best")[1]}
Bestsex3c <- NULL;
for(i in 1: dim(GenotypeCounts)[1]){ Bestsex3c[i] <- SexLikelihoods(JustF1sAnd3Cs[i,], "best")[1]}
Bestsex6F <- NULL;
for(i in 1: dim(GenotypeCounts)[1]){ Bestsex6F[i] <- SexLikelihoods(JustF1sAnd6Fs[i,], "best")[1]}
Bestsex8H <- NULL;
for(i in 1: dim(GenotypeCounts)[1]){ Bestsex8H[i] <- SexLikelihoods(JustF1sAnd8Hs[i,], "best")[1]}

sexlnL <- cbind(Bestsex1A, Bestsex3c, Bestsex6F, Bestsex8H)

#SNPs that segregate reliably as x or y-linked tagged in "xysnps"
xysnps <- rep(0,1389)
xysnps[which(sexlnL[,1] == "AABY" & sexlnL[,2] == "AABY" & sexlnL[,3] == "AABY" & sexlnL[,4] == "AABY")] <- "AABY"
xysnps[which(sexlnL[,1] == "BBAY" & sexlnL[,2] == "BBAY" & sexlnL[,3] == "BBAY" & sexlnL[,4] == "BBAY")] <- "BBAY"

zwsnps <- rep(0,1389)
zwsnps[which(sexlnL[,1] == "BWAA" & sexlnL[,2] == "BWAA" & sexlnL[,3] == "BWAA" & sexlnL[,4] == "BWAA")] <- "BWAA"
zwsnps[which(sexlnL[,1] == "AWBB" & sexlnL[,2] == "AWBB" & sexlnL[,3] == "AWBB" & sexlnL[,4] == "AWBB")] <- "AWBB"

#Recode co-contiguous SNPs that exhibit X-linked segregation
#The main effect of this is to recode impossible haplotypes as missing data.

#Examine every contig that has at least one SNP with p<0.01 for sex linkage
#Namely,
data.frame(clean.SNPs[,1:8], sexlnL)[which(clean.SNPs$SNPsInContig > 1 & clean.SNPs[,1] %in% levels(factor(clean.SNPs[which(sex.linked.fisher.log10p > 2),1]))),]


plotclust <- function(x){
	snpmat <- t(x)
	snpmat <- snpmat[!is.na(apply(snpmat,1,sum)),]
	plot(hclust(dist(snpmat)))
}

#Example contig segregation inspection
par(mfrow = c(5,2))
target <- c(1073:1074)
plotclust(clean.genos[target, which(clean.IDs$Sex == "M" & clean.IDs$Fam == "F1")])
plotclust(clean.genos[target, which(clean.IDs$Sex == "F" & clean.IDs$Fam == "F1")])
plotclust(clean.genos[target, which(clean.IDs$Sex == "M" & clean.IDs$Fam == "A")])
plotclust(clean.genos[target, which(clean.IDs$Sex == "F" & clean.IDs$Fam == "A")])
plotclust(clean.genos[target, which(clean.IDs$Sex == "M" & clean.IDs$Fam == "C")])
plotclust(clean.genos[target, which(clean.IDs$Sex == "F" & clean.IDs$Fam == "C")])
plotclust(clean.genos[target, which(clean.IDs$Sex == "M" & clean.IDs$Fam == "F")])
plotclust(clean.genos[target, which(clean.IDs$Sex == "F" & clean.IDs$Fam == "F")])
plotclust(clean.genos[target, which(clean.IDs$Sex == "M" & clean.IDs$Fam == "H")])
plotclust(clean.genos[target, which(clean.IDs$Sex == "F" & clean.IDs$Fam == "H")])

Xrecoded.genos <- clean.genos

target <- c(155:156)
wormsex <- "F"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 2" & clean.IDs$Fam %in% fams)] <- t(c("MM","MM"))
wormsex <- "M"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))

#Remove 332, which is on a contig with a BBXAYA site (333) and makes no sense. 
xysnps[332] <- 0

target <- c(547:548) #AABY 
wormsex <- "F"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM"))
wormsex <- "M"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))

target <- c(687:688) #AABY 
wormsex <- "F"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM"))
wormsex <- "M"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))

target <- c(848:849) #AABY 
wormsex <- "F"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM"))
wormsex <- "M"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))

target <- c(865:866) #AABY 
wormsex <- "F"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM"))
wormsex <- "M"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))

target <- c(1090:1091) #AABY 
wormsex <- "F"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM"))
wormsex <- "M"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))


target <- c(618:619) #BBAY
wormsex <- "F"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM"))
wormsex <- "M"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))

target <- c(1073:1074) #These are AABY/ABBY 
wormsex <- "F"
fams <- c("H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM"))
fams <- c("A")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM"))
wormsex <- "M"
fams <- c("H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))
fams <- c("A")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "2 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 2" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))


target <- c(1249:1250) #AAXBYA
wormsex <- "F"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM"))
wormsex <- "M"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))

target <- c(530:533) #AAXBYA & BBXAYB
wormsex <- "F"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1 1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MP","MP", "MP", "MP"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0 2 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MM","MM", "MM", "MM"))
wormsex <- "M"
fams <- c("A","H","F","C")
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "1 1 1 1" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("PY","PY"))
Xrecoded.genos[target,which(apply(clean.genos[target,], 2, function(x){paste0(x, collapse = " ")}) == "0 0 2 0" & clean.IDs$Fam %in% fams & clean.IDs$Sex == wormsex)] <- t(c("MY","MY"))


#recode AABY single-snps contigs
which(xysnps =="AABY" & clean.SNPs$SNPsInContig ==1)
# 13 SNPs
for(i in 1:length(which(xysnps =="AABY" & clean.SNPs$SNPsInContig ==1))){
	target <- 	which(xysnps =="AABY" & clean.SNPs$SNPsInContig ==1)[i]
	wormsex <- "F"
	Xrecoded.genos[target,which(clean.genos[target,] == "0" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- "MM"
	Xrecoded.genos[target,which(clean.genos[target,] == "1" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- "MP"
	Xrecoded.genos[target,which(clean.genos[target,] == "2" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- NA
	wormsex <- "M"
	Xrecoded.genos[target,which(clean.genos[target,] == "0" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- "MY"
	Xrecoded.genos[target,which(clean.genos[target,] == "1" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- NA
	Xrecoded.genos[target,which(clean.genos[target,] == "2" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- "PY"	
}

#recode BBAY single-snps contigs
which(xysnps =="BBAY" & clean.SNPs$SNPsInContig ==1)
# 3 SNPs
for(i in 1:length(which(xysnps =="BBAY" & clean.SNPs$SNPsInContig ==1))){
	target <- 	which(xysnps =="BBAY" & clean.SNPs$SNPsInContig ==1)[i]
	wormsex <- "F"
	Xrecoded.genos[target,which(clean.genos[target,] == "2" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- "MM"
	Xrecoded.genos[target,which(clean.genos[target,] == "1" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- "MP"
	Xrecoded.genos[target,which(clean.genos[target,] == "0" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- NA
	wormsex <- "M"
	Xrecoded.genos[target,which(clean.genos[target,] == "2" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- "MY"
	Xrecoded.genos[target,which(clean.genos[target,] == "1" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- NA
	Xrecoded.genos[target,which(clean.genos[target,] == "0" & clean.IDs$Generation ==2 & clean.IDs$Sex == wormsex)]	<- "PY"	
}

#Identify the recoded SNPs
Xreco <- NULL
for(i in 1:1389){Xreco[i] <- "MP" %in% levels(as.factor(as.character(Xrecoded.genos[i,])))}
sum(Xreco)  #38 recoded SNPs

#** NA-out the impossible genotype **
Xrecoded.genos[Xreco == T & Xrecoded.genos == 0] <- NA
Xrecoded.genos[Xreco == T & Xrecoded.genos == 1] <- NA
Xrecoded.genos[Xreco == T & Xrecoded.genos == 2] <- NA

##So at this point there are 10 X-linked multisnp marker and 16 single AABY/BBAY markers, and mendelian errors are NA'd out. 
##One AABY SNP has been removed, 332, because it's in a contig with another SNP and they don't cosegregate cleanly.

#Make a dataset that has unique markers, F2s only. 
t(data.frame(paste0("SNP",row.names(Xrecoded.genos[Xreco,which( clean.IDs$Generation ==2)][which(duplicated(Xrecoded.genos[Xreco,]) ==F),])),"X",Xrecoded.genos[Xreco,which( clean.IDs$Generation ==2)][which(duplicated(Xrecoded.genos[Xreco,]) ==F),])) -> rqtlGenos

#Here these are recoded according to the demands of rqtl X chromosome treatment
rqtlGenos[rqtlGenos == "MY"] <- "AA"
rqtlGenos[rqtlGenos == "PY"] <- "BB"
rqtlGenos[rqtlGenos == "MM"] <- "AA"
rqtlGenos[rqtlGenos == "MP"] <- "AB"

#Now these data are formatted to be an rqtl cross object
data.frame(c("worm", "", clean.IDs[which(clean.IDs$Gen == 2), 1]),c("Sex", "", as.character(clean.IDs[which(clean.IDs$Gen == 2), 3])), c("Family", "", as.character(clean.IDs[which(clean.IDs$Gen == 2), 5])), c("Ind", "", clean.IDs[which(clean.IDs$Gen == 2), 6]), rqtlGenos) -> frame.rqtl
row.names(frame.rqtl) <- NULL
colnames(frame.rqtl) <- NULL
write.csv(frame.rqtl, file = "frame.qtl.csv", quote = F, row.names = F)
read.cross(file = "frame.qtl.csv", format = "csv", geno = c("AA","AB","BB")) -> newX

orderMarkers(newX, "X", error = 0.015) -> newX

#Estimate genotyping error in the reduced dataset
loglik <- err <- seq(0,0.02, 0.001)
for(i in seq(along=err)){
	cat(i, "of",length(err), "\n")
	tempmap <- est.map(newX, error.prob = err[i])
	loglik[i] <- sum(sapply(tempmap, attr, "loglik"))
	}
ml.err <- err[which(loglik == max(loglik))]

n.order.reps <- 20
maplikesX <- NULL
for(j in 1:n.order.reps){
tempX <- newX
	tempX <- orderMarkers(tempX, error = ml.err)
 	assign(paste0("tempX.rep", j), tempX)
maplikesX[j] <- attr(tempX$geno[["X"]]$map, "loglik")
print(paste("Rep", j , "Loglikelihood =", maplikesX[j], date()))
 }

bestX <- which(maplikesX == max(maplikesX)[1])
newX <- get(paste0("tempX.rep", bestX))
write.cross(newX, format = "csv", filestem="Ordered.Xchr.007")

Sb676 <- read.cross(file = "/Applications/lepmap3/SbMap676.csv", format = "csv", geno = c("A","H","B"), na.strings = c("-","NA"))
Sb676$geno[[11]] <- Sb676$geno[[10]]
names(Sb676$geno)[11] <- "X"
Sb676$geno[["X"]] <- newX$geno[["X"]]

write.cross(Sb676, format= "csv", filestem = "Sb702")


