library(dplyr)
library(ggplot2)
library(FNN)

# This is the main code used to generate the model of stochastic committment 
# in a migrating progenitor population. Sourcing this .R file will create
# the 'kid.model.3d' function which is then used to run the model within R.

# The function will return a list of dataframes : 'data' is the relevant 
# cell simulation data, 'tip.data' is the tip position data. 

# setting record = True, will return a list of the above list for each time point (ie
# all timepoint data is recorded, rather than just the final timepoint)

# For all cases in the Lawlor et al, submission there is only a single tip 
# at position (0,0,0), however the code allows for multiple tips to be incorporated.

# This code has been tested on windows with R version 3.4.2 and Linux with R version 3.5.0.


kid.model.3d <- function (c,
                          cc,
                          r,
                          rv.attract,
                          state.thresh,
                          mito.index,
                          re.enter,
                          steps,
                          record = F,
                          starting.cells = 300,
                          niche.reduce = 1) {
  
# This is the main function to run the model.
# c - tip attraction value
  
# cc - [no longer used, but remains here so that downstream analysis 
#       scripts that pass a value to this don't fail, use 'cc = 0.0']

# r  - random migration value

# rv.attract - [no longer used, but remains here so that downstream analysis 
#               scripts that pass a value to this don't fail, use 'rv.attract = 0.0']

# state.thresh - state threshold required for committment, typically set to 30.

# mito.index - mitotic index, typically set to 0.01
  
# re.enter - re-entry value by which state is reduced once cells re-enter the cap.

# steps - number of time steps, typically set to 400.

# record - record all time point data
  
# starting.cells - number of cells at beginning of simulation, typically left at default value of 300.

# niche.reduce - scaling value to reduce niche size (1.0 is no scaling)


#create list to store data 
data.record <- list()

ii = 1

#initiate data frames
sim.data <- data.frame()
sim.tip.data <- data.frame()

# values below dictate the niche size
# for scaling value of 1.0, niche values are as written here.
niche.size = 2 * niche.reduce # this sets the induction cutoff, ie 2 units below the tip centre.
tip.size = 5 * niche.reduce # this sets the tip size

#these values determine the range of starting positions for cells
xlow = -6 * niche.reduce 
xhigh = 6 * niche.reduce
ylow = -1 * niche.reduce
yhigh = 6 * niche.reduce
zlow = -6 * niche.reduce
zhigh = 6 * niche.reduce

sim.data[1:starting.cells,'x'] = runif(starting.cells, xlow, xhigh)
sim.data[1:starting.cells,'y'] = runif(starting.cells, ylow, yhigh)
sim.data[1:starting.cells,'z'] = runif(starting.cells, zlow, zhigh)

sim.data[1:starting.cells, 'type'] = 'cap'
sim.data[1:starting.cells, 'count'] = 0
#record whether cells have undergone any induction, regardless of current state.
sim.data[1:starting.cells, 'any.induction'] = F 

#generate a tip
sim.tip.data[1,'x'] = 0
sim.tip.data[1, 'y'] = 0
sim.tip.data[1, 'z'] = 0
sim.tip.data[1, 'type'] = 'tip'

#loop through for the number of steps to run the simulation
while((ii < steps) & (nrow(sim.tip.data) > 0)) {

#reset tip df
sim.tip.data <- sim.tip.data[c('x','y','z', 'type')]

# find nn tip
# In this version the tip is always at 0,0,0, but the simulation was originally
# designed to have multiple tips, or moving tips.

nn <- get.knnx(data = sim.tip.data[, c('x','y','z')],
                      query = sim.data[,c('x','y','z')],
                      k = 1)

#record nn data for each cell
sim.data['nn.tip'] <- nn[['nn.index']][,1]
sim.data['nn.dist'] <- nn[['nn.dist']][,1]

sim.data['nn.x'] <- sim.tip.data[nn$nn.index,'x']
sim.data['nn.y'] <- sim.tip.data[nn$nn.index,'y']
sim.data['nn.z'] <- sim.tip.data[nn$nn.index,'z']

# generate random values for movement each turn
sim.data['ranx'] <- runif(nrow(sim.data), -1, 1)
sim.data['rany'] <- runif(nrow(sim.data), -1, 1)
sim.data['ranz'] <- runif(nrow(sim.data), -1, 1)

#calculate random x, y, z direction and displacement.
#displacement is based on normal distribution with sd equal to r value.

sim.data <- sim.data %>% mutate('ranx.norm' = ranx / sqrt(ranx^2 + rany^2 + ranz^2), 
                                'rany.norm' = rany / sqrt(ranx^2 + rany^2 + ranz^2),
                                'ranz.norm' = ranz / sqrt(ranx^2 + rany^2 + ranz^2),
                                'ran.disp' = rnorm(n = n(), sd = r))

# calculate unit vector in direction of tip
# this version has been simplified as tip position is always 0,0,0
# should be changed for situation with multiple tips etc

sim.data <- sim.data %>% mutate('tip.vx' = x / sqrt(x^2 + y^2 + z^2), 
                                'tip.vy' = y / sqrt(x^2 + y^2 + z^2), 
                                'tip.vz' = z / sqrt(x^2 + y^2 + z^2) )

# determine how much each cell moves. direction is based on distance to tip (ie attraction or repulsion).
# value is +1 at 5 units, and decrease with inverse square of distance.

#this value determines tip attraction vs repulsion

sim.data <- sim.data %>% mutate('movex' = ifelse(type == 'cap', 
                                                 c * sign(tip.size - nn.dist) * ( tip.size^2 / ifelse(nn.dist >= tip.size, nn.dist^2, tip.size^2) ) * tip.vx + (ran.disp * ranx.norm), 0),
                                'movey' = ifelse(type == 'cap', 
                                                 c * sign(tip.size - nn.dist) * ( tip.size^2 / ifelse(nn.dist >= tip.size, nn.dist^2, tip.size^2) ) * tip.vy + (ran.disp * rany.norm), 0),
                                'movez' = ifelse(type == 'cap', 
                                                 c * sign(tip.size - nn.dist) * ( tip.size^2 / ifelse(nn.dist >= tip.size, nn.dist^2, tip.size^2) ) * tip.vz + (ran.disp * ranz.norm), 0) )  

# determine whether cell are within induction zone and if so add to their state

induction.cutoff = 0 - niche.size # with scaling of 1.0 this cutoff value is -2

sim.data <- sim.data %>% mutate('count' = ifelse( y < induction.cutoff, count + 1, ifelse(count > 0, count - re.enter, count)))

sim.data <- sim.data %>% mutate('any.induction' = ifelse(count > 0, T, any.induction))

#once count value reaches the state.threshold, cell commit and are marked at type 'rv''
sim.data <- sim.data %>% mutate('type' = ifelse(count >= state.thresh, 'rv', type)) 

sim.data <- sim.data %>% mutate('count' = ifelse(type == 'rv', state.thresh, count))

#apply movements
sim.data <- sim.data %>% mutate('x' = x + movex, 'y' = y + movey, 'z' = z + movez)

# sample proportion of cap cells, determine by mito.index value, 
# duplicate these and add back to the main data

mitos <- sim.data %>% filter(type == 'cap') %>% sample_n(nrow(filter(sim.data, type == 'cap')) * mito.index)
if(nrow(mitos) > 0) {
  row.names(mitos) <- (nrow(sim.data)+1):(nrow(sim.data) + nrow(mitos))
  sim.data <- rbind(sim.data, mitos)
}

sim.data['step'] <- ii
sim.data['id'] <- as.integer(rownames(sim.data))

if (record) {
  
  data.record[[ii]] <- list('cells' = sim.data, 'tips' = sim.tip.data)
  
}

ii <- ii + 1

}

if (record) {
  return(data.record)
} else {
return(list('data' = sim.data, 'tip.data' = sim.tip.data))
}

}

###################################


