% This program simulates an extension-time trajectory based on a hidden-Markov model
% with a skew normal noise distribution, optimizes the parameters using HMM-DB, 
% and compares the best-fit parameter values with the input model parameter values.

clear all
close all  
% global data Qnst priorg c 
t0=cputime;
% The input parameters
time=10;  % Time of simulation
T=5000*time;  % Number of data points
Qnst=3; % Number of states

c=1e6;  % Maximum molecular transition rate
dt=2e-4; % smapling time interval
g0=zeros(Qnst,Qnst);
g0=[-1 11 7;
    0 -0.5 10;
    0 0 0];
mu0=(0:Qnst-1).*10;
% mu0=[0 13 15];
Sigma0=[9 10 16];
skew0=[-3 0 3];

ld=diag(true(Qnst,1));  % True for diagonal element
ltriu=triu(true(Qnst),1); % True for upper diagonal elements, excluding diagonal elements
ltril=ltriu'; % True for lower diagonal elements
tmp=g0';
g0(ltril)=tmp(ltril);  % Make g0 symmetric

% disp('g0')
% disp(g0)

ng=0.5*Qnst*(Qnst+1)-1; % number of independent energy parameters
nx=ng+3*Qnst;  % Total number of fitting parameters in x array
nmu=ng+Qnst;   % state position end index in x.
nsig=nmu+Qnst;  % state fluctuation end index in x.
x0=zeros(1,nx);
tmp=g0(ld);
x0(1:Qnst-1)=tmp(1:Qnst-1);
x0(Qnst:ng)=g0(ltril);
x0(ng+1:end)=[mu0 Sigma0 skew0];

disp('x0')
disp(x0)

gs=g0(ld);  % energy state vector
g1=repmat(gs,[1 Qnst]);
q=c.*exp(g1-g0);  %q is the transition rate matrix
q(ld)=0;
q(ld)=-sum(q');

disp('Rate constants:')
disp(q)

if(any(q.*dt)>1)
    disp('The transition is too fast!')
    disp('q.*dt')
    disp(q.*dt)
end

G=(g1-g1')./2;
G=exp(G);  % G(i,j)=exp((gs(i)-gs(j))/2)

% Calculate the symmetrix rate matrix Q
Q=G'.*q;

% Compute the eigenvalue decomposition of Q
[Ue,lambda]=eig(Q);
lambda=lambda(ld);  % lambda is now a column array of eigenvalues
U=cell(Qnst,1); % The symmetric U matrix for spectrum decomposition
for i=1:Qnst
    U{i}=Ue(:,i)*Ue(:,i)';
end
lambda_exp=exp(lambda.*dt);

% Compute the transition probability matrix transmat
transmat=zeros(Qnst,Qnst);
for i=1:Qnst
    transmat=transmat+lambda_exp(i).*U{i};
end
transmat=G.*transmat;

transmat0=transmat;
disp('transmat0')
disp(transmat0)

% [transmat0,mu1,Sigma1]=x2hn(x0,Qnst,c);
tmp=exp(-gs);
statepop0=tmp./sum(tmp);  % State population
lifetime0=-1000./q(ld);  % State lifetime in ms
disp('State population')
disp(statepop0')
disp('Lifetime')
disp(lifetime0')
 
% Simulate a hidden Markov process
initial_prob0 = zeros(Qnst,1);
initial_prob0(1)=1;
[data, hidden] = mhmm_sample_YZ_skew(T, initial_prob0, transmat0, mu0, Sigma0, skew0);

% statistics
aij=zeros(Qnst,Qnst);
for i=1:Qnst
    indexi=~logical(hidden-i);
    aij(i,i)=sum(indexi & circshift(indexi,[-1 0]));
    for j=i+1:Qnst
        indexj=~logical(hidden-j);
        indexj=circshift(indexj,[-1 0]);
        aij(i,j)=sum(indexi & indexj);
        aij(j,i)=aij(i,j);
    end
end
disp(['T= ' num2str(T)])
disp('Number of specific transitions:')
disp(aij)


priorg=initial_prob0;
% Estimate the parameters in the simulated Markov process
nx=length(x0);

% set the initial parameter for optimization
xA=zeros(1,nx);
xA(1:Qnst-1)=x0(1:Qnst-1)+2.*(rand(1,Qnst-1)-0.5);
xA(Qnst:ng)=10;
xA(ng+1:nx)=x0(ng+1:nx)+4.*(rand(1,nx-ng)-0.5);

options = optimoptions('fminunc','Algorithm','quasi-newton','GradObj','on','TolFun',1e-5,'PlotFcns',@optimplotfval);
% options = optimset('Algorithm','active-set','GradObj','on','UseParallel', 'always','Display','iter','TolFun',1e-5,'PlotFcns',@optimplotfval);
tic
f=@(x)hmmfun_grad_skew(x,data,Qnst,ng,priorg,c,dt);
% [x,fmin] = fmincon(f,xA,[],[],[],[],lb,ub,[],options); 
[x,fmin] = fminunc(f,xA,options);

disp('Minimal BIC:')
disp(fmin)

t2=toc;
disp(['CPU time for optimization: ' num2str(t2)])

disp('Comparison of the fitting (starting point, input, output):')
disp([xA;x0;x])

[transmat,mu,Sigma,skew,en_all,en_s]=x2h_grad_skew(x,Qnst,c,dt);
[statepop,lifetime,rate]=state_info_grad(c,en_all);

% Calculate idealized state
sig=sqrt(Sigma);

obslik=zeros(Qnst,T);
data_scaled=zeros(Qnst,T);
for i=1:Qnst
 [obslik(i,:),data_scaled(i,:)]=skewnormpdf(data,mu(i),sig(i),skew(i));
end
path = viterbi_path(priorg, transmat, obslik);
numb_mismatch=sum(~(path==hidden'));
ratio_mismatch=numb_mismatch/T;
disp(['Ratio of mismatch: ' num2str(ratio_mismatch,4)])
  
index=[];
ideal_path=zeros(size(path));
hidden_path=zeros(size(path));
ext_av=zeros(1,Qnst);
% Calculate the relevant parameters.
for j=1:Qnst
   delta=skew(j)/sqrt(1+skew(j)*skew(j));
   ext_av(j)=mu(j)+sig(j)*delta*sqrt(2/pi);
   index=~logical(path-j);
   ideal_path(index)=ext_av(j);
   index=~logical(hidden-j);
   hidden_path(index)=ext_av(j);
end   

xt=(1:T).*dt;
figure
plot(xt,data,'-b',xt,hidden_path,'o-r',xt,ideal_path,'*k')
xlabel('Time (s)')
ylabel('Extension (nm)')

% Plot probability distribution and its best-fit.
bin_size=round(T/500);
% Set the minimum bin size
if(bin_size<10)
 bin_size=10;
end
% Calculate probability density
[ydist,xdist]=hist(data,bin_size);
dx=xdist(2)-xdist(1);
ydist=ydist./T./dx;
xfit=linspace(min(xdist),max(xdist),200);
z={'k','b','g','r','c','m'};
figure('name','Histogram distribution')
plot(xdist,ydist,'ok')
hold on
yfit_total=zeros(size(xfit));
for i=1:Qnst
%    [y,x]=skewnormpdf(xfit,mu(i),sig(i),skew(i));
    yfit=statepop(i).*skewnormpdf(xfit,mu(i),sig(i),skew(i));
    plot(xfit,yfit,'-','color',z{i})
    yfit_total=yfit_total+yfit;
end
plot(xfit,yfit_total,'-','color',z{Qnst+1})
% ------------------------------------------------------------------------
% calculate the standard deviations of the fitted parameters

np=11;
xu=zeros(1,nx);
xb=zeros(1,nx);

xu(1:ng)=x(1:ng)+0.2;
xb(1:ng)=x(1:ng)-0.2;
xu(ng+1:nsig)=x(ng+1:nsig)+1;
xb(ng+1:nsig)=x(ng+1:nsig)-1;
xu(nsig+1:nx)=x(nsig+1:nx)+0.2;
xb(nsig+1:nx)=x(nsig+1:nx)-0.2;


plot_or_not=true;

[xfit,yfit,sigfit,xx,yy]=error_fun(f,x,xb,xu,np,plot_or_not);

disp('Error:')
disp(sigfit)



