###########################
## R code for: 
## (1) a function for simulating mouse population dynamics after the introduction of an autosomal Y-shredding or Y-linked X-shredding gene drive
## (2) running the function for different parameter inputs and plotting the results
###########################

rm(list=ls())
library(reshape2)
library(compiler)
library(iterators)
library(snow)
library(doSNOW)
library(foreach)

###########################################################
## Set up parallel processing
## nproc is the number of processing cores to use
###########################################################

nproc <- 6
cl.tmp = makeCluster(rep('localhost',nproc), type='SOCK')
registerDoSNOW(cl.tmp)
getDoParWorkers()

######################################
### Gene-drive simulation function
######################################

gd <- function(dir, rowNum, input) {
  
  ######################################
  ## assign all variables in input
  ######################################
  
  for (i in 1:ncol(input)) {assign(names(input)[i], input[,i])}
  
  ######################################
  ## Functions
  ######################################
  
  ## calculate population growth rate from population projection matrix
  r.func <- function(x) log(Re((eigen(x)$values)[1]))
  
  ## calculate stable age distibution from population projection matrix
  stable.age.dist <- function(x) ((x %*% (Re((eigen(x)$vectors)[,1])))/(sum((x %*% (Re((eigen(x)$vectors)[,1]))))))[,1]
  
  # function to reverse text strings
  strReverse <- function(x) {sapply(lapply(strsplit(x, NULL), rev), paste, collapse="")} 
  
  ## Create population
  create <- function(nInit, nInit.vec, nG, Pr){
    
    pop <- data.frame(id=1:nInit)
    pop$age <- rep(rep(c(genTime,genTime*2),2),nInit.vec)
    
    ## genotypic and phenotypic sex
    pop$g.sex <- c('f','m')[rep(1:2,c(sum(nInit.vec[1:2]),sum(nInit.vec[3:4])))] ## genotypic sex
    pop$p.sex <- pop$g.sex ## phenotypic sex
    
    ## Autosomes
    pop$aut.p <- pop$aut.m <- nGuides ## start with full complement of cutting sites
    
    ## Allosomes
    pop$all.m <- 'X' ## maternal allosome (sex chromosome)
    pop$all.p <- ifelse(pop$g.sex=='f','X','Y') ## paternal allosome (sex chromosome)
    
    ## Gene drive inoculation (additional to starting population size)
    if (nG>0) {
      
      ## female gene-drive carriers introduced for Y-shredder
      if (YshredEff > 0) {
        g.df <- pop[rep(1,nG),] 
        g.df$aut.m <- -999 ## -999 for gene drive
      }
      
      ## male gene-drive carriers introduced for Y-shredder
      if (XshredEff > 0) {
        g.df <- pop[rep((nInit/2) + 1,nG),]
        g.df$all.p <- 'G' ## G represents Y chromosome carrying the gene drive
      }
      
      ## set unique rownames
      rownames(g.df) <- g.df$id <- (pop$id[nrow(pop)]+1):(pop$id[nrow(pop)]+nG)
      
    }
    pop <- rbind(pop,g.df)
    
    ## Autosome functionality assuming gene-drive cassette is placed in an essential haploinsufficient developmental gene (0 - nonfunctional, 1 - functional)
    ## Assuming here that the gene-drive sequence has a recoded copy of the gene
    if (position=='exon') {
      pop$aut.m.func <- pop$aut.p.func <- 1
    }
    
    ## correction for resistant sites through polymorphism
    pop$aut.m[pop$aut.m!=-999] <- rbinom(sum(pop$aut.m!=-999),pop$aut.m,1-Pr)
    pop$aut.p[pop$aut.p!=-999] <- rbinom(sum(pop$aut.p!=-999),pop$aut.p,1-Pr)
    
    ## Classification of reproductive individuals
    pop$reprod <- 1
    
    ## return
    pop
  }
    
  ## Reproduction (breeding, inheritance)
  reproduce <- function(pop, m){
    
    smart.round <- function(x) {
      y <- floor(x)
      indices <- tail(order(x-y), round(sum(x)) - sum(y))
      y[indices] <- y[indices] + 1
      y
    }
    
    ## select possible mates
    rep.f.ind <- which(pop$p.sex=='f') ## females
    rep.m.ind <- which(pop$p.sex=='m') ## males
    
    ## pairing
    if (length(rep.f.ind)>0 & length(rep.m.ind)>0) {
      
      ## pair males to each female
      if (length(rep.f.ind)>1) {
        rep.f.ind <- sort(sample(rep.f.ind,min(c(length(rep.f.ind),length(rep.m.ind)*matesPerMale)),replace=F))
      }
      sr <- length(rep.f.ind)/length(rep.m.ind)
      if (length(rep.m.ind)==1) {
        rep.m.ind <- rep(rep.m.ind,min(c(length(rep.f.ind),matesPerMale))) ## if 1 male, the male can mate with the minimum of c(length(rep.f.ind),matesPerMale)
      } else if (sr<=1) { 
        rep.m.ind <- sample(rep.m.ind,length(rep.f.ind),replace=F) ## if fewer females than males, sample males that breed
      } else { 
        rep.m.ind.temp <- NULL
        counter <- 0
        while(counter <= ceiling(sr)) {
          sample.size <- min(length(rep.m.ind),length(rep.f.ind)-length(rep.m.ind.temp))
          rep.m.ind.temp <- c(rep.m.ind.temp, sample(rep.m.ind,sample.size,replace=F))
          counter <- counter + 1
        }
        rep.m.ind <- rep.m.ind.temp
      }

      ## create vector of offspring numbers produced from each pairing
      (n.rep.f <- length(rep.f.ind)) # number of females with mates (includes infertile females, and females with litter size of 0)
      m.vec <- ifelse(pop$all.m[rep.f.ind]=='X'&pop$all.p[rep.f.ind]=='X',m,m*XOfertMultiplier)
      n.offspr.vec <- ifelse(runif(length(m.vec))<(m.vec%%1),ceiling(m.vec),floor(m.vec)) # how many offspring per reproducing female (need to account for non-integers in m.vec)
      n.offspr.vec[which(pop$reprod[rep.f.ind]==0)] <- 0 # correct for infertile females that males mated with unknowingly
      (n.offspr <- sum(n.offspr.vec))
      
    } else  {
      
      n.offspr <- 0
    }
    
    ## births
    if(n.offspr > 0){
      
      rep.f.df <- offspring <- pop[rep(rep.f.ind,n.offspr.vec),]
      #rep.m.df <- pop[rep.m.ind,]
      rep.m.df <- pop[rep(rep.m.ind,n.offspr.vec),]
      
      f.aut.m.start <- rep.f.df$aut.m
      f.aut.p.start <- rep.f.df$aut.p
      m.aut.m.start <- rep.m.df$aut.m
      m.aut.p.start <- rep.m.df$aut.p
      offspring$age <- 0
      
      ## Germline X-shredding in males (at meiosis)
      if (XshredEff > 0) {
        IdPossShred <- which(rep.m.df$all.p=='G' & rep.m.df$all.m=='X')
        rep.m.df$all.m[IdPossShred] <- ifelse(runif(length(IdPossShred))<XshredEff,'O','X')
      }
      
      ## Genetic inheritance for Y-shredder
      if (YshredEff > 0) {
        
        ## Germline homing for Y-shredder (modifying rep.f.df and rep.m.df)
    
        ## mothers
        homing.ind.m <- which(rep.f.df$aut.m>0 & rep.f.df$aut.p==-999)
        if(length(homing.ind.m)>0) {
          rmulti <- t(apply(transit[match(rep.f.df$aut.m[homing.ind.m], rownames(transit)),,drop=F], 1, function(x) rmultinom(n=1,size=1,prob=x)))
          rep.f.df$aut.m[homing.ind.m] <- as.numeric(colnames(transit)[apply(rmulti,1,function(x) which(x==1))])
        }
        homing.ind.p <- which(rep.f.df$aut.m==-999 & rep.f.df$aut.p>0)
        if(length(homing.ind.p)>0) {
          rmulti <- t(apply(transit[match(rep.f.df$aut.p[homing.ind.p], rownames(transit)),,drop=F], 1, function(x) rmultinom(n=1,size=1,prob=x)))
          rep.f.df$aut.p[homing.ind.p] <- as.numeric(colnames(transit)[apply(rmulti,1,function(x) which(x==1))])
        }
        
        ## fathers
        homing.ind.m <- which(rep.m.df$aut.m>0 & rep.m.df$aut.p==-999)
        if(length(homing.ind.m)>0) {
          rmulti <- t(apply(transit[match(rep.m.df$aut.m[homing.ind.m], rownames(transit)),,drop=F], 1, function(x) rmultinom(n=1,size=1,prob=x)))
          rep.m.df$aut.m[homing.ind.m] <- as.numeric(colnames(transit)[apply(rmulti,1,function(x) which(x==1))])
        }
        homing.ind.p <- which(rep.m.df$aut.m==-999 & rep.m.df$aut.p>0)
        if(length(homing.ind.p)>0) {
          rmulti <- t(apply(transit[match(rep.m.df$aut.p[homing.ind.p], rownames(transit)),,drop=F], 1, function(x) rmultinom(n=1,size=1,prob=x)))
          rep.m.df$aut.p[homing.ind.p] <- as.numeric(colnames(transit)[apply(rmulti,1,function(x) which(x==1))])
        }
      }
	  
  	  ## Autosome allocation
  	  aut.m.rand <- runif(n.offspr)
  	  aut.p.rand <- runif(n.offspr)
  	  offspring$aut.m <- ifelse(aut.m.rand<0.5, rep.f.df$aut.m, rep.f.df$aut.p)
  	  offspring$aut.p <- ifelse(aut.p.rand<0.5, rep.m.df$aut.m, rep.m.df$aut.p)
  
      ## Allosome allocation
      all.m.rand <- runif(n.offspr)
      all.p.rand <- runif(n.offspr)
      offspring$all.m <- ifelse(all.m.rand<0.5, rep.f.df$all.m, rep.f.df$all.p)
      offspring$all.p <- ifelse(all.p.rand<0.5, rep.m.df$all.m, rep.m.df$all.p)
      
      ## Correct allosome allocation for bias if XO or YO in germline of parents
      XO.ind.m <- which((rep.f.df$all.m=='O'&rep.f.df$all.p=='X')|(rep.f.df$all.m=='X'&rep.f.df$all.p=='O')) 
  	  if (length(XO.ind.m)>0) {
  		  offspring$all.m[XO.ind.m] <- ifelse(all.m.rand[XO.ind.m]<X2Obias, 'X', 'O') ## different inheritance probabilities for XO
  	  }	
      XO.ind.p <- which((rep.m.df$all.m=='X'&rep.m.df$all.p=='O')) 
      if (length(XO.ind.p)>0) {
        offspring$all.p[XO.ind.p] <- ifelse(all.p.rand[XO.ind.p]<X2Obias, 'X', 'O') ## different inheritance probabilities for XO
      }	
      YO.ind <- which((rep.m.df$all.m=='O'&rep.m.df$all.p=='G')) ## only need correction for GO germline, since YO genotype is embryonic lethal
  	  if (length(YO.ind)>0) {
  		  offspring$all.p[YO.ind] <- ifelse(all.p.rand[YO.ind]<Y2Obias, 'G', 'O') ## different inheritance probabilities for YO
  	  }
	  
      ## For autosomal Y-shredder:
      ## Transfer functionality of autosomes
      ## Recalculate functionality of autosomes for alleles that were functional in the parents
      ## Conduct Y-shredding in the zygote
      if (YshredEff > 0) {
        
        if (position=='exon') {
          
          ## transfer
          offspring$aut.m.func <- ifelse(aut.m.rand<0.5, rep.f.df$aut.m.func, rep.f.df$aut.p.func)
          aut.m.start <- ifelse(aut.m.rand<0.5, f.aut.m.start, f.aut.p.start)
          offspring$aut.p.func <- ifelse(aut.p.rand<0.5, rep.m.df$aut.m.func, rep.m.df$aut.p.func) 
          aut.p.start <- ifelse(aut.p.rand<0.5, m.aut.m.start, m.aut.p.start)
          aut.m.diff <- aut.m.start-offspring$aut.m ## loss of cutting sites
          aut.p.diff <- aut.p.start-offspring$aut.p ## loss of cutting sites
  
          ## recalculate (note the gene drive is assumed to carry a functional version of the haploinsufficient gene)
          offspring$aut.m.func <- ifelse(offspring$aut.m==-999,1,ifelse(offspring$aut.m<0 | offspring$aut.m.func==0,0,ifelse(runif(nrow(offspring))>(1-pRnonFunc)^aut.m.diff,0,1)))
          offspring$aut.p.func <- ifelse(offspring$aut.p==-999,1,ifelse(offspring$aut.p<0 | offspring$aut.p.func==0,0,ifelse(runif(nrow(offspring))>(1-pRnonFunc)^aut.p.diff,0,1)))
        }
        
        ## convert back to true number of cutting sites
        offspring$aut.m <- ifelse(offspring$aut.m%in%-(1:(nGuides+1)),-offspring$aut.m-1,offspring$aut.m) ## correct back to number of cutting sites
        offspring$aut.p <- ifelse(offspring$aut.p%in%-(1:(nGuides+1)),-offspring$aut.p-1,offspring$aut.p) ## correct back to number of cutting sites
      
        ## Y-shredding (implemented in zygote)
        IdPossShred <- which(offspring$all.p=='Y' & (offspring$aut.m==-999|offspring$aut.p==-999))
        if (length(IdPossShred>0)) offspring$all.p[IdPossShred] <- ifelse(runif(length(IdPossShred))<YshredEff,'O','Y')
      }

      ## kill genotype YOs and OOs which will not develop (need at least one X to develop)
      ## and also any individual with < 2 functional autosomes (for autosomal Y-shredder only)
      mort <- which(!(offspring$all.m=='X'|offspring$all.p=='X'))
      if (YshredEff > 0) {mort <- sort(unique(c(mort, which(offspring$aut.m.func==0 | offspring$aut.p.func==0))))}
      if(length(mort)>0) {offspring <- offspring[-mort,]}  # kill these ones

      ## genetic and phenotypic sex allocation
      offspring$g.sex <- ifelse(offspring$all.p%in%c('Y','G'),'m','f')
      offspring$p.sex <- ifelse(offspring$all.p%in%c('Y','G'),'m','f')
      
      ## fix ids and rownames
      offspring$id <- max(pop$id)+seq_len(nrow(offspring))
      rownames(offspring) <- offspring$id
      
      # add offspring to population
      if (nrow(offspring)>0) {
        pop <- rbind(pop, offspring)
      }
    }
    pop
  }
  
  ## Mortality
  mortality <- function(pop, s){
    mort <- which(runif(nrow(pop)) > s) # kill these ones
    if (length(mort)>0) {
      pop <- pop[-mort,]
    }
    pop
  }
  
  # Ageing
  age <- function(pop){
    pop$age <- pop$age + genTime # advance age 
    pop
  }
  
  #################################################
  ## Create transition matrix for germline homing events
  #################################################
  
  GeneDriveSeqRec <- function(S, Pc, Pn) {
    
    mat <- matrix(0, S+1, S+1)
    rownames(mat) <- colnames(mat) <- 0:S
    diag(mat) <- sapply(0:S, function(v) (1-Pc)^v)
    n.vec <- unlist(apply(lower.tri(mat)[-1,,drop=F],2,function(x) which(x))) ## cutting sites remaining
    i.vec <- rep(0:(S-1), apply(lower.tri(mat)[-1,,drop=F],2,sum)[-(S+1)]) ## number of resulting sites
    mat[lower.tri(mat)] <- sapply(1:length(n.vec), function(x) choose(n.vec[x],i.vec[x])*((1-Pc)^i.vec[x])*(Pc^(n.vec[x]-i.vec[x]))*Pn^(n.vec[x]-i.vec[x]))
    
    ## add gene drive column
    mat <- cbind(mat, rep(0,S+1))
    colnames(mat)[ncol(mat)] <- -999
    n.vec2 <- 0:S ## cutting sites remaining
    mat[,'-999'] <- sapply(0:S,function(n) sum(sapply(1:n, function(i) choose(n,i)*(Pc^i)*((1-Pc)^(n-i))*(1-(Pn^i)))))
    
    return(mat)
  }
  GeneDriveSimRec <- function(S,Pc,Pn) {
    
    mat <- matrix(0,S,S+2)
    mat[1,1] <- Pc*Pn
    mat[1,2] <- (1-Pc)
    
    if (S>1) {
      for (nS in 2:S) {
        mat[nS,2:(nS+1)] <- (1-Pc)*mat[nS-1,1:nS]
        mat[nS,1:nS] <- mat[nS,1:nS] + Pc*Pn*mat[nS-1,1:nS]
        for (ni in 2:nS) {
          prob <- 0
          for (j in 2:ni) {
            prob <- prob + choose(ni-2,j-2)*Pc^j*(1-Pc)^(ni-j)*Pn*(1-Pn)^j
          }
          for (nl in seq(nS-ni,0,by=-1)) {
            if ((nS-ni)>0) {
              mat[nS,nS-ni-nl+1] <- mat[nS,nS-ni-nl+1] + prob*mat[nS-ni,nS-ni-nl+1]
            } else {
              mat[nS,1] = mat[nS,1] + prob
            }
          }
        }
      }
    }
    mat[,S+2] = 1 - rowSums(mat[,1:(S+1),drop=F])
    mat <- rbind(c(1,rep(0,S+1)), mat)
    rownames(mat) <- 0:S
    colnames(mat) <- c(0:S,'-999')
    return(mat)
  }
  
  ## create transition matrix depending on whether simulataneous/sequential cutting and position of gene drive
  ## when transType=='sim', intervening sequence deletion is indicated by -(s + 1)
  if (transType=='seq') {
    transit <- GeneDriveSeqRec(S=nGuides,Pc=Pc,Pn=Pn)
  } else if (transType=='sim' & position=='intron') {
    transit <- GeneDriveSimRec(S=nGuides,Pc=Pc,Pn=Pn)
  } else {
    transit.seq <- GeneDriveSeqRec(S=nGuides,Pc=Pc,Pn=Pn)
    transit.sim <- GeneDriveSimRec(S=nGuides,Pc=Pc,Pn=Pn)
    diff.mat <- (transit.sim-transit.seq)[,-(nGuides+2)]
    if (nGuides > 1) {diag(diff.mat[-1,]) <- 0}
    colnames(diff.mat) <- -(as.numeric(colnames(diff.mat))+1)
    transit <- cbind(diff.mat,transit.seq)
    transit[,'-999'] <- 1-rowSums(transit[,-ncol(transit)])
  }
  
  #############################################################################
  ### Derive survival for stable population and starting population vector
  #############################################################################
  
  ## calculate fertility rate for r=0
  r0.func <- function(s) {
    a <- matrix(0,4,4)
    a[1,1:2] <- s*m*0.5
    a[3,1:2] <- s*m*0.5
    a[2,1:2] <- a[4,3:4] <- s
    minim <- abs(r.func(a))
    return(minim)
  }
  (s <- optimise(r0.func,interval=c(1e-10,100))$minimum)
  
  ## get stable age distribution
  a <- matrix(0,4,4)
  a[1,1:2] <- s*m*0.5
  a[3,1:2] <- s*m*0.5
  a[2,1:2] <- a[4,3:4] <- s
  (sad <- stable.age.dist(a))
  
  ### adjust rmax for generation time
  (rmax.adj <- rmax*(genTime/52))
  
  ## calculate survival rate for r=rmax
  rmax.func <- function(s) {
    a <- matrix(0,4,4)
    a[1,1:2] <- s*m*0.5
    a[3,1:2] <- s*m*0.5
    a[2,1:2] <- a[4,3:4] <- s
    minim <- abs(r.func(a)-rmax.adj)
    return(minim)
  }
  (s.max <- optimise(rmax.func,interval=c(1e-10,100))$minimum)
  
  ## intercept and slope of logistic survival equation
  (alpha.s <- -log((1/s.max)-1)) 
  (beta.s <- -alpha.s - log((1/s)-1)) 
  
  ## starting n vector
  nInit <- K
  (nInit.vec <- round(sad*nInit))
  
  ###################
  ### Simulation ####
  ###################
  
  ## genotype levels for each strategy
  if (YshredEff > 0) {
    genotype.levels <- apply(expand.grid(list(aut=c('WW','WG','WR','GG','GR','RR'),all=c('XX','XO','XY'))), 
                           1, function(x) paste0(x, collapse=''))
  }
  if (XshredEff > 0) {
    genotype.levels <- c('XX','XO','XY','XG')
  }
  
  # Object for storing results (population size in time)
  year <- seq(0,nYear,by=genTime/52)
  gen <- 0:(nYear*(52/genTime))
  nT <- length(gen)-1
  var <- c('N',genotype.levels)
  results <- array(NA, dim=c(length(year), length(var), nIter), dimnames=list(year=year,var=var,iter=1:nIter))
  results[1,,] <- c(nInit+nG, rep(0,length(genotype.levels)))
  if (YshredEff > 0) {
    results[1,'WWXX',] <- nInit/2
    results[1,'WGXX',] <- nG
    results[1,'WWXY',] <- nInit/2
  }
  if (XshredEff > 0) { 
    results[1,'XY',] <- nInit/2
    results[1,'XG',] <- nG
    results[1,'XX',] <- nInit/2
  }

  
  # Run 
  for (iter in 1:nIter) {
    
    # initial population
    N <- nInit+nG
    pop <- create(nInit=nInit, nInit.vec=nInit.vec, nG=nG, Pr=Pr)
    
    for(t in 2:(nT+1)) {
      
      # demography
      pop <- reproduce(pop, m=m)
      (N.now <- results[match(floor(gen[t-1]), gen),'N',iter])
      (s.now <- ifelse(is.na(rmax),s,1/(1+exp(-(alpha.s+beta.s*N.now/K)))))
      pop <- mortality(pop, s=s.now)
      pop <- age(pop)

      ## calculate summaries
      (N <- nrow(pop))
      if (YshredEff > 0) {
        autosome.genotype <- paste0(ifelse(pop$aut.m==-999,'G',ifelse(pop$aut.m==0,'R','W')),
                                    ifelse(pop$aut.p==-999,'G',ifelse(pop$aut.p==0,'R','W')))
        autosome.genotype[autosome.genotype%in%c('GW','RW','RG')] <- strReverse(autosome.genotype[autosome.genotype%in%c('GW','RW','RG')])
      }
      allosome.genotype <- paste0(pop$all.m,pop$all.p)
      allosome.genotype[allosome.genotype=='OX'] <- strReverse(allosome.genotype[allosome.genotype=='OX'])
      if (YshredEff > 0) {genotype <- factor(paste0(autosome.genotype, allosome.genotype), levels=genotype.levels)}
      if (XshredEff > 0) {genotype <- factor(allosome.genotype, levels=genotype.levels)}
      
      # store results (just for each whole generation time)
      results[t,'N',iter] <- N
      results[t,genotype.levels,iter] <- as.numeric(table(genotype))
      
      # break when one sex remains or N = 0
      if(N==0) {break}
      
    }
    
    print(iter)
  }
  
  ## melt into required format
  res <- melt(results[,'N',],varnames=c('year','iter'),value.name='N')
  for (p in 1:length(genotype.levels)) {
    genotype.label <- genotype.levels[p]
    genotype.res <- melt(results[,genotype.label,],varnames=c('year','iter'),value.name=genotype.label)
    res <- cbind(res, genotype.res[,genotype.label,drop=F])
    print(p)
  }
  
  ## add all the parameters to the results data frame too
  for (i in 1:ncol(input)) {
    res[[names(input)[i]]] <- input[,i]
  }
  
  ## adjust iteration number if necessary
  res$iter <- res$iter + iter.const
  
  ## save results
  save.nm <- paste0('row',rowNum+save.const)
  assign(save.nm, res)
  save(list=save.nm, file=paste(dir,save.nm,sep='/'))

}

############################################
## compile the function so it runs faster
############################################

gd.comp <- cmpfun(gd)

############################################################################
## create data frame of parameter inputs
############################################################################

input <- list()
input$nYear <- 10 # maximum number of years
input$nIter <- 5 # number of iterations
input$K <- 10000 # carrying capacity of offspring
input$genTime <- 5.2 # generation time (in weeks)
input$position <- 'exon' # position of gene drive, one of 'intron' or 'exon'
input$transType <- 'sim' ## transition matrix type, either 'seq' (sequential) or 'sim' (simultaneous)
input$nG <- 100 # initial number carrying the gene drive
input$m <- 6 # litter size
input$rmax <- 7.76 # maximum annual population growth rate
input$matesPerMale <- c(1,3,5) # mean mates per male
input$nGuides <- 3 # number of guide RNAs
input$Pc <- 0.95 # cutting probability
input$Pn <- 0.1 # NHEJ probability
input$Pr <- 0 # probability of inherent resistance (affects number of susceptible sites at initiation)
input$pRnonFunc <- 1 # probability of single-site deletion causing loss-of-function (if position=='exon')
input$YshredEff <- c(0, seq(0.5,1,0.02)) # probability of Y-shredding (set YshredEff > 0 for autosomal Y-shredder)
input$X2Obias <- 2/3 # probability of transferring X to offspring when XO genotype (only relevant when YshredEff > 0)
input$XshredEff <- c(0, seq(0.5,1,0.02)) # probability of X-shredding (only relevant for Y-linked X-shredder)
input$Y2Obias <- 2/3 # probability of transferring Y to offspring after germline X-shredding has occurred (only relevant when XshredEff > 0)
input$XOfertMultiplier <- 0.6 # multiplier of m (litter size) for sub-fertile XO females

## create simulation experiment
inputs <- expand.grid(input)
inputs <- subset(inputs, !(YshredEff>0 & XshredEff>0) & !(YshredEff==0 & XshredEff==0)) ## ensure only one shredder is turned on
inputs$Y2Obias <- ifelse(inputs$XshredEff==0,inputs$Y2Obias,ifelse(inputs$matesPerMale==1,0.5,ifelse(inputs$matesPerMale==3,0.75,1)))
inputs$matesPerMale <- ifelse(inputs$XshredEff==0,inputs$matesPerMale,3)
inputs$strategy <- factor(ifelse(inputs$YshredEff>0,'Y-shredder','X-shredder'),levels=c('Y-shredder','X-shredder'))
rownames(inputs) <- 1:nrow(inputs)
inputs

#########################################################
## run models (just for 5 iterations per scenario)
#########################################################

## directory for saving results
dir <- 'res_X_Y_shredder'
dir.create(dir)

## adjustment of iteration and filename for adding extra iterations
inputs$iter.const <- 0
inputs$save.const <- 0
rownames(inputs) <- 1:nrow(inputs)
inputs

## run models
system.time(
  foreach (rowNum=1:nrow(inputs), .packages='reshape2',.verbose=F) %dopar% {gd.comp(dir=dir, rowNum=rowNum, input=inputs[rowNum,])}
)

############################
## Process and plot results
############################

library(data.table)

## bind results together
res.nms <- list.files(dir)
(rm.genotypes <- c('XX','XO','XY','XG',apply(expand.grid(list(aut=c('WW','WG','WR','GG','GR','RR'),all=c('XX','XO','XY'))), 
                                             1, function(x) paste0(x, collapse='')))) 
res.list <- lapply(res.nms, function(x) {load(paste(dir,x,sep='/'))
  print(x)
  df <- eval(as.name(x))
  return(subset(df, select=names(df)[!names(df)%in%rm.genotypes]))})
dat <- rbindlist(res.list)
dat <- subset(dat, year==10) ## retain final timestep only
dat[is.na(dat)] <- 0 ## replace NAs with 0s
dat$shredEff <- ifelse(dat$strategy=='Y-shredder',dat$YshredEff,dat$XshredEff) ## new shredding efficiency variable
dat$matesPerMale <- factor(dat$matesPerMale)

## summarise
final.N <- aggregate(N ~ shredEff + matesPerMale + Y2Obias + strategy , data=dat, FUN=mean)
quants.final.N <- aggregate(N ~ shredEff + matesPerMale + Y2Obias + strategy, data=dat, FUN=function(x) quantile(x,probs=c(0.025,0.975)))
final.N <- cbind(final.N,quants.final.N[,'N'])
names(final.N)[(ncol(final.N)-1):ncol(final.N)] <- c('lower','upper')
head(final.N)

## plot
lwidth <- 2.5
mycols <- c('black','cyan','magenta')
par(mfrow=c(1,2),las=1, mgp=c(1.5,0.3,0), mar=c(3,3,2,0.75), tcl=-0.2, cex.axis=0.9, cex.lab=1.2, oma=c(0,0,0,0))
df <- subset(final.N, strategy=='Y-shredder')
df[,c('N','lower','upper')] <- df[,c('N','lower','upper')]/1000
plot(df$shredEff, df$N, type='l', col='black', xlim=c(0.5,1), ylim=c(0,12.3), xaxt='n',yaxt='n', lwd=lwidth,
     xlab=expression(italic(P)[Y]), ylab="Final population size ('000s)", xpd=NA)
rect(par("usr")[1], par("usr")[3], par("usr")[2], par("usr")[4], col = "white")
abline(h=c(0,2,4,6,8,12),v=seq(0.5,1,0.1), lty=3, lwd=1, col='lightgrey')
abline(h=10,lty=2, lwd=1, col='black')
axis(1,at=seq(0,1,0.1),labels=T)
axis(2,at=seq(0,12,2),labels=T)
for (i in 1:3) {
  mycol <- mycols[i]
  df.new <- subset(df, matesPerMale==c(1,3,5)[i])
  lines(df.new$shredEff, df.new$N, type='l', col=mycol, lwd=lwidth)
  poly.col <- as.numeric(col2rgb(mycol, alpha = FALSE))/255
  polygon(c(df.new$shredEff,rev(df.new$shredEff)), c(df.new$lower,rev(df.new$upper)), col=rgb(poly.col[1],poly.col[2],poly.col[3],alpha=0.2),
          border=rgb(poly.col[1],poly.col[2],poly.col[3],alpha=0.3))
}
legend('topright', legend=c(1,3,5), title=expression(italic(F)[max]), lty=1, 
       lwd=lwidth, col=mycols, inset=c(0.03,0.26), bg='white', seg.len = 0.8)
mtext(text='(a) Autosomal Y-shredder', side=3, adj=0, line=0.5)
par(mar=c(3,0.75,2,3))
df <- subset(final.N, strategy=='X-shredder')
df[,c('N','lower','upper')] <- df[,c('N','lower','upper')]/1000
plot(df$shredEff, df$N, type='l', col='black', xlim=c(0.5,1), ylim=c(0,12.3), xaxt='n',yaxt='n', lwd=lwidth,
     xlab=expression(italic(P)[X]), ylab="", xpd=NA)
rect(par("usr")[1], par("usr")[3], par("usr")[2], par("usr")[4], col = "white")
abline(h=c(0,2,4,6,8,12),v=seq(0.5,1,0.1), lty=3, lwd=1, col='lightgrey')
abline(h=10,lty=2, lwd=1, col='black')
axis(1,at=seq(0,1,0.1),labels=T)
axis(2,at=seq(0,12,2),labels=F)
for (i in 1:3) {
  mycol <- mycols[i]
  df.new <- subset(df, Y2Obias==c(1,0.75,0.5)[i])
  lines(df.new$shredEff, df.new$N, type='l', col=mycol, lwd=lwidth)
  poly.col <- as.numeric(col2rgb(mycol, alpha = FALSE))/255
  polygon(c(df.new$shredEff,rev(df.new$shredEff)), c(df.new$lower,rev(df.new$upper)), col=rgb(poly.col[1],poly.col[2],poly.col[3],alpha=0.2),
          border=rgb(poly.col[1],poly.col[2],poly.col[3],alpha=0.3))
}
legend('topright', legend=c('1.0','0.75','0.5'), title=expression(italic(P)['Y|YO']), lty=1,
       lwd=lwidth, col=mycols, inset=c(0.03,0.26), bg='white', seg.len = 0.8)
mtext(text='(b) Y-linked X-shredder', side=3, adj=0, line=0.5)
box()


