/**********************************************************************************************************************************
* Simulator for rate-based neurons with synaptic and structural plasticity
*	devised by Michael Fauth (mfauth[at]gwdg.de), 2017-2019
*
* Usage:
* 	[binary] [filename-prefix] [random stimulation interval s] [simulation time in h] [in-group-connectivity] [stimulation file]
*
* This program simulates the dynamics of a network of rate-based neurons with synaptic and structural plasticity as well as 
* short term depression.
*
* The specified filename-prefix will be combined with different suffixes to produce different output files. [prefix].rates
*  will hold the neuron rates, [prefix].weights, .synapses and .synapses2  contain the flattened weight and number of synapse
*  matrices as well as the occupancy of all potential synapses (binary representation) for each log-interval. Note, logging 
* will slow down the simulation and produce vast amounts of data such that the log interval is typically chosen to be large 
* (determined within the stimulation file, see below).
* To monitor effects at short timescales, specific mechanisms have been implemented:
* - The program assumes that the first neurons belong to one of three stimulation groups which have a specific size (and 
*   overlap). The initial number of synapses within these stimulation groups is specified by the initial in-group connectivity 
*   parameter.
* - A histogram of weights within and between the stimulation groups and the rest of the neurons (control) is saved at every 
*   log interval (.histograms).
* - The program will detect spontaneous reactivations (Up-states) within the first group and log their timings as well as the 
*   group rates and connectivity. 
* - The stimulation file can be used to activate a fast-log-mechanism that logs all rates every random stimulation interval 
*   (.switches).
* Moreover, te program will output a histogram of synapse-lifetimes and histograms of the pairwise activitiy of neurons
* within and across the stimulation groups during the non-random-stimulation intervals
*
* The neurons in the simulation receive stimulations currents specified in a stimuation file with following line structure:
* 	[time in h] [mode] [start neuron index] [stop neuron index] [stimulation current (unitless)] [log interval in seconds]
* If mode is "S", neurons betwen start and stop neuron indices (0 to N-1) receive the specified stimulation current at the 
* specified time. If mode is "R" a random stimulation protocoll will be initialized. Here a number of neurons are randomly
* selected and stimulated. After the specified random stimulation interval, the pattern is changed. Other parameters are
* ignored. This paradigm ends as soon as a stimulation is applied. If the mode is "F", a fast-log-mechanism is activated or 
* deactivated,  which writes out the rates of all neurons at each timestep. 
*
*
*
*  Further Notes:
*  - Instead of linked-lists and stl-structures, sorted arrays are used for time-critical processes, as they led to drastic 
*    speed up under all tested compiler optimizations.
*  - Inhomogeneous Poisson processes are simulated by generating a exponentiall distributed threshold and decreasing it by 
*    the current removal rate each timestep.
*********************************************************************************************************************************/




#include <stdio.h>
#include <sstream>
#include <iostream>
#include <fstream>
#include <algorithm>
#include <vector>
#include <random>
#include <math.h>
#include <csignal>
using namespace std;


// Network  parameters

const int N_cells = 240;			// Number of cells in the network
const double I_low = -5.; 			// Initial current at all neurons (usually overwritten by stim-file)
const double dt = 0.1;				// Euler-Integration timestep

// Log parameters
const double CORRELOGRAM_START     = 6.;
const double LOG_UPON_CHANGE_START = 30.;


// Stimulation parameters
const int N_group = 30;				// Size of groups used for log-routines and initial connectivity
const int overlap = 0;				// Overlap of groups used for log-routines and initial connectivity
unsigned int  stim_number = 15;		// Number of neurons stimulated during each random current pattern
double stim_current = 50.;			// Stimulation current used for random stimuations
const int	 pattern_all = 0;		// 1= apply random stimulations also to stimulation groups

int in_group_connectivity;          // Initial number of synapses within groups


/// Neuron model
const double tau = 0.155;			// Membrane potential timescale for neuron
const float rho       = 1.;			// Skewness of generalized logistic function (1=normal sigmoid)
const double v_off    = 0.0;		// Rate offset v = v_off + (1-v_off)*sigmoid( potential )

const double I_off = 0.;			// Constant offset current



// Noise
const double sigma_I = 1.5;								// Shot noise amplitude

// Synaptic plasticity parameter
const double w_max = 0.7;								// Scaling of the weight
const double v_thresh = 0.5;							// Threshold activity for LTP/LTD
const double mu = 0.10; 								// Learning rate scaling for LTP and LTD
const double delta_LTP = mu * 1.0;						// LTP rate
const double delta_LTD = mu * 0.1;						// LTD rate
const double delta_decay = -1./2./24./3600.;			// Decay rate


// Structural plasticity parameters
const int P_max = 16;									// Maximal number of potential synapses per connection
const bool use_dist = false; 							// true if distribution from Fares et al. should be used instead of fixed P (truncated to P_max)

const double creation_rate = 1. / 3600. / 24.;  		// rate at which an unused potential synapse is realized
const double removal_rate     = 24.*creation_rate;		// maximal removal rate
const double beta             = 20. / w_max;			// Inclination
const double epsilon	      = 0.35 * w_max;			// Offset weight
const double min_removal_rate = 0.03 * creation_rate; 	// Offset removal rate

const double w_0     = 0.001;							// initial weight of new synapses


// Short term plasticity
const double tau_r = 5.;
const double F_max = 1.;


// Inhibition
const double w_inh = 3.5 * w_max;						// Inhibitory weight


// Spike frequency adaptation:
// ODE as in Jercog et al.
const double tau_relax = 5.;
const double tau_ad    = 0.015;
// scales maximal value I = tau_relax/tau_ad *v -> tau_relax/(35 * (P_max*w_max-w_inh))


unsigned int times_histogram[40];


// Signal handler
// Issue final log cycle after SIGTERM or SIGKILL
bool stop_flag = false;
void sig_handler(int signo)
{
  cout << "\n\nReceived Signal " << signo <<"\n" << std::flush;
  if (signo == SIGKILL)
    stop_flag = true;
  if (signo == SIGTERM)
    stop_flag = true;
}


// Excitatory gain function
float logist [10000];
inline double sigma(double x){
	// Generalized logistic function
	// return v_off+(1.-v_off)*pow((1.+exp(-x)),-rho);
	// Lookup table for logistic function for speedup
	return x<-50 ? v_off : x>50 ? 1. : logist[(int)(100.*(x+50.))];
}


// Inhibitory gain function
inline double inl(double x){
	return x;
}


// Generator for exponentially distributed deletion threshold
//  Inhomogeneous Poisson process for removal is simulated by two components:
//	- Exponentially distributed threshold
//  - Integrator of the current removal rate
//  When Integrator crosses threshold, synapse is removed.
inline double generate_threshold(){
	return  - logf( 1.0f - (float) rand()/RAND_MAX );
}


// Generator for creation-time of next synapse
// - Interval is exponentially distributed,
// - Rate is scaled with number of free potential synapses
inline double creation_time(double time, short P, short S){
	return -logf(1.0f - (float) rand() / (RAND_MAX)) / (creation_rate*(P-S)) +time;
}


// Auxiliary function to determine whether two indicies i and j are in the same group
inline bool same_group(int i,int j){
	if ((int)(i/(N_group-overlap))==(int)(j/(N_group-overlap)))
		return true;
	else if ( (i%(N_group-overlap)<overlap) and ((int)(i/(N_group-overlap))==((int)(j/(N_group-overlap))+1)))
		return true;
	else if ( (j%(N_group-overlap)<overlap) and ((int)(j/(N_group-overlap))==((int)(i/(N_group-overlap))+1)))
		return true;
	else if (i<overlap and j>2*N_group-2*overlap and (j<3*N_group-3*overlap))
		return true;
	else if (j<overlap and i>2*N_group-2*overlap and (i<3*N_group-3*overlap))
		return true;
	else return false;
}


// Structure for stimulation information extracted from file
struct Stimulation{
	double time;
	char mode;
	int index1;
	int index2;
	double I_stim;
	int update_interval;
};


// Single synaptic contact
struct Synapse{
	short pre;				// presynaptic neuron index
	short place;			// which potential synapse
	double weight;			// weight
	double lifecounter;		// integrator for removal
	double creation_time;	// when synapse was created -> lifetime
};


// Connection between two neurons (with multiple potential synapses)
struct PotentialSynapse{
	int    pre;				// presynaptic neuron index
	double next_creation;	// time when next synapse is created
	short  P;				// Number of potential synaptic locations
	short  S;				// Number of functional synapses
	unsigned int places;	// binary representation which synapses are functional
};

// Neuron
class Neuron{

	PotentialSynapse PotSynapses[N_cells];

public:
	int N_syn;						// Number of realized synapses
	Synapse ExSynapses [(int)(N_cells*P_max*0.3)]; // Synapse array

	double u;						// Membrane potential
	double u_new;					// new value for synchronous update


	double f_STP;					// Short term plasticity utilization

	double stimulation_current;		// External current
	double homeostatic_current;
	double adaptable_current;
	double random_current;			// Noise

	double v;						// Firing rate




	Neuron(double I, int index);
	double update(Neuron * Neurons[], double time, double inhibitory_current, int i);
	double set_rate();
	double log_it(std::ofstream * weightfile,std::ofstream * synapsefile, std::ofstream * synapsefile2,int group_min,int group_max, unsigned int group_hist[], unsigned int out_hist[]);
};



// Neuron constructor
Neuron::Neuron(double I, int index): stimulation_current(I) {

	// Initialize values

	N_syn = 0;
	f_STP = 1.;
	u = I;
	v = sigma(u);
	homeostatic_current = 0.;
	adaptable_current = I_off;
	random_current = 0.;


	// Set up all-to-all potential connectivity and initial connectivity

	std::default_random_engine generator;
	// Number of synapse distribution measured from L5-L5 connections in Fares et al.
	std::discrete_distribution<int> P_dist {0,122,126,122,118,104,92,80,66,55,44,38,28,24,20,16,12,11,9,6,4};
	for(int i=0;i<N_cells;i++){
		PotSynapses[i].pre           = i;
		PotSynapses[i].places	     = 0;
		if (use_dist == true) // use number of synapse distribution
			PotSynapses[i].P         = P_dist(generator)+1;
		else
			PotSynapses[i].P		 = P_max;
		if (i==index)	// no autapses
			PotSynapses[i].P         = 0;

		// Initial connectivity within groups
		if ((i!=index) and (i<3*N_group-3*overlap) and (index<3*N_group-3*overlap) and same_group(i, index)){
			PotSynapses[i].S             = in_group_connectivity;
			PotSynapses[i].next_creation = creation_time(0.,P_max,in_group_connectivity);
			for(int k=0; k<in_group_connectivity; k++){
				// create synapse
				ExSynapses[N_syn].pre          = PotSynapses[i].pre;
				ExSynapses[N_syn].weight       = 0.9 * w_max;
				ExSynapses[N_syn].lifecounter  = generate_threshold()/removal_rate;
				ExSynapses[N_syn].creation_time= 0;
				ExSynapses[N_syn].place        = k;
				N_syn += 1;
				PotSynapses[i].places           += (1<<k);

				// resort ExSynapses list
				for(int j=N_syn-1; (j>0)  && (ExSynapses[j].pre<ExSynapses[j-1].pre); j--)
					std::swap(ExSynapses[j], ExSynapses[j-1]);
			}
		}
		else{ // Initial connectivity across groups
			PotSynapses[i].S             = 0;
			PotSynapses[i].next_creation = creation_time(0.,P_max,0);
		}

		// Sort list of PotSynases list such that PotSynapse with next synapse creation is at top
		if (i!=index)
			// As PotSynases list is an array and already sorted, we use one Bubble-Sort iteration to restore order after change
			for(int j=i; ((j>0)  and ((PotSynapses[j].next_creation<PotSynapses[j-1].next_creation) or
									 (PotSynapses[j-1].S == PotSynapses[j-1].P))); j--){
				std::swap(PotSynapses[j], PotSynapses[j-1]);
			}

	}

}


double Neuron::update(Neuron * Neurons[], double time, double inhibitory_current, int index){

	// RATE UPDATE

	// Initialize new potential from currents
	u_new = stimulation_current + adaptable_current + homeostatic_current + random_current - w_inh * inhibitory_current;
	// Add all excitatory inputs
	for(int i=0;i<N_syn;i++){
		u_new += Neurons[ExSynapses[i].pre]->v * ExSynapses[i].weight * Neurons[ExSynapses[i].pre]->f_STP;
	}

	// Membrane potential ODE
	u +=  dt*(u_new - u)/tau;

	// SHORT-TERM DEPRESSION ODE (neuron wide)
	f_STP  += dt*( (1.-f_STP)/tau_r  -  F_max*f_STP*v);

	// weight & hazard update
	for(int i=0;i<N_syn;i++){

		// SYNAPTIC PLASTICITY
		if (v>v_thresh)
			if ( Neurons[ExSynapses[i].pre]->v > v_thresh )
				ExSynapses[i].weight += dt *delta_LTP * (w_max-ExSynapses[i].weight);
			else
				ExSynapses[i].weight -= dt* delta_LTD * ExSynapses[i].weight;
		else
			if ( Neurons[ExSynapses[i].pre]->v > v_thresh )
				ExSynapses[i].weight -= dt* delta_LTD * ExSynapses[i].weight;
			else
				ExSynapses[i].weight += dt*delta_decay * ExSynapses[i].weight;



		// SYNAPSE REMOVAL

		// Integrate removal probabilities
		// i.e. reduce initiall exponenially distibuted lifecounter by current removal rate)
		ExSynapses[i].lifecounter -= dt*( min_removal_rate/removal_rate + (1-min_removal_rate/removal_rate)*sigma(beta*(epsilon - ExSynapses[i].weight)));

		// Remove synapse if lifecounter hit zero
		if (ExSynapses[i].lifecounter<=0){

			// Put lifetime into (logarithmic) histogram
			double lifetime=time-ExSynapses[i].creation_time;
			times_histogram[(int) min(max(log10(lifetime/3600.)/log10(sqrt(10))+8.,0.),39.)]+=1;

			// Remove synapse -> move to last position and shorten array by 1, keep place and pre
			int tmp_pre = ExSynapses[i].pre;
			short tmp_place = ExSynapses[i].place;
			for(int j=i; j<N_syn-1; j++)
				std::swap(ExSynapses[j],ExSynapses[j+1]);
			N_syn -=1;

			// Find index k of corresponding PotSynapse in (time-sorted) PotSynapses-list
			int k;
			for(k=0;k<N_cells; k++) {
				if (PotSynapses[k].pre == tmp_pre) break;
			}

			// Decrease number of actual synapses and draw net formation time
			PotSynapses[k].S -=1;
			PotSynapses[k].next_creation = creation_time(time, PotSynapses[k].P, PotSynapses[k].S);

			if (PotSynapses[k].places & (1 << tmp_place)) // Sanity check: Did the PoSynapse know about the synapse
				PotSynapses[k].places -= (1 << tmp_place); // if yes: remove
			else
				cout << "Trying to remove non-existent synapse\t" << PotSynapses[k].places << "\t" << (1 << tmp_place) <<"\n";

			// Resort Pot-List to have next creation time at upmost position
			// Bubble up updated PotSynapse (also beyond inactive potentials)
			for(int j=k; (j>0)  && ((PotSynapses[j].next_creation<PotSynapses[j-1].next_creation) ||
									(PotSynapses[j-1].S==PotSynapses[j-1].P) ); j--)
				std::swap(PotSynapses[j], PotSynapses[j-1]);
			// Bubble down updated Pot_Synapse
			// Should be mutually exclusive with above, but also does no harm if executed
			for(int j=k; ((j<N_cells-1)  && (PotSynapses[j].next_creation>PotSynapses[j+1].next_creation)
										 && (PotSynapses[j+1].P-PotSynapses[j+1].S>0)); j++)
				std::swap(PotSynapses[j], PotSynapses[j+1]);

			// Adjust index to continue with next ExSynapse
			i-=1;
		}



	} // loop over ExSynapses


	// SYNAPSE CREATION
	while((PotSynapses[0].next_creation<time)){ // multiple creations included
		// check whether there is a free spot (fully occupied connections also have creation times)
		if ((PotSynapses[0].S<PotSynapses[0].P) && (N_syn < (int)(N_cells*P_max*0.3)-1)){

			// create synapse
			ExSynapses[N_syn].pre          = PotSynapses[0].pre;
			ExSynapses[N_syn].weight       = w_0;
			ExSynapses[N_syn].lifecounter  = generate_threshold()/removal_rate;
			ExSynapses[N_syn].creation_time= time;

			// select random free place (k-th free synapse <-> ExSynapses[N_syn].place)
			int k = 1 + (rand()%(PotSynapses[0].P-PotSynapses[0].S));
			for(ExSynapses[N_syn].place=0;k>0; ExSynapses[N_syn].place++){
				if ((PotSynapses[0].places & (1 << ExSynapses[N_syn].place)) == 0)
					k--;
				if (k==0)
					break;
			}

			// Insert synapse at free place in PotSynapse and update values
			if ((ExSynapses[N_syn].place < PotSynapses[0].P)
				and ((PotSynapses[0].places & (1<<ExSynapses[N_syn].place)) == 0)) // Sanity check: Is there a free spot?
				PotSynapses[0].places += (1<<ExSynapses[N_syn].place);
			else
				cout << "Trying to generate synapse in nirvana or at occupied space " << ExSynapses[N_syn].place << " " << PotSynapses[0].places  << " " << ((PotSynapses[0].places & (1<<ExSynapses[N_syn].place)) ? 1 : 0) <<"\n";
			PotSynapses[0].S +=1;
			N_syn += 1;

			//Resort ExSynapses list according to pre-indices
			for(int j=N_syn-1; (j>0)  && (ExSynapses[j].pre<ExSynapses[j-1].pre); j--)
				std::swap(ExSynapses[j], ExSynapses[j-1]);

		}

		// Bubble Sort PotSynapses-list (bubble down)
		if(PotSynapses[0].S<PotSynapses[0].P){
			PotSynapses[0].next_creation = creation_time(time, PotSynapses[0].P, PotSynapses[0].S);
			for(int j=0; (j<N_cells-1)  &&  (PotSynapses[j].next_creation>PotSynapses[j+1].next_creation) &&
											(PotSynapses[j+1].P -PotSynapses[j+1].S>0); j++)
				std::swap(PotSynapses[j], PotSynapses[j+1]);
		}
		else // Put inactive potentials to the bottom (last remains the autapses)
			for(int j=0; (j<N_cells-2); j++)
				std::swap(PotSynapses[j], PotSynapses[j+1]);
	}

	// Sanity check: Test whether PotSynapses list is correctly sorted for one neuron every 5 min
	//if((index==15) and (abs(time-round(time/3600.)*3600.)< dt/2)){
	//	for(int j=0; j<N_cells-1; j++){
	//		//cout << "(" << PotSynapses[j].pre <<","<<PotSynapses[j].P<<","<<PotSynapses[j].S<<","<<PotSynapses[j].next_creation<<")" << (j%10 ? " " :"\n");
	//		if ((PotSynapses[j].next_creation>PotSynapses[j+1].next_creation) and (PotSynapses[j+1].S<PotSynapses[j+1].P)){
	//			cout << "Unsorted Pot list at " << time << "\n";
	//		}
	//		if  ((PotSynapses[j].S==PotSynapses[j].P) and (PotSynapses[j+1].S<PotSynapses[j+1].P))
	//			cout << "Inactive pot before active at " << time << " pos "<< j << "\n";
	//	}
	//}

	return 1.; // unused
}

double Neuron::set_rate(){
	v = sigma(u);
	return v;
}



// Log-Function: Neuron writes private variables to file
double Neuron::log_it(std::ofstream * weightfile,std::ofstream * synapsefile,std::ofstream * synapsefile2, int group_min, int group_max, unsigned int group_hist[], unsigned int out_hist[]){
	int j = 0;
	double weight = 0.;
	short synapses = 0;
	unsigned int places = 0;

	// Loop over all possible presynaptic partners
	for(int i = 0; i<N_cells; i++){
		// Loop over all ExSynapses j with the respective pre (list order needed!)
		// and accumulate weight and synapse information
		synapses = 0; weight = 0.; places=0;
		for(;(j<N_syn) && (ExSynapses[j].pre==i); j++){
			weight += ExSynapses[j].weight;
			if ((i>=group_min) and (i<=group_max)) group_hist[min( (int)(70.*ExSynapses[j].weight/w_max),70)]+=1;
			else out_hist[min(70, (int)(70.*ExSynapses[j].weight/w_max)) ]+=1;
			synapses += 1;
			places   += (1 << ExSynapses[j].place);
		}
		// Write sum of weights to file
		*weightfile   << weight << "\t";
		// Write number of synapses to file
		*synapsefile  << synapses << "\t";
		// Write binary representation of occupied synapses to file
		*synapsefile2 << places << "\t";
 	}
	return 1.; // unused
}





















int main(int argc, char **argv)
{

	printf("Welcome to the Synaptic and Structural Plasticity Network Simulation\n");

	// Link signal handlers
	signal(SIGTERM, sig_handler);
	signal(SIGKILL, sig_handler);

	// Fill sigmoid lookup table
	for(int i=0; i<10000;i++)
		logist[i] = v_off+(1.-v_off)*pow(1.+exp(-(-50.+ (100*i/10000.))), -rho);

	// Reset histograms
	for(int i=0;i<40;i++){times_histogram[i]=0;}

	unsigned int rate_histogram[4][40][40];
	for(int h=0;h<4;h++)
		for(int i=0;i<40;i++)
			for(int j=0;j<40;j++)
				rate_histogram[h][i][j] = 0;

	// Define local flags
	int random_active = 0;
	int fastlog_flag  = 0;

	// Create list of neuron indices that receive random stimulation
	std::vector<int> indices;
	for(int i=(1-pattern_all)*(3*N_group-3*overlap);i<N_cells;i++) indices.push_back(i);




	// Commandline argument parsing (no error handling whatsoever
	if (argc != 6){
		printf("Usage %s [filename-prefix] [log interval in s] [simulation time in h] [initial in group connectivity] [stimulation file]\n\n", argv[0]);
		exit(1);
	}
	string filename                         =  argv[1];
	unsigned int random_interval = atoi(argv[2])*(unsigned int)(1/dt);
	double T_max                            = atof(argv[3]);
	in_group_connectivity               = atoi(argv[4]);


	// Read in stimulation file
	vector <Stimulation> Stim_list;
	Stimulation tmp;
	std::ifstream infile(argv[5]);
	for( std::string line; getline( infile, line ); ){
		std::istringstream iss(line);
		iss >> tmp.time >> tmp.mode >> tmp.index1 >> tmp.index2 >> tmp.I_stim >> tmp.update_interval;
		//cout <<  tmp.time << " " << tmp.mode<< " " << tmp.index1 << " " << tmp.index2 << " " << tmp.I_stim<< " "  << tmp.update_interval<< "\n";
		// Check whether times are in order and indices are valid
		if ((tmp.mode=='S') and (tmp.index2<=tmp.index1))
			printf("Indices in stim-file are unordered. Skipping line.\n");
		else if (Stim_list.size() and (tmp.time <= Stim_list.back().time))
			printf("Times in stim-file are unordered. Skipping line.\n");
		else
			Stim_list.push_back(tmp);
	}
	infile.close();


	// Initialize files for standard log
	std::ofstream ratefile(filename + ".rates");
	std::ofstream weightfile(filename + ".weights");
	std::ofstream synapsefile(filename + ".synapses");
	std::ofstream synapsefile2(filename + ".synapses2");
	std::ofstream histfile(filename + ".histograms");
	std::ofstream ratehistfile(filename + ".ratehists");
	std::ofstream ratefile3(filename + ".rates3");
	std::ofstream switchfile(filename + ".switches");


	// Random number genertation
	srand(time(NULL));
	std::random_device rd;
    std::mt19937 e2(rd());
    std::normal_distribution<> dist(0, 1.0);


	// Create Neurons
	Neuron * Neurons[N_cells];
	for(int i=0;i<N_cells;i++){
		Neurons[i] = new Neuron(I_low, i);
	}

	// Create a file which lists the presynaptic neurons for each neuron
	//		Preparatory for the use sparse matrices during logging.
	// 		The following is a hack for the used all-to-all connectivity,
	//		which ensures backward compatibility of analytic framework.
	std::ofstream wiringfile(filename + ".wiring");
	for(int i=0;i<N_cells;i++){
		for(int j=0;j<N_cells;j++)
			wiringfile << j << " ";
		wiringfile << "\n";
	}
	wiringfile.close();




	unsigned int steps = 0;
	// Set initial log interval to 1 hour
	unsigned int update_intervall = 3600./dt;

	// Integration variable for inhibitory population
	double sum_rates = N_cells*sigma(I_low);


	


	short acute_log_flag = 0;

	// Up-State variables for group 1
	int up_flag = 0;
	double last_down_transition = 0.;
	double last_up_transition = 0.;
	float avg_w=0.;
	float avg_S=0.;



	//=============================================== MAIN LOOP==============================================
	for (double t=0;t<3600*T_max;t+=dt){

		// APPLY STIMULATION PARADIGM
		while((Stim_list.size()>0) && (t>Stim_list[0].time*3600.)){		// Apply all stimulations before current time
			switch(Stim_list[0].mode){
				case 'S': // Normal stimulation 
					// Random patterns off
					if (random_active){
						cout << " Random patterns deactivated\n";
						// Reset all currents to get rid of random pattern
						for(int i = 0;i<N_cells;i++) Neurons[i]->stimulation_current = 0.;
						// If time larger than 30h, log everything
						if(t>LOG_UPON_CHANGE_START*3600.)
							acute_log_flag = 1;
						random_active =0;
					}
					update_intervall = 1./dt * Stim_list[0].update_interval;
					
					// Apply the specified current to specified neurons
					for(int i = Stim_list[0].index1;i<=Stim_list[0].index2;i++)
						Neurons[i]->stimulation_current = Stim_list[0].I_stim;
					break;
					
				case 'R':
					stim_current = Stim_list[0].I_stim;
					stim_number  = Stim_list[0].index2;
					update_intervall = 1./dt * Stim_list[0].update_interval;
					indices.clear();
					for(int i=Stim_list[0].index1; i<N_cells;i++) indices.push_back(i);
					// Switch on random patterns
					if (!random_active){
						random_active = 1;
						if(t>LOG_UPON_CHANGE_START*3600.)
							acute_log_flag =1;
						cout << " Random patterns activated\n";
					}
					break;
					
				case 'F':
					// Ignore all other parameters
					fastlog_flag = 1-fastlog_flag;
					cout << "  Set fast-log flag to " << fastlog_flag << "\n";
					break;
			}
			
			// Discard applied stimulation
			Stim_list.erase(Stim_list.begin(), Stim_list.begin()+1);
		}




		// CALCULATE NEW RATES
		for(int i = 0; i<N_cells; i++){
			Neurons[i]->update(Neurons, t, inl(sum_rates), i);
		}

		// UPDATE RATES SYNCHRONOUSLY

		// Decay of the sum of rates ODE
		sum_rates -= dt/tau*sum_rates;
		// Activity in stimulation group 1
		float sum_group1 = 0;
		for(int i = 0;i<N_cells; i++){
			// Update rates and add to sum of rates ODE
			// (numerically problematic for small dt)
			sum_rates += dt/tau * Neurons[i]->set_rate();
			if(i<N_group) sum_group1 += Neurons[i]->v;

			// New values for shot noise current
			Neurons[i]->random_current = sigma_I * dist(e2);
		}


		

		
		// Update 2d-histogram of rates during the wake phases
		if ((t>3600.*CORRELOGRAM_START) and !random_active)
			for(int i=0;i<N_cells;i++) for(int j=0;j<N_cells;j++)
				rate_histogram[ 1*(i<3*N_group-3*overlap)  + 1*(j<3*N_group-3*overlap) + 1*(i<3*N_group-3*overlap)*same_group(i,j)][(int)(40.*Neurons[i]->v)][(int)(40.*Neurons[j]->v)]++;
		
		
		// UP-DOWN-LOGGING
		// upon Down transition

		// Update [prefix].switches:
		//   Write average rates, weights, synapses as well as durations
		//   of Up- and Down-state in stimulation group 1 upon Down transition


		// Up-transition
		// (detected by more than 50% of max activity in group)
		if ((up_flag == 0 ) and (sum_group1 > 0.5 * N_group)){
			last_up_transition = t;
			up_flag = 1;
			avg_w = 0.; avg_S = 0.;
			for(int i = 0; i<N_group; i++)
				for(int j =0;j<Neurons[i]->N_syn;j++)
					if (Neurons[i]->ExSynapses[j].pre < N_group){
						avg_w  += Neurons[i]->ExSynapses[j].weight;
						avg_S  += 1.;
					}
		}
		// Down-transition
		if ((up_flag == 1 ) and (sum_group1 < 0.5 * N_group)){
			float tmp_w = avg_w; // keep weights from Up-transition
			float tmp_S = avg_S; // keep synapses from Up-transition
			avg_w = 0.; avg_S = 0.;
			for(int i = 0; i<N_group; i++)
				for(int j =0;j<Neurons[i]->N_syn;j++)
					if (Neurons[i]->ExSynapses[j].pre < N_group){
						avg_w  += Neurons[i]->ExSynapses[j].weight;
						avg_S  += 1.;
					}
			// Write: 	  Time of Up-transition,		   Duration of previous Down-state, 				 Duration of Up-state, 		  Avg. weight at Up-Transition, 	        Avg. weight at Down-Transition, 		  Avg. #synapses at Up-Transition,		    Avg. #synapses at Down-Transition,
			switchfile << last_up_transition/3600. <<" "<< last_up_transition - last_down_transition <<" "<< t-last_up_transition <<" "<< tmp_w/((double)(N_group*N_group)) <<" "<< avg_w/((double)(N_group*N_group)) <<" "<< tmp_S/((double)(N_group*N_group)) <<" "<< avg_S/((double)(N_group*N_group)) << "\n" << std::flush;

			// Update values
			up_flag = 0;
			last_down_transition = t;
			}


		// FAST-LOGGING
		// each timestep

		// Update [prefix].rates3
		//   Write rates of all neurons @ each timestep

		if (fastlog_flag){
			ratefile3.precision(9);
			ratefile3 << t << "\t";
			ratefile3.precision(5);
			for(int i = 0; i<N_cells; i++){
				ratefile3 << Neurons[i]->v <<" ";
			}
			ratefile3 << "\n" << std::flush;
		}





		// STANDARD LOGGING
		// each update_interval or if flag is set

		// Update: [prefix].rates
		//   Write time and all rates

		// Update: [prefix].weights
		//   Write sum of weights on each connection

		// Update: [prefix].synapses
		//   Write number of synapses on each connection

		// Update: [prefix].histograms
		//   Write number of synapses on each connection


		if ((0 == steps%update_intervall) || (t+dt>3600*T_max)||stop_flag||acute_log_flag){
			acute_log_flag = 0;

			// Initialize weight histogram
			unsigned int aa_vec[71] ={0}; // intra-stim-group
			unsigned int oa_vec[71] ={0}; // inter-stim-group
			unsigned int ac_vec[71] ={0}; // stim-group-from/to-control
			unsigned int cc_vec[71] ={0}; // control-to-control

			// Write: time
			ratefile << t << "\t";

			for(int i = 0; i<N_cells; i++){
				// Write: rates
				ratefile << Neurons[i]->v <<" ";

				// Write, weight, #synapses, synapse occupance; Update weight histograms according to post-neuron
				if (i<(3*N_group))
					Neurons[i]->log_it(&weightfile, &synapsefile, &synapsefile2, N_group*(int)(i/N_group) ,N_group*(int)(i/N_group) + (N_group-1) , aa_vec, oa_vec);
				else
					Neurons[i]->log_it(&weightfile, &synapsefile, &synapsefile2, 3*N_group, N_cells, cc_vec, ac_vec);
			}

			// Write weight histograms
			for(int i=0;i<71;i++)
				histfile << aa_vec[i] << " ";
			for(int i=0;i<71;i++)
				histfile << oa_vec[i] << " ";
			for(int i=0;i<71;i++)
				histfile << ac_vec[i] << " ";
			for(int i=0;i<71;i++)
				histfile << cc_vec[i] << " ";

			histfile << "\n" << std::flush;
			ratefile << "\n" << std::flush;
			weightfile << "\n" << std::flush;
			synapsefile << "\n"<< std::flush;
			synapsefile2 << "\n"<< std::flush;


			// Sanity Check: Printout information (about neuron 0) to terminal
			cout << "\tt=" << t/3600. <<"\tv=" << Neurons[0]->v <<"\tS="<<Neurons[0]->N_syn << "\tu="<< Neurons[0]->u << "\tu_new="<< Neurons[0]->u_new << "\tI_homeo="<< Neurons[0]->homeostatic_current << "\tI_ad="<< Neurons[0]->adaptable_current << "\tI_rnd="<< Neurons[0]->random_current <<"\tSum of rates="<< sum_rates<< "\n";

			// Reset step counter every hour to prevent overflow
			//   Suitable if log is aligned with full hour
			//if (0 == steps%((int)(3600./dt)))
			//	steps=0;
		}



		// CHANGE PATTERN FOR RANDOM STIMULATION
		if (random_active and (0 == steps%random_interval)){
			std::random_shuffle(indices.begin(), indices.end());
			for(unsigned int i=0;i<indices.size();i++){
				if(i<stim_number) 
					Neurons[indices[i]]->stimulation_current = stim_current;
				else 
					Neurons[indices[i]]->stimulation_current = 0.;
			}
		}

		// Increase number of simulated steps
		steps++;

		// End simulation upon signal
		if(stop_flag) break;

	}
	// ========================================== END OF MAIN LOOP ==========================================

	// Write flattened 2d histograms acquired during wake phases to [prefix].ratehists
	for(int h=0;h<4;h++){
		for(int i=0;i<40;i++){
			for(int j=0;j<40;j++)
				ratehistfile << rate_histogram[h][i][j] << " ";
		}
		ratehistfile << "\n";
	}

	// Final update of synapse lifetime histogram:
	//   Put the lifetime up to simulation end int histogram
	//   Problematic, because these synapses were not removed,
	//   but would live of even longer.
	for(int i = 0; i<N_cells; i++){
		for(int j=0; j< Neurons[i]->N_syn;j++ ){

			//put lifetime into histogram
			double lifetime= 3600*T_max - Neurons[i]->ExSynapses[j].creation_time;
			times_histogram[(int) min(max(log10(lifetime/3600.)/log10(sqrt(10))+8.,0.),39.)]+=1;
		}
	}

	// Print synapse lifetime histogram to terminal
	cout <<"t_min \t t_max \t count\n";
	for(int i=0;i<40;i++)
		cout << pow(sqrt(10.), i-8.) << "\t" << pow(sqrt(10.),i-7.) << "\t" << times_histogram[i]<<"\n";



	// Close all used files
	ratefile.close();
	weightfile.close();
	synapsefile.close();
	synapsefile2.close();
	ratehistfile.close();
	ratefile3.close();
	switchfile.close();
}