function [pval x1 fit] = av_test(x,method,time,varargin)
% function [pval x1 fit] = av_test(x,'pos|neg|resp',time,condstim,condblank)
% function [pval x1 fit] = av_test(x,'nonlin',time,condvis,condaudio,condbimodal,condblank)
%---
% Test audiovisual properties of neurons
%
% Input:
% - x           array of size 17*nx*ncond*nrep - data averaged in bins of 250ms
% - method      see below for available methods
% - condstim    stimulated condition number(s)
% - condblank   blank condition number(s)
% - time        interval(e.g. [0 2] means time interval [0 2s])
%
% Output:
% - p       vector of p-values
% - x1      nx*ncond*nrep - data after subtracting pre-stim activity,
%           time averaging inside the window, and of relevant conditions
%           only
% - fit     nx*ncond*2 - fit to different conditions under the null
%           hypothesis and under the full model
%
% Methods:
% 'posnp'   ranksum (response to stim > response to blank)
% 'negnp'   ranksum (response to stim < response to blank)
% 'pos'     T-test (response to stim > response to blank)
% 'neg'     T-test (response to stim < response to blank)
% 'resp'    F-test (H: individual responses for each stim vs. H0: all similar to blank)
% 'nonlin'  F-test (H: individual responses for each stim vs. H0: bimodal responses are linearly predicted)
% 'posb'    bootstrap (response to stim > response to blank)

nx = size(x,2);
nrep = size(x,4);

% subtract activity before stim
subtractprestim = eval('false');
if subtractprestim
    disp 'av_test: doign pre-stim activity subtraction'
    x = fn_subtract(x,mean(x(1:2,:,:,:)));
else
    disp 'av_test: not subtracting pre-stim activity'
end

% average over time period
if isempty(time)
    if size(x,1)~=1, error 'first dimension must be singleton when not specifying a time window', end
    idx = 1;
elseif isscalar(time)
    error 'syntax has changed: please make third argument a 2-element vector for time window to be tested'
else
    if size(x,1)~=17, error 'number of time periods must be 17 (250ms bins)', end
    idx = 3+(time(1)/.25:time(2)/.25-1);
end
x = shiftdim(mean(x(idx,:,:,:),1),1);

switch method
    case {'posnp' 'negnp'}
        % conditions
        [condstim condblank] = deal(varargin{1:2});
        nstim = length(condstim); nblank = length(condblank);
        if nstim>1 || nblank>1
            disp 'several different conditions will be pooled together for ranksum test'
        end
        
        % arrange for ranksum test
        xstim = matrix(x(:,condstim,:));
        xblank = matrix(x(:,condblank,:));
        x1 = [permute(xstim,[1 3 2]) permute(xblank,[1 3 2])];

        % compute p-values
        pval = NaN(1,nx);
        fit = [nmedian(xstim,2) nmedian(xblank,2)];
        idxok = row(find(~any(isnan(fit),2)));
        tail = fn_switch(method,'posnp','right','negnp','left');
        fn_progress('compute p-value',nx)
        for ix=idxok
            fn_progress(ix)
            if isempty(xblank)
                pval(ix) = signrank(xstim(ix,:),'tail',tail);
            else
                pval(ix) = ranksum(xstim(ix,:),xblank(ix,:),'tail',tail);
            end
        end
    case {'pos' 'neg' 'resp' 'posb'}
        % conditions
        [condstim condblank] = deal(varargin{1:2});
        nstim = length(condstim); nblank = length(condblank);
        
        % arrange for GLM test
        x1 = x(:,[condstim condblank],:);
        y = fn_reshapepermute(x1,{[2 3] 1});   % (ncond*nrep)*nx
        regbase = eye(nstim+nblank);
        regressor = repmat(regbase,nrep,1);
        isstim = repmat([true(1,nstim) false(1,nblank)],1,nrep);
        
        % handle different sets of valid trials (invalid trials have NaNs) for different neurons
        validtrials = ~isnan(y); % ntrial*nx
        [uvalid, ~, idxu] = unique(validtrials','rows'); % nx*1
        nu = size(uvalid,1);
        
        % compute p-values (loop on sub-groups of neurons)
        pval = NaN(1,nx);
        fit = NaN(nx,nstim+nblank,strcmp(method,'resp')+1);
        for ku = 1:nu
            ix = (idxu==ku);
            oktrial = uvalid(ku,:);                             % 1*ntrialok
            yk = permute(y(oktrial,ix),[1 3 2]);                % ntrialok*1*nx
            regk = regressor(oktrial,:);
            isstimk = isstim(oktrial);
            nreppercond = sum(regk,1);
            if sum(nreppercond(1:nstim))<=2 || (nblank>0 && sum(nreppercond(nstim+(1:nblank)))<=2)
                % not enough stim or blank trials left in the valid trials
                pval(ix) = NaN;
                continue
            end
            okcond = (nreppercond>0);
            regk = regk(:,okcond);
            nstimk = sum(okcond(1:nstim)); nblankk = sum(okcond(nstim+(1:nblank)));
            switch method
                case 'pos'
                    [pval(ix) beta] = fn_GLMtest(regk,yk,[ones(1,nstimk)/nstimk -ones(1,nblankk)/nblankk],'T');
                    fit(ix,okcond) = (regbase(:,okcond)*beta)';
                case 'neg'
                    [pval(ix) beta] = fn_GLMtest(regk,yk,[-ones(1,nstimk)/nstimk ones(1,nblankk)/nblankk],'T');
                    fit(ix,okcond) = (regbase(:,okcond)*beta)';
                case 'resp'
                    [pval(ix) beta beta0] = fn_GLMtest(regk,yk,1:nstimk,'F');
                    fit(ix,okcond,1) = (regbase(okcond,okcond(nstim+1:end))*beta0)';
                    fit(ix,okcond,2) = (regbase(okcond,okcond)*beta)';
                case 'posb'
                    pval(ix) = fn_bootstrap(yk(isstimk,:),yk(~isstimk,:),'mean','tail','right');
            end
        end
        
    case 'nonlin'
        % conditions
        if isempty(varargin)
            % default conditions
            condvis = 1:2;
            condaudio = 3:4;
            condbimodal = [5 7; 8 6];
            condblank = 9;
            if size(x,2)~=9, error 'number of conditions in data should be 9', end
        else
            if length(varargin)~=4, error 'argument', end
            [condvis condaudio condbimodal condblank] = dealc(varargin);
            if ~isscalar(condblank), error 'there should be only one blank condition', end
            if ~isequal(size(condbimodal),[length(condvis) length(condaudio)])
                error 'number of bimodal conditions must be arranged as an array with as many columns as visual conditions and as many rows as auditory conditions'
            end
        end
        conds = [row(condvis) condaudio row(condbimodal) condblank];
        nv = length(condvis); na = length(condaudio);
        ncond = length(conds); % = nv+na+nv*na+1
        if size(x,2)~=ncond, error 'not implemented yet, be careful with ordering of conditions', end
        
        % arrange for GLM test
        x1 = x;
        y = fn_reshapepermute(x,{[2 3] 1}); % ntrial*nx
        
        % regressors
        % V: visual, A: audio, N: bimodal nonlinear term, K: blank
        % hypothesis H0: responses are explained as V+K, A+K, V+A+K and K
        % full model H:  responses are explained as V+K, A+K, V+A+N+K and K
        reg = zeros(ncond,ncond);
        % V
        reg(1:nv,1:nv) = eye(nv);
        for ia=1:na, reg(nv+na+(ia-1)*nv+(1:nv),1:nv) = eye(nv); end
        % A
        reg(nv+(1:na),nv+(1:na)) = eye(na);
        for iv=1:nv, reg(nv+na+(0:na-1)*nv+iv,nv+(1:na)) = eye(na); end
        % change ordering of conditions in regressor to match the data
        reg(conds,:) = reg;
        % N
        reg(nv+na+(1:nv*na),nv+na+(1:nv*na)) = eye(nv*na);
        % K
        reg(:,nv+na+nv*na+1) = 1;
        % duplicate for the number of repetition
        regbase = reg;
        reg = repmat(reg,[nrep 1]);
        
        % tested hypothesis (or rather dimensions specific to the full
        % model)
        h = nv+na+(1:nv*na); %#ok<NASGU>
        
        % handle different sets of valid trials (invalid trials have NaNs) for different neurons
        validtrials = ~isnan(y);                            % ntrial*nx
        [uvalid, ~, idxu] = unique(validtrials','rows');    % nx*1
        nu = size(uvalid,1);
        
        % compute p-values (loop on sub-groups of neurons)
        pval = NaN(1,nx);
        fit = NaN(nx,ncond,2);
        for ku = 1:nu
            ix = (idxu==ku);
            oktrial = uvalid(ku,:);                	% 1*ntrialok
            okcond = any(reshape(oktrial,[ncond nrep]),2);
            if any(~okcond([1:nv+na ncond]))
                disp 'no trial for blank or one of the unimodal conditions, not testing'
                pval(ix) = NaN;
                continue
            end
            yk = permute(y(oktrial,ix),[1 3 2]);	% ntrialok*1*nx
            regk = reg(oktrial,:);                  % ntrialok*ncond
            regk(:,~okcond) = [];      	% remove (bimodal) regressors that are not used!
            okbimodal = okcond(nv+na+(1:nv*na));
            if ~any(okbimodal)
                disp 'no trial for any bimodal condition, not testing'
                pval(ix) = NaN;
                continue
            end
            h = nv+na+(1:sum(okbimodal));   % and update indices of regressors to test
            [pval(ix) beta beta0] = fn_GLMtest(regk,yk,h,'F');
            fit(ix,:,1) = (regbase(:,[1:nv+na ncond])*beta0)';
            fit(ix,:,2) = (regbase(:,okcond)*beta)';
        end
        
    case 'nonlinboot'
        % conditions
        if isempty(varargin)
            % default conditions
            condvis = 1;
            condaudio = 3;
            condbimodal = 5;
            condblank = 9;
            if size(x,2)~=9, error 'number of conditions in data should be 9', end
        else
            if length(varargin)~=4, error 'argument', end
            [condvis condaudio condbimodal condblank] = dealc(varargin{:});
        end
        conds = [condvis condaudio condbimodal condblank];
        
        % arrange for bootstrap
        y = fn_reshapepermute(x,{[2 3] 1}); % ntrial*nx
        
        % handle different sets of valid trials (invalid trials have NaNs) for different neurons
        validtrials = ~isnan(y);                            % ntrial*nx
        [uvalid, ~, idxu] = unique(validtrials','rows');    % nx*1
        nu = size(uvalid,1);
        
        % compute p-values (loop on sub-groups of neurons)
        pval = NaN(1,nx);
        for ku = 1:nu
            ix = (idxu==ku);
            nxk = sum(ix);
            yk = reshape(y(:,idx),[4 nrep nxk]);        % 4*nrep*nxk
            oktrial = reshape(uvalid(ku,:),[4 nrep]);   % 4*nrep
            ykc = cell(1,4);
            for i=1:4, ykc{i} = squeeze(yk(i,oktrial(i,:),:)); end
            if 0
                % not enough valid trials
                pval(ix) = NaN;
                continue
            end
            pval(ix) = nonlinearBootstrap(ykc);
        end
        
    case 'os'
        nrep = min(row(squeeze(sum(~isnan(y(1,:,:)),2))));
        % orientation selectivity
        A = repmat([ones(1,8); 1 0 -1 0 1 0 -1 0; 0 1 0 -1 0 1 0 -1]',nrep,1);
        [stats.OIp beta] = fn_GLMtest(A,zgrat,2:3);
        stats.OI = 2*norm(beta(2:3))/(abs(beta(1))+norm(beta(2:3)));
        sgn = sign(beta(1));
        theta = (0:7)'/8*(2*pi);
        tpref = mod(1/2*atan2(beta(3),beta(2)),pi); % preferred orientation (e.g. 0 if beta=[1 0], pi/4 if beta=[0 1], pi/2 if beta=[-1 0])
        if sgn==-1, tpref=mod(tpref+pi/2,pi); end % (e.g. pi/2 for [1 0])
        stats.OIpref = tpref;
        
    otherwise
        error('unknown test ''%s''',method)
end


