function C = av_popanalysis(t,data,frames,condnames,subconds,clip)
% function [C = ] av_popanalysis(t,data,framerange|framebin,condnames,subconds,clip)
%---
% Perform and display population analysis
%
% Input:
% - data        4D array: nt*nx*ncond*nrep
% - framerange  2-elements vector - index of first and last range over which
%               to average; use empty matrix to use all frames; the display
%               will be a single correlation matrix (variable = conditions)
% - framebin    scalar - frame binning factor; the display will be a grid
%               (variable = conditions) of correlation matrices (variable =
%               frame bins)
% - condnames   cell array - condition names
%               or structure
% - subconds    vector of indices - selection of a subset of conditions

if ~isvector(t)
    error 'syntax has changed, please add time as first argument'
end
[nt nx nc nrep] = size(data);
if nargin<4 || isempty(condnames)
    condnames = fn_num2str(1:nc,'cell'); 
elseif isstruct(condnames)
    scond = condnames;
    condnames = scond.condnames;
    if nc>scond.ncond
        data = data(:,:,scond.subcond,:);
        nc = scond.ncond;
    end
end
if length(condnames)~=nc, error 'length of condition names does not match number of conditions', end

% take appropriate part of data
if nargin>=5 && ~isempty(subconds)
    data = data(:,:,subconds,:);
    condnames = condnames(subconds);
    nc = length(condnames);
end
if isscalar(frames)
    data = fn_bin(data,[frames 1 1]);
elseif isempty(frames)
    frames = 1:nt;
    data = mean(data,1);
elseif length(frames)==2
    data = mean(data(frames(1):frames(2),:,:,:),1);
elseif isvector(frames)
    data = mean(data(frames,:,:,:),1);
else
    error 'frames argument must be empty, a scalar or a 2-elements vector'
end
nt = size(data,1);

% compute correlations
method = 'correlation';
x = fn_reshapepermute(data,{[1 3 4] 2}); % (nt*nc*nrep) (variables) * nx (observations)

C = 1-squareform(pdist(x,method));
if nrep>1
    % average all combinations of the same couple of conditions, but do not
    % include correlations between repetition i and repetition i
    C = permute(reshape(C,[nt*nc nrep nt*nc nrep]),[1 3 2 4]);
    okpair = logical(1-eye(nrep));
    C = nmean(C(:,:,okpair),3);
end

% clipping
if nargin<6 || isempty(clip)
    clip = [min(C(:)) max(C(:))]; 
    dodisplay = (nargout==0) || nargin>=5;
else
    dodisplay = true;
end

% output?
if ~dodisplay, return, end

% display
if nt==1
    imagesc(flipud(C),clip);
    set(gca,'xtick',1:nc,'xticklabel',condnames, ...
        'ydir','normal','ytick',1:nc,'yticklabel',condnames(end:-1:1))
    %rotateXLabels(gca,90)
    axis image
    title(['frames ' num2str(frames(1)) '-' num2str(frames(end))])
else
    C = reshape(C,[nt nc nt nc]);
    for i=1:nc
        for j=1:i
            Cij = squeeze(C(:,i,:,j));
            fn_subplot(gcf,nc+2,nc+2,(i-1+1)*(nc+2)+j+1,.02)
            %subplot(nc+2,nc+2,(i-1+1)*(nc+2)+j+1)
            imagesc(t,t,Cij,clip)
            axis image
            if j==1, ylabel(condnames{i},'horizontalalignment','right','rotation',0), end
            if i==nc, xlabel(condnames{j}), end
        end
    end
end
if nargout==0, clear C, end
