function stats = av_statsIndividualNeurons(tt,x,s,periods,driftflag)
% function stats = av_statsIndividualNeurons(tt,x,s[,periods[,'drift']])
%---
% Input:
% - tt  nt vector - time
% - x   (nt*ncond*nrep) or (nt*nneuron*ncond*nrep) array - data
% - s   structure describing the conditions (see av_conditions)
% - periods     periodes tested [default: {[1 2] [2 3] [3 4]}]
% - 'drift'
%
% Output:
% - stats   structure with statistics on the responses

% Input
siz = size(x);
if length(siz)==3, x = reshape(x,[siz(1) 1 siz(2:3)]); end
nneuron = size(x,2);
if nargin<4 || isempty(periods)
    periods = {[1 2] [2 3] [3 4]};
end
dodrift = (nargin>=5) && strcmp(driftflag,'drift');

% Drift: subtract a drift that is specific to neurons, but common to all
% conditions and repetitions - BOF CA MARCHE PAS...
if dodrift
    nt = size(x,1);
    drifttemplate = fn_normalize((1:nt)',1,'zscore')/nt;
    driftscore = fn_mean(sum(fn_mult(x,drifttemplate),1),[3 4]);
    x = fn_subtract(x,fn_mult(driftscore,drifttemplate));
end


% Multiple neurons
if nneuron>1
    fn_progress('stats',nneuron)
    stats = av_statsIndividualNeurons(tt,x(:,1,:,:),s,periods);
    stats(nneuron) = stats(1); % initialize structure
    for i=2:nneuron
        fn_progress(i)
        stats(i) = av_statsIndividualNeurons(tt,x(:,i,:,:),s,periods);
    end
    return
else
    x = squeeze(x); % remove the 'neuron' dimension
end

% Sizes
[nv na] = deal(s.nv,s.na);
ivs = s.ivs;
ivall = union(s.ivs,s.gratings);
ias = s.ias;
iavs = s.iavs;
ncond = size(x,2);
if ncond>s.ncond
    x = x(:,s.subcond,:);
    ncond = s.ncond;
end
condnames = s.condnames;
nperiod = length(periods);
for i=1:nperiod
    periods{i} = (tt>periods{i}(1) & tt<periods{i}(2));
end
dt = diff(tt(1:2));
nt = length(tt);
nrep = size(x,3);
periodsGLM = {[1 2] [2 3] [3 4] [1 4]};
npglm = length(periodsGLM);

% Init stats
stats = struct( ...
    'responses',zeros(1,ncond), ...
    'visual',0, ...
    'auditory',0, ...
    'gratings',0, ...
    'disk',0, ...
    'ftuningp',ones(1,npglm), ...
    'amplitudeRamp',0, ...
    'frequencyRamp',0, ...
    'visualp',ones(1,npglm), ...
    'auditoryp',ones(1,npglm), ...
    'gratingsp',ones(1,npglm), ...
    'diskp',ones(1,npglm), ...
    'attributes','', ...
    'OI',[],'OIp',1,'OIpref',[],'DI',[],'DIp',1,'DIpref',[],'DIfit',[], ...
    'OIbasic',[],'DIbasic',[],...
    'nonlinearp',ones(1,npglm),...
    'zeou',[]);
if any(isnan(x(:))), return, end

% Subtract baseline
irest = (tt<1);
z = fn_subtract(x,mean(x(irest,:,:)));
% z = fn_subtract(x,fn_mean(x(irest,:,:),[1 2 3])); % use this to perform some control test on the baseline 

% Responses to individual conditions
for i=1:ncond
    stats.responses(i) = testPeriods(squeeze(z(:,i,:)),periods);
%     figure(123)
%     plot(tt,fn_filt(squeeze(zi),1/dt,'kl'))
%     title(num2str(p))
%     pause
end

% Visual neuron, Auditory neuron
stats.visual = testPeriods(z(:,ivall,:),periods);
stats.auditory = testPeriods(z(:,ias,:),periods);
stats.gratings = testPeriods(z(:,s.gratings,:),periods);
if ~isempty(ivs), stats.disk = testPeriods(z(:,ivs(1:2),:),periods); end
if ~isempty(ias)
    stats.amplitudeRamp = testPeriods(z(:,ias(1:2),:),periods);
    if length(ias)>=4, stats.frequencyRamp = testPeriods(z(:,ias(3:4),:),periods); end
end

% Same but with a GLM
for i=1:npglm
    frames = (tt>periodsGLM{i}(1) & tt<periodsGLM{i}(2));
    zr = squeeze(mean(z(frames,:,:)));
    stats.visualp(i) = fn_GLMtest(eye(length(ivall)),zr(ivall,:));
    stats.auditoryp(i) = fn_GLMtest(eye(na),zr(ias,:));
    if ~isempty(s.gratings), stats.gratingsp(i) = fn_GLMtest(eye(8),zr(s.gratings,:)); end
    if ~isempty(ivs), stats.diskp(i) = fn_GLMtest(eye(2),zr(ivs(1:2),:)); end
    if length(ias)==4, stats.ftuningp(i) = anova1(zr(ias(3:4),:)',[],'off'); end
end

if ~isempty(s.gratings)
    % Orientation selectivity index
    % sine fit
    zgrat = column(mean(z(tt>1 & tt<3,s.gratings,:),1));
    % (orientation selectivity)
    A = repmat([ones(1,8); 1 0 -1 0 1 0 -1 0; 0 1 0 -1 0 1 0 -1]',nrep,1);
    [stats.OIp beta] = fn_GLMtest(A,zgrat,2:3);
    stats.OI = 2*norm(beta(2:3))/(abs(beta(1))+norm(beta(2:3)));
    sgn = sign(beta(1));
    theta = (0:7)'/8*(2*pi);
    tpref = mod(1/2*atan2(beta(3),beta(2)),pi); % preferred orientation (e.g. 0 if beta=[1 0], pi/4 if beta=[0 1], pi/2 if beta=[-1 0])
    if sgn==-1, tpref=mod(tpref+pi/2,pi); end % (e.g. pi/2 for [1 0])
    stats.OIpref = tpref;
    % (direction selectivity)
    B = repmat([ones(8,1) cos(2*(theta-tpref)) cos(theta-tpref)],nrep,1);
    [stats.DIp betb] = fn_GLMtest(B,zgrat,3);
    if sign(betb(1)*betb(2))<0
        error 'this was unexpected'
    end
    stats.DI = 2*abs(beta(3))/sum(abs(betb));
    if sign(betb(3))==sgn
        stats.DIpref = tpref;
    else
        stats.DIpref = mod(tpref+pi,2*pi);
    end
    stats.DIfit = mean(reshape(B*betb,8,nrep),2);
    % % (check)
    % figure(3)
    % rgrat = fn_mean(z(tt>1 & tt<3,s.gratings,:),[1 3]);
    % a = [ones(1,8); 1 0 -1 0 1 0 -1 0; 0 1 0 -1 0 1 0 -1]';
    % b = [ones(8,1) cos(2*(theta-tpref)) cos(theta-tpref)];
    % plot([rgrat(:) a*beta b(:,1:2)*betb(1:2) b*betb stats.DIfit])
    % line(stats.DIpref/(2*pi)*8+1,0,'marker','*')
    
    % basic OSI/DSI
    rgrat = fn_mean(z(tt>1 & tt<3,s.gratings,:),[1 3]);
    [rbest ibest] = max(rgrat);
    rpref = mean(rgrat(fn_mod(ibest+[0 4],8)));
    rorth = mean(rgrat(fn_mod(ibest+[2 6],8)));
    stats.OIbasic = (rpref-rorth)/rpref;
    rnull = rgrat(fn_mod(ibest+4,8));
    stats.DIbasic = (rbest-rnull)/rbest;
end

% non-linearity 
% (global)
a_v_av = [row(s.ivs) s.ias row(s.iavs(s.iavs>0))];
nav = sum(s.iavs(:)>0);
X = eye(length(a_v_av));
kav = 0;
for ka=1:s.na
    for kv=1:s.nv
        iv = s.ivs(kv);
        ia = s.ias(ka);
        iav = s.iavs(kv,ka);
        if iav
            kav = kav+1;
            X(s.nv+s.na+kav,(s.ivs==iv)) = 1;
            X(s.nv+s.na+kav,s.nv+find(s.ias==ia)) = 1;
        end
    end
end
X = repmat(X,nrep,1);
pp = zeros(1,npglm);
for i=1:npglm
    rall = mean(z(tt>periodsGLM{i}(1) & tt<periodsGLM{i}(2),a_v_av,:),1);
    pp(i) = fn_GLMtest(X,column(rall),s.nv+s.na+(1:nav));
end
stats.nonlinearp = pp;

% Summary description
stats.attributes = sprintf('%s (%s,%s,oi=%.2f,di=%.2f) - %s - NLp=%s', ...
    fn_switch(stats.visual,'VISUAL',''), ...
    fn_switch(stats.gratings,'gratings',''), ...
    fn_switch(stats.disk,'disk',''), ...
    fn_switch(stats.OIp<.05,stats.OI,[]), ...
    fn_switch(stats.DIp<.05,stats.DI,[]), ...
    fn_switch(stats.auditory,'AUDITIVE',''), ...
    num2str(stats.nonlinearp,'%.2f '));


% % Loop 
% for icond=1:nrow*ncol
%     [iaidx ividx] = ind2sub([ncol nrow],icond);
%     iv = ivs(ividx);
%     ia = ias(iaidx);
%     iav = iavs(ividx,iaidx); 
%     if iav==0
%         if dodisplay, set(ha(icond),'xtick',[],'ytick',[],'box','on'), end
%         continue
%     end
%     baseline = mean(row(data(irest,:)));
%     datairaw = squeeze(data(:,1,[iv ia iav],:));
%     datai = fn_filt(datairaw,1/dt,'lk');
% end   
a=1;

%---
function ok = testPeriods(z,periods)
% z is a time x repetition 2D array or time x condition x repetition 3D array
% ok is true if z is significantly non-zero in at least one of the periods
% (correction for multiple testing is applied)

if isempty(z), ok=[]; return, end

PTHR = .01;
% do3 = (ndims(z)==3);
nperiod = length(periods);
p = zeros(1,nperiod);
% if do3, q = zeros(1,nperiod); end
for k=1:nperiod
    zk = squeeze(mean(z(periods{k},:,:)))'; % repetition x condition 
    p(k) = signrank(zk(:));
    %     if do3, q(k) = kruskalwallis(zk,[],'off'); end
end
% if do3
%     fprintf('stats: %.3f-%.3f-%.3f / %.3f-%.3f-%.3f\n',p,q)
% end
[m km] = min(p);
ok = (m<(PTHR/nperiod));
ok = ok*sign(mean(row(z(periods{km},:))));
