
%% Time

dt = 0.1e-3; % s
tstim = 0.5;
T = 2;
tt = (0:dt:T)-tstim;
nt = length(tt);

%% Auditory input

delayA = 30e-3; % s
decayA = 100e-3;
tsmoothA = 5e-3;

audioin = zeros(nt,1);
poststimA = (tt-delayA>=0);
audioin(poststimA) = exp(-(tt(poststimA)-delayA)/decayA);
audioin = fn_smooth(audioin,tsmoothA/dt);
audioin = audioin / max(audioin);

%% Visual input

delayV = 60e-3; % s
plateauV = 200e-3;
decayV = 200e-3;
tsmoothV = 20e-3;

visualin = zeros(nt,1);
visualin(tt>=delayV) = 1;
postplateauV = (tt>=delayV+plateauV);
visualin(postplateauV) = exp(-(tt(postplateauV)-delayV)/decayV);
visualin = fn_smooth(visualin,tsmoothV/dt);
visualin = visualin / max(visualin);

%% Model parameters

% transfer function
subthr_slope = 0.05;
input2spike = @(x) subthr_slope*min(max(0,x),1) + max(0,x-1);

% which model
ModelNumber = eval('1');

switch ModelNumber
    case 1
        % simple inhibitory + generic; generic cell receives direct
        % auditory input
        ncell = 2;
        
        % baseline and connectivity
        baselinedark = [1; .3];
        baselinemodulation = [-.9; 0];
        audiostrength = [.9; .1];
        visualstrengthnonpref = [0; .2];
        visualstrengthpref = [0; .7];
        connectivity = [0 0; -.4 0];
    case 2
        % deshinibitory circuit; auditory input to second inhibitory cell
        % and to generic cell
        ncell = 3;
        
        % baseline and connectivity
        bb1 = [.5 1.5]; % baseline of deshinibitory cell
        bb2 = [1 .1];   % target baseline of inhibitory cell
        bo1 = input2spike(bb1);
        ci = diff(bb2)/diff(bo1);	% connectivity between the two inhibitory cells
        baselinedark = [bb1(1); 1-ci*bo1(1); .3];
        baselinemodulation = [diff(bb1); 0; 0];
        audiostrength = [0; .9; .1];
        visualstrengthnonpref = [0; 0; .2];
        visualstrengthpref = [0; 0; .7];
        connectivity = [0 0 0; ci 0 0; 0 -.4 0];
    case 3
        % deshinibitory circuit; auditory input to both inhibitory cells,
        % but not to generic cell
        ncell = 3;
        
        % baseline and connectivity
        baselinedark = [.1; 1; .3];
        baselinemodulation = [.9; 0; 0];
        audiostrength = [.9; .3; 0];
        visualstrengthnonpref = [0; 0; .2];
        visualstrengthpref = [0; 0; .72];
        connectivity = [0 0 0; -1.5 0 0; 0 -.4 0];
end

%% Run the dynamic system

contexts = {'dark' 'light' 'non-pref. V' 'pref. V' 'non-pref. AV' 'pref. AV'};
ncontext = 6;
Input = zeros(nt,ncell,ncontext,4);     % cell inputs: time * context * cells * (baseline/audio/visual/recurrent)
Output = zeros(nt,ncell,ncontext);      % cell output: time * context * cells


% Simulation
for kcontext = 1:ncontext
    
    % Contextual parameters
    context = contexts{kcontext};
    disp(['Simulation: ' context])
    switch context
        case 'dark'
            baseline = baselinedark;
        otherwise
            baseline = baselinedark + baselinemodulation;
    end
    doA = ~any(strfind(context,' V'));
    doV = any(strfind(context,'V'));
    if doV
        if strfind(context,'non-pref')
            visualstrength = visualstrengthnonpref;
        else
            visualstrength = visualstrengthpref;
        end
    end
    
    % External input
    Input(:,:,kcontext) = repmat(row(baseline),nt,1);
    if doA, Input(:,:,kcontext,2) = audioin*row(audiostrength); end
    if doV, Input(:,:,kcontext,3) = visualin*row(visualstrength); end

    % Run the model
    curoutput = input2spike(baseline);
    curoutput = input2spike(baseline+connectivity*curoutput); % reach equilibrium
    for i=1:nt
        % input
        Input(i,:,kcontext,4) = connectivity*curoutput;
        curinput = sum(Input(i,:,kcontext,:),4)';
        
        % output
        curoutput = input2spike(curinput);
        Output(i,:,kcontext) = curoutput;
    end
    
end

%%
% Display
doinputsum = eval('false');
downsampling = 50;
sub = 1:downsampling:nt;
nrow = 3;
ncol = 2+doinputsum;
figwidth = fn_switch(ncol,2,556,3,779);

fn_figure('Simulation dark vs. light',[figwidth 797],'color','w')

% Some generic info
if ncell==2
    % Input from A1 to V1
    if eval('false')
        subplot(nrow,ncol,1)
        plot(tt(sub),audioin(sub),'k')
        fn_axis('tight',[1 1.1],'noy0')
        xlabel 'time (s)'
        ylabel 'Input from A1 to V1'
    end
    
    % Input -> Output nonlinearity
    subplot(nrow,ncol,2)
    xx = 0:.01:2;
    plot(xx,input2spike(xx),'k')
    axis([xx([1 end]) input2spike(xx([1 end]))])
    set(gca,'ytick',[0 .05 1.05],'ygrid','on','DataAspectRatio',[1.5 1 1])
    ylabel 'input -> output nonlinearity'
end

% "Deshinibitory cell"
if ncell==3
    % Input
    subplot(nrow,ncol,1)
    InputDarkLightL1 = squeeze(Input(:,1,[1 2],[1 2]));
    hl = plot(tt(sub),InputDarkLightL1(sub,[1 2 3]));
    set(hl(1:2),'AlignVertexCenters','on')
    fn_set(hl,'color',[0 .5 0; 0 .75 0; 0 0 0],'linestyle',{'--' '--' '-'})
    fn_axis('tight',[1 1.1],'noy0')
    xlabel 'time (s)'
    ylabel 'L1, inhibitory cells'
    title 'Inputs'
    fn_colorlegend({'baseline input (dark)' 'baseline input (light)' 'auditory input'}, ...
        'northeast','fontsize',9)
    
    % Output
    subplot(nrow,ncol,ncol)
    hl = plot(tt(sub),squeeze(Output(sub,1,1:2)),'linewidth',2);
    fn_set(hl,'color',[0 .5 0; 0 .75 0])
    fn_axis('tight',[1 1.1],'noy0')
    title 'Output'
    fn_colorlegend({'dark' 'light'},'northeast','fontsize',9)

    % Sum of inputs to L1 inh.
    if doinputsum
        subplot(nrow,ncol,2)
        hl = plot(tt(sub),sum(InputDarkLightL1,3),'linewidth',2);
        fn_set(hl,'color',[0 .5 0; 0 .75 0])
        fn_axis('tight',[1 1.1],'noy0')
        hold on, plot(tt([1 end]),[1 1],'--k'), hold off
        title '(Sum of inputs)'
    end
end

% Input to L1 inh.
subplot(nrow,ncol,ncol+1)
InputDarkLightL1 = squeeze(Input(:,ncell-1,[1 2],:));
if ncell==2
    hl = plot(tt(sub),InputDarkLightL1(sub,[1 2 3]));
    set(hl(1:2),'AlignVertexCenters','on')
    fn_set(hl,'color',[0 .5 0; 0 .75 0; 0 0 0],'linestyle',{'--' '--' '-'})
    leg = {'baseline input (dark)' 'baseline input (light)' 'auditory input'};
else
    hl = plot(tt(sub),InputDarkLightL1(sub,[1 3 7 8]));
    set(hl(1:2),'AlignVertexCenters','on')
    fn_set(hl,'color',[0 0 1; 0 0 0; .8 0 0; 1 .3 .3],'linestyle',{'--' '-' '-' '-'})
    leg = {'baseline' 'auditory input' 'inhibition (dark)' 'inhibition (light)'};
end
fn_axis('tight',[1 1.1],'noy0')
xlabel 'time (s)'
ylabel 'L1, inhibitory cells'
title 'Inputs'
fn_colorlegend(leg,'northeast','fontsize',9)

% Output of L1 inh.
subplot(nrow,ncol,ncol+ncol)
hl = plot(tt(sub),squeeze(Output(sub,ncell-1,1:2)),'linewidth',2);
fn_set(hl,'color',[0 .5 0; 0 .75 0])
fn_axis('tight',[1 1.1],'noy0')
title 'Output'
fn_colorlegend({'dark' 'light'},'northeast','fontsize',9)

% Sum of inputs to L1 inh.
if doinputsum
    subplot(nrow,ncol,ncol+2)
    hl = plot(tt(sub),sum(InputDarkLightL1,3),'linewidth',2);
    fn_set(hl,'color',[0 .5 0; 0 .75 0])
    fn_axis('tight',[1 1.1],'noy0')
    hold on, plot(tt([1 end]),[1 1],'--k'), hold off
    title '(Sum of inputs)'
end

% Input to L2 gen.
subplot(nrow,ncol,2*ncol+1)
InputDarkLightL2 = squeeze(Input(:,ncell,[1 2],[1 2 4]));
hl = plot(tt(sub),InputDarkLightL2(sub,[1 3 5 6]));
set(hl(1),'AlignVertexCenters','on')
fn_set(hl,'color',[0 0 1; 0 0 0; .8 0 0; 1 .3 .3],'linestyle',{'--' '-' '-' '-'})
fn_axis('tight',[1 1.1],'noy0')
ylabel 'L2, generic cells'
fn_colorlegend({'baseline input' 'auditory input' 'L1 inhibition (dark)' 'L1 inhibition (light)'}, ...
    'southeast','fontsize',9)

% Output of L2 gen.
subplot(nrow,ncol,2*ncol+ncol)
hl = plot(tt(sub),squeeze(Output(sub,ncell,1:2)),'linewidth',2);
fn_set(hl,'color',[0 .5 0; 0 .75 0])
fn_axis('tight',[1 1.1],'noy0')
fn_colorlegend({'dark' 'light'},'northeast','fontsize',9)

% Sum of inputs to L2 gen.
if doinputsum
    subplot(nrow,ncol,2*ncol+2)
    hl = plot(tt(sub),sum(InputDarkLightL2,3),'linewidth',2);
    fn_set(hl,'color',[0 .5 0; 0 .75 0])
    hold on, plot(tt([1 end]),[1 1],'--k'), hold off
    fn_axis('tight',[1 1.1],'noy0')
    title '(Sum of inputs)'
end

fn_nicegraph(gcf)

%%

% Display
fn_figure('Bimodal simulation pref. vs. non-pref.',[figwidth 797],'color','w')

% Input to L2 generic (preferred visual stimulation)
subplot(nrow,ncol,ncol+1)
InputNonPrefL2 = squeeze(Input(:,ncell,6,[1 2 4 3]));
hl = plot(tt(sub),InputNonPrefL2(sub,1:4));
set(hl(1),'AlignVertexCenters','on')
fn_set(hl,'color',[0 0 1; 0 0 0; 1 .3 .3; 0 0 1],'linestyle',{'--' '-' '-' '-'})
fn_axis('tight',[1 1.1],'noy0'), axi = axis;
ylabel 'L2, gen., preferred visual stim.'
fn_colorlegend({'baseline input' 'auditory input' 'L1 inhibition' 'visual input'}, ...
    'northeast','fontsize',9)

% Output of L2 generic (preferred visual stimulation)
subplot(nrow,ncol,ncol+ncol)
VAB = Output(:,ncell,[4 2 6]);
hl = plot(tt(sub),[VAB(sub,1:3) VAB(sub,1)+VAB(sub,2)-VAB(20,1)]);
fn_set(hl,'color',[0 0 1; 0 .75 0; 1 0 0; 0 0 0],'linestyle',{'-' '-' '-' '--'},'linewidth',{2 2 2 1})
fn_axis('tight',[1 1.1],'noy0'), axo = axis;
fn_colorlegend(hl([1 2 4 3]),{'visual only' 'auditory only' 'linear prediction' 'bimodal'},'northeast','fontsize',9)

% Sum of inputs to L2 generic (preferred visual stimulation)
if doinputsum
    subplot(nrow,ncol,ncol+2)
    VAB = sum(Input(:,ncell,[4 2 6],:),4);
    hl = plot(tt(sub),VAB(sub,1:3));
    fn_set(hl,'color',[0 0 1; 0 .75 0; 1 0 0],'linestyle',{'-' '-' '-'},'linewidth',{2 2 2})
    hold on, plot(tt([1 end]),[1 1],'--k'), hold off
    fn_axis('tight',[1 1.1],'noy0'), axsi = axis;
    fn_colorlegend(hl,{'visual only' 'auditory only' 'bimodal'},'northeast','fontsize',9)
end

% Input to L2 generic (non-preferred visual stimulation)
subplot(nrow,ncol,1)
InputPrefL2 = squeeze(Input(:,ncell,5,[1 2 4 3]));
hl = plot(tt(sub),InputPrefL2(sub,1:4));
set(hl(1),'AlignVertexCenters','on')
fn_set(hl,'color',[0 0 1; 0 0 0; 1 .3 .3; 0 0 1],'linestyle',{'--' '-' '-' '-'})
axis(axi)
ylabel 'L2, gen., non-preferred visual stim.'
fn_colorlegend({'baseline input' 'auditory input' 'L1 inhibition' 'visual input'}, ...
    'northeast','fontsize',9)
title Inputs

% Output of L2 generic (non-preferred visual stimulation)
subplot(nrow,ncol,ncol)
VAB = Output(:,ncell,[3 2 5]);
hl = plot(tt(sub),[VAB(sub,1:3) VAB(sub,1)+VAB(sub,2)-VAB(20,1)]);
fn_set(hl,'color',[0 0 1; 0 .75 0; 1 0 0; 0 0 0],'linestyle',{'-' '-' '-' '--'},'linewidth',{2 2 2 1})
axis(axo)
fn_colorlegend(hl([1 2 4 3]),{'visual only' 'auditory only' 'linear prediction' 'bimodal'},'northeast','fontsize',9)
title Output

% Sum of inputs to L2 generic (preferred visual stimulation)
if doinputsum
    subplot(nrow,ncol,2)
    VAB = sum(Input(:,ncell,[3 2 5],:),4);
    hl = plot(tt(sub),VAB(sub,1:3));
    fn_set(hl,'color',[0 0 1; 0 .75 0; 1 0 0],'linestyle',{'-' '-' '-'},'linewidth',{2 2 2})
    hold on, plot(tt([1 end]),[1 1],'--k'), hold off
    fn_axis('tight',[1 1.1],'noy0'), axsi = axis;
    fn_colorlegend(hl,{'visual only' 'auditory only' 'bimodal'},'northeast','fontsize',9)
    title '(Sum of inputs)'
end

fn_nicegraph(gcf)

%% Predictions

nex = 10;

fn_figure('predictions','color','w')
clf 

subplot(121)
axis image, axis([-1 1 -1 1]), hold on
plot([-1 1],[0 0],'k--')
plot([0 0],[-1 1],'k--')
plot([-1 1],[-1 1],'k--')
rng(0), dark = rand(nex,1);
rng(1), light = randn(nex,1)/4;
scatter(dark,light,'r')
hold off
xlabel 'auditory response in dark'
ylabel 'auditory response in light'
fn_nicegraph

subplot(122)
axis image, axis([-1 1 -1 1]), hold on
plot([-1 1],[0 0],'k--')
plot([0 0],[-1 1],'k--')
plot([-1 1],[-1 1],'k--')
rng(3), scatter(-rand(nex,1),rand(nex,1),'b')
xlabel 'auditory response in dark'
ylabel 'auditory response in light'
hold off
fn_nicegraph
