function corrected_mass = correct_cell_elongation(mass, time, idx_kink, idx_division, cellnum, date)
% last update 04/03/2019
% code to correct cell elongation induced bias on antinode during ana/telo
% cells are assumed to have one of the following geometry:
% 1) sphere 2) overlapping spheres and 3) spherical doublet (i.e., two
% spheres touching each other)


global channel_height;
global channel_width;
global cell_density;
global fluid_density;
global transit_duration;



corrected_mass = mass;
transit_duration1 = transit_duration/60; %change of unit (from min to  h)

num_mitosis = length(idx_division); %this is number of mitosis in this dataset
if length(idx_kink) ~=num_mitosis
    disp(' ');
    input('error!. Check Data file. number of kink and division does not match!');
    disp(' ');
end

for k=1:num_mitosis
    fprintf('...Now correcting cell elongation induced bias for cell #%2.0f from "%s"', cellnum, date);
    display('...');
    
    
    idx_transition_start1 = idx_kink(k); %this is where cell is last round
    
    idx_transition_end1 = min(find((time - time(idx_kink(k)) - transit_duration1)>0));
    %this is where cell became full doublet (i.e., non-overlapping spheres).
    
    
    idx_doublet_start1 = idx_transition_end1; %start of full doublet
    idx_doublet_end1 = idx_division(k); %end of full doublet
    idx_doublet_start(k) = idx_doublet_start1; %storing this info for plot at the end of correction
    
    %assume that the volume of the cell stays constant during elongation period
    antinode_exp = mean(mass(idx_transition_start1:idx_transition_end1));
    
    
    %correct during active elongation (i.e., period from singlet to doublet)
    for i=idx_transition_start1+1:idx_transition_end1 -1
        
        
        singletvolume = antinode_exp/(cell_density - fluid_density);
        singletradius=(singletvolume/(4*pi/3))^(1/3);
        singletdiameter = singletradius*2;
        doubletradius = singletradius/(2^(1/3));
        doubletlength = doubletradius*4; %longest elongation of doublet
        
        duration = time(idx_transition_end1) - time(idx_transition_start1);
        %total duration in seconds from last singlet to first full doublet
        
        temp_time = time(i) - time(idx_transition_start1);
        temp_length = singletdiameter+(doubletlength-singletdiameter)*temp_time/duration;
        %assuming linear elongation of the cell
        
        [r_overlap1, x_intersect1] = getoverlap(singletradius, temp_length);%um
        
        d1 = 2*r_overlap1 - x_intersect1; %distance between two centers
        if temp_length < max(channel_height, channel_width)
            theta_max_temp = pi/2;
        elseif r_overlap1*2 > max(channel_height, channel_width)
            theta_max_temp = 0;
        else
            theta_max_temp = asin((max(channel_height, channel_width) - 2*r_overlap1)/d1);
        end
        
        corrected_mass(i) = correct_antinode_angle(singletradius, r_overlap1, x_intersect1, mass(i), theta_max_temp/2);
        %here, we are assuming the cells are rotated by the half maximal angle
        %allowed in the channel
        
    end
    
    
    %correct for the period when the cell is already in full doublet
    for i=idx_doublet_start1:idx_doublet_end1
        
        singletvolume = mass(i)/(cell_density - fluid_density);
        singletradius=(singletvolume/(4*pi/3))^(1/3);
        doubletradius = singletradius/(2^(1/3));
        
        d2 = 2*doubletradius; %diameter
        
        if doubletradius*4 < max(channel_height, channel_width)
            theta_max_temp = pi/2;
        elseif doubletradius*2 > max(channel_height, channel_width)
            theta_max_temp = 0;
        else
            theta_max_temp = asin((max(channel_height, channel_width) - 2*doubletradius)/d2);
        end
        
        corrected_mass(i) = correct_antinode_angle(singletradius, doubletradius, 0, mass(i), theta_max_temp/2);
        %here, we assume the cells are rotated by the half maximal angle
        %allowed in the channel
    end
    
end

%plot the corrected mass trace
hFig = figure(1); hold off;
scrsize = get(0, 'Screensize');
set(hFig, 'OuterPosition', [0.005*scrsize(3) 0.5*scrsize(4) 0.33*scrsize(3) 0.49*scrsize(4)])

hold off;
plot(time, mass, '.-'); hold on; %raw data
plot(time, corrected_mass, '.-', 'Color', 'r'); %corrected data
plot(time(idx_kink), mass(idx_kink), '*g'); %location of M/A transition
xlabel('Time after metaphase-to-anaphase transition (h)');
ylabel('Buoyant mass (pg)');
legend({'raw data', 'corrected data', 'metaphase-to-anaphase transition'}, 'Location', 'southwest');
set(gca, 'FontSize', 12);


input('Elongation correction finished. Move on? (ENTER to proceed, CTRL+C to stop the analysis)');

end



%========================================================================
function corrected_antinode = correct_antinode_angle(singletradius, r_overlap1, x_intersect1, antinode_exp, theta1)

%function that corrects antinode bias due to cell elongation

%written by JK
%last edit: 04/03/2019


global cantilever_length;
global r_overlap;
global x_intersect;
global theta;
global and_pos;



%define radius of overalpping spheres, overlap distance and rotation angle
%to be global so that later functions can access the values
r_overlap = r_overlap1;
x_intersect = x_intersect1;
theta = theta1;





%% Here Work on actual frequency deviation due to spherical or doublet mass distribution


l_2=4.6941; % eigenvalue for second flexural bending mode from Euler-Bernuuili beam equation
and_pos = 0.4708; %normalized position of the antinode (i.e., antinode position when cantilever length is 1)



r_t = r_overlap /cantilever_length; %radius of overlapping spheres normalized by cantilever length
y = x_intersect /cantilever_length; %intersected length between two overlapping spheres normalized by cantilever length

d=2*r_t - y; %tdistance between two centers of the spheres normalized by cantilever length


%========== overlapping spheres ========

if x_intersect >=0 %overlapping spheres
    
    f_antinode_left = @(x) pi*(r_t^2 - (x-and_pos+d/2*cos(theta)).^2).*((0.5 *( (cosh(l_2*x)-cos(l_2*x))-(sinh(l_2*x)-sin(l_2*x))*(cosh(l_2)+cos(l_2))/(sinh(l_2)+sin(l_2)))).^2);
    f_antinode_right = @(x) pi*(r_t^2 - (x-and_pos-d/2*cos(theta)).^2).*((0.5 *( (cosh(l_2*x)-cos(l_2*x))-(sinh(l_2*x)-sin(l_2*x))*(cosh(l_2)+cos(l_2))/(sinh(l_2)+sin(l_2)))).^2);
    f_antinode_overlap_angle = @getoverlap_antinode_angle;
    
    %separate into three regions. Left of sphere center, right of the sphere
    %center and overlapping regions
    freq_antinode_left = integral(f_antinode_left, and_pos-r_t-d/2*cos(theta), and_pos-r_t+d/2*cos(theta));
    freq_antinode_right = integral(f_antinode_right, and_pos+r_t-d/2*cos(theta), and_pos+r_t+d/2*cos(theta));
    freq_antinode_overlap = integral(f_antinode_overlap_angle, and_pos-r_t+d/2*cos(theta), and_pos+r_t-d/2*cos(theta));
    
    freqshift_antinode_overlap_angle = freq_antinode_left+freq_antinode_right+freq_antinode_overlap;
    
else
    disp(' ');
    disp('Currently not supported for separated spheres');
    disp(' ');
    return;
end


%========== spherical singlet ========

r= (singletradius/(cantilever_length)); %normalized radius of singlet

antinode_singlet = @(x)  pi*(r.^2 - (and_pos-x).^2).*((0.5 *( (cosh(l_2*x)-cos(l_2*x))-(sinh(l_2*x)-sin(l_2*x))*(cosh(l_2)+cos(l_2))/(sinh(l_2)+sin(l_2)))).^2);
freqshift_antinode_singlet= integral(antinode_singlet, and_pos-r, and_pos+r);

%========= correction part ============
%here correct antinode by taking the difference between expected frequency
%shift caused by singlet of the same volume and overlapping spheres with
%certain angle
corrected_antinode = antinode_exp*(1+ (freqshift_antinode_singlet - freqshift_antinode_overlap_angle)/freqshift_antinode_overlap_angle);


end




function [r_overlap, x_intersect] = getoverlap(singletradius, total_length)
%this function is to calculate radius and intersection length of
%overlapping spheres that has the same volume of a singlet, and given
%elongation length

%written by JK
%last edit: 04/03/2019

%% ============== INPUT PARAMETERS ================================
singletvolume = 4*pi/3*(singletradius)^3;

%% ================= DEFINE SWEEP ====================================
min_sweep = singletradius/(2)^(1/3); %radius of each sphere when cell is fully doublet
max_sweep = singletradius; %radius when cell is fully singlet (sphere)
N=500; %number of sweeps to accurately find matching radius and intersection length

r_sweep = linspace(min_sweep, max_sweep, N);
x_sweep = 4*r_sweep - total_length;

%% ================= FIND MINIMA ===================================
Vol_half=[]; Vol_doublet=[];
for i=1:length(r_sweep)
    half_doublet = @(x)  pi*(r_sweep(i).^2 - x.^2);
    Vol_half(i) = integral(half_doublet, -1*r_sweep(i), r_sweep(i)-x_sweep(i)/2);
    Vol_doublet(i)=Vol_half(i)*2;
end

Vol_diff = Vol_doublet - singletvolume;
[m, idx] = min(abs(Vol_diff));

r_overlap = r_sweep(idx);
x_intersect = x_sweep(idx);



end


function f_antinode_overlap_angle = getoverlap_antinode_angle(x)
%function for getting a mass distribution function for overalapping regions

%written by JK
%last edit: 04/03/2019

global x_intersect;
global r_overlap;
global theta;
global cantilever_length;
global and_pos;



l_2=4.6941; % eigenvalue for second flexural bending mode from Euler-Bernuuili beam equation


r_t = r_overlap /cantilever_length;
y = x_intersect /cantilever_length;

d=2*r_t - y; %distance between two center of the spheres relative to cantilever length


r1= sqrt(r_t^2 - (x-and_pos-d/2.*cos(theta)).^2); %right radius
r2 = sqrt(r_t^2 - (x-and_pos+d/2.*cos(theta)).^2); %left radius



for i=1:length(r1);
    G= [0 0 r1(i); 0 -abs(d*sin(theta)) r2(i)];
    M= area_intersect_circle_analytical(G);
    
    area(i) = M(1,1) + M(2,2) - M(1,2);
end
f_antinode_overlap_angle = area.*((0.5 *( (cosh(l_2*x)-cos(l_2*x))-(sinh(l_2*x)-sin(l_2*x))*(cosh(l_2)+cos(l_2))/(sinh(l_2)+sin(l_2)))).^2);


end




%code calculating area of intersecting circle.
%from https://www.mathworks.com/matlabcentral/fileexchange/15899-analytical-intersection-area-between-two-circles


function M = area_intersect_circle_analytical(varargin)
% Compute the overlap area between 2 circles defined in an array.
% Computation is vectorized, and intersection area are computed an
% analytical way.
%
% Input: Circles data presented in an array G of three columns.
%        G contains parameters of the n circles
%           . G(1:n,1) - x-coordinate of the center of circles,
%           . G(1:n,2) - y-coordinate of the center of circles,
%           . G(1:n,3) - radii of the circles
%        Each row of the array contains the information for one circle.
%
%        Input can also be provided in three different vectors. These
%        vectors can be row or column vectors. The 1st one corresponds to
%        x-coordinate of the center of circles, the 2nd one to the
%        y-cooridnate and the 3rd one to the radii of the circles.
%
%
% Output: Square matrix M(n,n) containing intersection areas between
% circles
%         M(i,j) contains the intersection area between circles i & j
%         By definition, M(i,i) corresponds to the area of circle i.
%
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% For 2 circles i & j, three cases are possible depending on the distance
% d(i,j) of the centers of the circles i and j.
%   Case 1: Circles i & j do not overlap, there is no overlap area M(i,j)=0;
%             Condition: d(i,j)>= ri+rj
%             M(i,j) = 0;
%   Case 2: Circles i & j fully overlap, the overlap area has to be computed.
%             Condition: d(i,j)<= abs(ri-rj)
%            M(i,j) = pi*min(ri,rj).^2
%   Case 3: Circles i & j partially overlap, the overlap area has to be computed
%            decomposing the overlap area.
%             Condition: (d(i,j)> abs(ri-rj)) & (d(i,j)<(ri+rj))
%            M(i,j) = f(xi,yi,ri,xj,yj,rj)
%                   = ri^2*arctan2(yk,xk)+ ...
%                     rj^2*arctan2(yk,d(i,j)-xk)-d(i,j)*yk
%             where xk = (ri^2-rj^2+d(i,j)^2)/(2*d(i,j))
%                   yk = sqrt(ri^2-xk^2)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Guillaume JACQUENOT
% guillaume.jacquenot@gmail.com
% 2007_06_08

% Some explanations, comments were added. Input arguments are detailed.
% Verifications of input arguments are performed.
% The number of calculations has been divided by two.





if nargin==0
    % Performs an example
    % Creation of 5 circles
    x = [0,1,5,3,-5];
    y = [0,4,3,7,0];
    r = [1,5,3,2,2];
elseif nargin==1
    temp = varargin{1};
    x = temp(:,1);
    y = temp(:,2);
    r = temp(:,3);
elseif nargin==3
    x = varargin{1};
    y = varargin{2};
    r = varargin{3};
else
    error('The number of arguments must 0, 1 or 3')
end

% Inputs are reshaped in
size_x = numel(x);
size_y = numel(y);
size_r = numel(r);

x = reshape(x,size_x,1);
y = reshape(y,size_y,1);
r = reshape(r,size_r,1);

% Checking if the three input vectors have the same length
if (size_x~=size_y)||(size_x~=size_r)
    error('Input of function must be the same length')
end

% Checking if there is any negative or null radius
if any(r<0)
    disp('Circles with null or negative radius won''t be taken into account in the computation.')
    temp = (r>=0);
    x = x(temp);
    y = y(temp);
    r = r(temp);
end

% Checking the size of the input argument
if size_x==1
    M = pi*r.^2;
    return
end

% Computation of distance between all circles, which will allow to
% determine which cases to use.
[X,Y] = meshgrid(x,y);
D     = sqrt((X-X').^2+(Y-Y').^2);
% Since the resulting matrix M is symmetric M(i,j)=M(j,i), computations are
% performed only on the upper part of the matrix
D = triu(D,1);

[R1,R2] = meshgrid(r);
sumR    = triu(R1+R2,1);
difR    = triu(abs(R1-R2),1);


% Creating the resulting vector
M = zeros(size_x*size_x,1);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Case 2: Circles i & j fully overlap
% One of the circles is inside the other one.
C1    = triu(D<=difR);
M(C1) = pi*min(R1(C1),R2(C1)).^2;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Case 3: Circles i & j partially overlap
% Partial intersection between circles i & j
C2 = (D>difR)&(D<sumR);
% Computation of the coordinates of one of the intersection points of the
% circles i & j
Xi(C2,1) = (R1(C2).^2-R2(C2).^2+D(C2).^2)./(2*D(C2));
Yi(C2,1) = sqrt(R1(C2).^2-Xi(C2).^2);
% Computation of the partial intersection area between circles i & j
M(C2,1) = R1(C2).^2.*atan2(Yi(C2,1),Xi(C2,1))+...
    R2(C2).^2.*atan2(Yi(C2,1),(D(C2)-Xi(C2,1)))-...
    D(C2).*Yi(C2,1);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute the area of each circle. Assign the results to the diagonal of M
M(1:size_x+1:size_x*size_x) = pi.*r.^2;

% Conversion of vector M to matrix M
M = reshape(M,size_x,size_x);

% Creating the lower part of the matrix
M = M + tril(M',-1);


% Display results if no argument is provided
if nargin==0
    f = figure;
    hAxs = axes('Parent',f);
    hold on, box on, axis equal
    xlabel('x')
    ylabel('y','Rotation',0)
    title('Compute the intersection area between 2 circles a vectorized way for several cicles')
    text(-2,-2,'Numbers on lines represent the intersection')
    text(-2,-3,'area between 2 circles')
    axis([-8 9 -4 10])
    colour = rand(size_x,3);
    for t = 1: size_x
        plot(x(t)+r(t).*cos(0:2*pi/100:2*pi),...
            y(t)+r(t).*sin(0:2*pi/100:2*pi),'color',colour(t,:),'parent',hAxs)
        plot(x(t),y(t),'+','color',colour(t,:),'parent',hAxs)
        for u = t+1:size_x
            if M(t,u)~=0
                plot([x(t),x(u)],[y(t),y(u)],'k-','parent',hAxs)
                text((x(t)+x(u))/2,(y(t)+y(u))/2,num2str(M(t,u)),...
                    'HorizontalAlignment','center',...
                    'BackgroundColor',[1 1 1],'parent',hAxs)
            end
        end
        
    end
end

end

