function main()
%last update 04/03/2019
%Written by Joon Ho Kang, input from Teemu P. Miettinen.
%central code to analyze the mitotic mass accumulation rate, 
%taking into account for cell elongation during anaphase/telophase

clear all; close all;

%define variables to be global so that they can be accessed from different codes
global cantilever_length;
global channel_height;
global channel_width;
global cell_density;
global fluid_density;
global transit_duration;

global breakup;
global deltat_G2;
global deltat_ana;
global t_kink_b;
global t_kink_a;

%% ==== Assumptions for cell elongation correction during ana/telo ============
cantilever_length = 350; %um, length of cantilever used in this study
channel_height = 15; %um, this is for caculating maximal rotation angle.
channel_width = 20; %um, this is for caculating maximal rotation angle
cell_density = 1.054; %g/cm^3, typical cell density for L1210 cells
fluid_density = 1.003; %g/cm^3, typical density of cell culture media (RPMI)
transit_duration = 12; %min, typical time for cell elongation in L1210 cells



%% =========== Parameters for MAR analysis ================
breakup = 10; %min, period of each short segment where linear line is fitted. 
%Each segments are horizontally shifted by the half the breakup size (i.e., breakup/2)

deltat_G2 = 0.5;% h, duration of late G2; time before mitotic entry
deltat_ana = 0.8; %h, duration of anaphase/cytokinesis
t_kink_b = 0.5; %early mitosis duration; (i.e., t_kink_b hour before M/A transition to end of G2)
t_kink_a = 0; %time after M/A where anaphase/cytokinesis analysis starts.

%% =========== Load data for processing ============
filename = 'sampledata.mat'; %here type in your file name
matObj = matfile(filename);
varlist = who(matObj);
cell_num = size(varlist,1); %number of cells in the data folder.
%This code is capable of running multiple cells.
%Just make sure to have them saved in the same .mat folder with different
%name (i.e., data1, data2, for mitotic cell#1 and cell #2, respectively);

%initialize
param_short = cell(cell_num,1);
name =cell(cell_num,1);
data = cell(cell_num,1);


for i=1:cell_num
    data{i} = struct2array(load(filename, varlist{i}));
    name{i} = varlist{i};
    
    disp(' '); fprintf('...Started analyzing cell #%2.0f from "%s"', i, name{i});
    disp('......'); disp(' ');
    
    temp_time = data{i}(:, 1);%time in hours
    temp_mass = data{i}(:, 2);%mass in pg for this particular cell being analyzed
    
    
    [m, idx_kink] = min(abs(temp_time)); %M/A transition found by time zero (data should be in time after metaphase-to-anphase transition)
    idx_division = find(diff(temp_mass)<-20); %division point found by sudden reduction in mass
    mass_corrected = correct_cell_elongation(temp_mass, temp_time, idx_kink, idx_division, i, name{i}); %correct cell elongation
    [temp_data_s] = getMAR(mass_corrected, temp_time, idx_kink, idx_division, i, name{i});
    param_short(i) = temp_data_s; %store data to save as .mat file
end


save('processed_sampledata.mat', 'param_short'); %save results.


%plot processed result

hFig = figure(3); hold off;
scrsize = get(0, 'Screensize');
set(hFig, 'OuterPosition', [0.66*scrsize(3) 0.07*scrsize(4) 0.33*scrsize(3) 0.92*scrsize(4)])

for i=1:size(param_short,1)
    subplot(2,1,1); hold on;
    plot(param_short{i,1}(:,1), param_short{i,1}(:,4), 'LineWidth', 2);
    subplot(2,1,2); hold on;
    plot(param_short{i,1}(:,1), param_short{i,1}(:,4)./param_short{i,1}(:,3), 'LineWidth', 2);
end

figure(3); subplot(2,1,1);
xlabel('Time after M/A transition (min)');
ylabel('MAR (pg/h)');
set(gca, 'FontSize', 12);
subplot(2,1,2);
xlabel('Time after M/A transition (min)');
ylabel('MAR/mass (1/h)');
set(gca, 'FontSize', 12);

disp(' ');
disp('...Quitting analysis... see Figure 3 for processed results');
end




